/**
 * @file DbusParser.cpp
 * @author CM/ESC3-Tino Lippold
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup CcDbusIf  DBUS Property Parser
 *
 * @brief  DBUS Property Parser.
 *
 * @{
 */

#include "DbusParser.h"
#include "IDbusProperty2EnumMapping.h"

#include "PmAppTrace.h"
#define FW_S_IMPORT_INTERFACE_ASSERT
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DbusParser.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace ccdbusif {

DbusParser::DbusParser()
{
   _tracesEnabled = false;
}

DbusParser::~DbusParser()
{
}

void DbusParser::parseProperties(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const
{
   if(nullptr == mapping)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true == _tracesEnabled)
   {
      if(BUS_TYPE_SYSTEM == busType)
      {
         ETG_TRACE_USR1((" parseProperties(): SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
      else
      {
         ETG_TRACE_USR1((" parseProperties(): SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
   }

   parsePropertiesInternal(outProperties, matchingInterfaces, inProperties, mapping);
}

void DbusParser::parseIntrospection(const ::std::string& data,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const
{
   if(nullptr == mapping)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true == _tracesEnabled)
   {
      if(BUS_TYPE_SYSTEM == busType)
      {
         ETG_TRACE_USR1((" parseIntrospection(): SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
      else
      {
         ETG_TRACE_USR1((" parseIntrospection(): SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
      ETG_TRACE_USR1((" parseIntrospection(): ___start___"));
      printf("%s", data.c_str()); fflush(stdout);
      ETG_TRACE_USR1((" parseIntrospection(): ___end___"));
   }
}

void DbusParser::parseInterfacesAdded(const ::std::string& addedObjectPath, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const
{
   if(nullptr == mapping)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true == _tracesEnabled)
   {
      if(BUS_TYPE_SYSTEM == busType)
      {
         ETG_TRACE_USR1((" parseInterfacesAdded(): SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
      else
      {
         ETG_TRACE_USR1((" parseInterfacesAdded(): SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
   }

#if 0
   <signal name="InterfacesAdded">
      <arg name="object" type="o"/>
      <arg name="interfaces" type="a{sa{sv}}"/>
   </signal>
#endif

   if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseInterfacesAdded(): addedObjectPath=%s", addedObjectPath.c_str())); }

   size_t i = 0;
   ::std::vector<DbusVariantProperty> outProperties;
   ::std::vector< int > matchingInterfaces;
   for(::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >::iterator it = inInterfaces.begin(); it != inInterfaces.end(); ++it)
   {
      if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseInterfacesAdded(): [%u] interface=%s", i, it->first.c_str())); }

      parsePropertiesInternal(outProperties, matchingInterfaces, it->second, mapping);

      outProperties.clear();
      ++i;
   }
}

void DbusParser::parseInterfacesRemoved(const ::std::string& removedObjectPath, const ::std::vector< ::std::string >& inInterfaces,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const
{
   if(nullptr == mapping)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true == _tracesEnabled)
   {
      if(BUS_TYPE_SYSTEM == busType)
      {
         ETG_TRACE_USR1((" parseInterfacesRemoved(): SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
      else
      {
         ETG_TRACE_USR1((" parseInterfacesRemoved(): SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str()));
      }
   }

#if 0
   <signal name="InterfacesRemoved">
      <arg name="object" type="o"/>
      <arg name="interfaces" type="as"/>
   </signal>
#endif

   if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseInterfacesRemoved(): removedObjectPath=%s", removedObjectPath.c_str())); }

   for(size_t i = 0; i < inInterfaces.size(); ++i)
   {
      if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseInterfacesRemoved(): [%u] interface=%s", i, inInterfaces[i].c_str())); }
   }
}

bool DbusParser::findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const ::std::string& propertyName, const int interface, const IDbusProperty2EnumMapping* mapping) const
{
   ::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it = inProperties.find(propertyName);

   if(inProperties.end() != it)
   {
      parsePropertyValue(outPropertyList, it->second, propertyName, interface, mapping);

      return true;
   }

   return false;
}

void DbusParser::parsePropertyValue(::std::vector<DbusVariantProperty>& outPropertyList, ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName, const int interface, const IDbusProperty2EnumMapping* mapping) const
{
   if(nullptr == mapping)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // 1. get interfaces
   ::std::vector<const ::std::map< std::string, int >*> interfaceList;
   ::std::vector< int > matchingInterfaces;
   matchingInterfaces.push_back(interface);
   mapping->getInterfaces(interfaceList, matchingInterfaces);

   // 2. parse properties
   ::std::string signature;
   ::std::string dummy;

   propertyValue.getSignature(signature);

   parseProperty(outPropertyList, interfaceList, propertyValue.getReadIterator(), signature, propertyName, 0, 0, false, dummy);

   if(0 == outPropertyList.size())
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void DbusParser::addDictionaryItem(::std::map< ::std::string, ::asf::dbus::DBusVariant >& dictionary, const ::std::string& key, const DbusVariant& value) const
{
   if(true == key.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ::std::map< ::std::string, ::asf::dbus::DBusVariant >::const_iterator it = dictionary.find(key); // else lint complains: Info 864: prio3: Expression involving variable 'dictionary' possibly depends on order of evaluation
   if(dictionary.end() != it)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ::asf::dbus::DBusVariant& item = dictionary[key];

   addVariantValue(item, value);
}

void DbusParser::addVariantValue(::asf::dbus::DBusVariant& item, const DbusVariant& value) const
{
   switch(value.getType())
   {
      case VARIANT_INT8:
      {
         // transfer as unsigned char
         unsigned char data = (unsigned char)value.getInt8();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_BYTE, &data));
         break;
      }
      case VARIANT_UINT8:
      {
         unsigned char data = value.getUInt8();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_BYTE, &data));
         break;
      }
      case VARIANT_INT16:
      {
         dbus_int16_t data = value.getInt16();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_INT16, &data));
         break;
      }
      case VARIANT_UINT16:
      {
         dbus_uint16_t data = value.getUInt16();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_UINT16, &data));
         break;
      }
      case VARIANT_INT32:
      {
         dbus_int32_t data = value.getInt32();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_INT32, &data));
         break;
      }
      case VARIANT_UINT32:
      {
         dbus_uint32_t data = value.getUInt32();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_UINT32, &data));
         break;
      }
      case VARIANT_INT64:
      {
         dbus_int64_t data = value.getInt64();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_INT64, &data));
         break;
      }
      case VARIANT_UINT64:
      {
         dbus_uint64_t data = value.getUInt64();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_UINT64, &data));
         break;
      }
      case VARIANT_ENUM:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
      case VARIANT_BOOL:
      {
         dbus_bool_t data = (dbus_bool_t)value.getBool();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_BOOLEAN, &data));
         break;
      }
      case VARIANT_STRING:
      {
         const char* data = value.getString().c_str();
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_STRING, &data));
         break;
      }
      case VARIANT_STRING_ARRAY:
      {
         // NOTE: to be checked with ASF team
         FW_NORMAL_ASSERT_ALWAYS();
#if 1
         // this implementation creates a variant entry with type array but array is always empty
         int type = DBUS_TYPE_ARRAY;
         const char* subType = "s";
         DBusMessageIter containerWriteIterator;
         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_open_container(item.getWriteIterator(), type, subType, &containerWriteIterator));

         const ::std::vector<std::string>& arrayValue = value.getStringArray();
         // printf("##### arrayValue.size()=%u\n", arrayValue.size());
         for(size_t i = 0; i < arrayValue.size(); i++)
         {
            const char* data = arrayValue[i].c_str();
            FW_NORMAL_ASSERT(FALSE != dbus_message_iter_append_basic(&containerWriteIterator, DBUS_TYPE_STRING, &data));
            // printf("##### data=%s\n", data);
         }

         FW_NORMAL_ASSERT(FALSE != dbus_message_iter_close_container(item.getWriteIterator(), &containerWriteIterator));
#endif

#if 0
         // copy from generated ASF code
         void serializeDBus (const ::org::bluez::Adapter1::TraceCategoryUpdate& in, DBusMessageIter* out) {
            DBusMessageIter o61ArrayIter;
            DBUS_ASSERT(dbus_message_iter_open_container(out, DBUS_TYPE_ARRAY, "s", &o61ArrayIter));
            const ::std::vector< ::std::string >* o62 = &in.getTraceCategory();
            ::std::vector< ::std::string >::const_iterator o63 = o62->end();
            for (::std::vector< ::std::string >::const_iterator o64 = o62->begin(); o64 != o63; ++o64) {
               const char* o65 = (*o64).c_str();
               DBUS_ASSERT(dbus_message_iter_append_basic(&o61ArrayIter, DBUS_TYPE_STRING, &o65));
            }
            DBUS_ASSERT(dbus_message_iter_close_container(out, &o61ArrayIter));
         }
#endif

#if 0
         // this implementation creates a variant entry with type array but array is always empty
         DBusMessageIter variantWriteIterator;
         DBusMessageIter containerWriteIterator;
         char typeSignature[2];
         char arraySignature[3];
         arraySignature[0] = (char)DBUS_TYPE_ARRAY;
         arraySignature[1] = typeSignature[0] = (char)DBUS_TYPE_STRING;
         arraySignature[2] = typeSignature[1] = '\0';
         dbus_message_iter_open_container(item.getWriteIterator(), DBUS_TYPE_VARIANT, arraySignature, &variantWriteIterator);
         dbus_message_iter_open_container(&variantWriteIterator, DBUS_TYPE_ARRAY, typeSignature, &containerWriteIterator);

         const ::std::vector<std::string>& arrayValue = value.getStringArray();
         // printf("##### arrayValue.size()=%u\n", arrayValue.size());
         for(size_t i = 0; i < arrayValue.size(); i++)
         {
            const char* data = arrayValue[i].c_str();
            dbus_message_iter_append_basic(&containerWriteIterator, DBUS_TYPE_STRING, &data);
            // printf("##### data=%s\n", data);
         }

         dbus_message_iter_close_container(&variantWriteIterator, &containerWriteIterator);
         dbus_message_iter_close_container(item.getWriteIterator(), &variantWriteIterator);
#endif

#if 0
         setTraces(true);
         ::std::vector<DbusVariantProperty> outProperties;
         ::std::vector<const ::std::map< std::string, int >*> interfaceList;
         ::std::string signature;
         ::std::string dummy;
         item.getSignature(signature);
         parseProperty(outProperties, interfaceList, item.getReadIterator(), signature, key, 0, 0, false, dummy);
#endif
         break;
      }
      case VARIANT_NONE:
      case VARIANT_UNKNOWN:
      default:
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
   }
}

void DbusParser::parsePropertiesInternal(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const IDbusProperty2EnumMapping* mapping) const
{
   if(nullptr == mapping)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // 1. get interfaces
   ::std::vector<const ::std::map< std::string, int >*> interfaceList;
   mapping->getInterfaces(interfaceList, matchingInterfaces);

   // 2. parse properties
   ::std::string signature;
   unsigned int i = 0;
   ::std::string dummy;
   outProperties.reserve(inProperties.size() + 20);

   for(::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it = inProperties.begin(); it != inProperties.end(); ++it)
   {
      it->second.getSignature(signature);
      parseProperty(outProperties, interfaceList, it->second.getReadIterator(), signature, it->first, i, 0, false, dummy);
      ++i;
   }
}

void DbusParser::parseProperty(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector<const ::std::map< std::string, int >*>& interfaceList, ::DBusMessageIter* iter, const ::std::string& signature, const ::std::string& property, const unsigned int counter, const unsigned int depth, const bool isDict, const ::std::string& dictKey) const
{
   if(nullptr == iter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   DbusVariantProperty variantEntry;
   unsigned int i = counter;
   bool add = true;

   // BYTE (uint8)
   if(0 == signature.compare("y"))
   {
      unsigned char value = 0;
      dbus_message_iter_get_basic(iter, &value);
      variantEntry.propData.setUInt8(value);
      if(true == _tracesEnabled)
      {
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value)); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%u", i, property.c_str(), value)); }
      }
   }
   // BOOLEAN (bool)
   else if(0 == signature.compare("b"))
   {
      dbus_bool_t value = FALSE;
      dbus_message_iter_get_basic(iter, &value);
      if(TRUE == value)
      {
         variantEntry.propData.setBool(true);
      }
      else
      {
         variantEntry.propData.setBool(false);
      }
      if(true == _tracesEnabled)
      {
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%d", i, property.c_str(), dictKey.c_str(), value)); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%d", i, property.c_str(), value)); }
      }
   }
   // INT16 (int16)
   else if(0 == signature.compare("n"))
   {
      dbus_int16_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      variantEntry.propData.setInt16(value);
      if(true == _tracesEnabled)
      {
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%d", i, property.c_str(), dictKey.c_str(), value)); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%d", i, property.c_str(), value)); }
      }
   }
   // UINT16 (uint16)
   else if(0 == signature.compare("q"))
   {
      dbus_uint16_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      variantEntry.propData.setUInt16(value);
      if(true == _tracesEnabled)
      {
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value)); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%u", i, property.c_str(), value)); }
      }
   }
   // INT32 (int32)
   else if(0 == signature.compare("i"))
   {
      dbus_int32_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      variantEntry.propData.setInt32(value);
      if(true == _tracesEnabled)
      {
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%d", i, property.c_str(), dictKey.c_str(), value)); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%d", i, property.c_str(), value)); }
      }
   }
   // UINT32 (uint32)
   else if(0 == signature.compare("u"))
   {
      dbus_uint32_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      variantEntry.propData.setUInt32(value);
      if(true == _tracesEnabled)
      {
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value)); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%u", i, property.c_str(), value)); }
      }
   }
   // INT64 (int64)
   else if(0 == signature.compare("x"))
   {
      dbus_int64_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      variantEntry.propData.setInt64(value);
      if(true == _tracesEnabled)
      {
         ::fw::FormattedOutputS64 printValue(value);
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), printValue.c_str())); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), printValue.c_str())); }
      }
   }
   // UINT64 (uint64)
   else if(0 == signature.compare("t"))
   {
      dbus_uint64_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      variantEntry.propData.setUInt64(value);
      if(true == _tracesEnabled)
      {
         ::fw::FormattedOutputU64 printValue(value);
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), printValue.c_str())); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), printValue.c_str())); }
      }
   }
   // DOUBLE (double)
   else if(0 == signature.compare("d"))
   {
      double value = 0.0;
      dbus_message_iter_get_basic(iter, &value);
      // not supported at the moment
      FW_NORMAL_ASSERT_ALWAYS();
      add = false;
      if(true == _tracesEnabled)
      {
         ::fw::FormattedOutputDouble printValue(value);
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), printValue.c_str())); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), printValue.c_str())); }
      }
   }
   // STRING (std::string)
   else if(0 == signature.compare("s"))
   {
      const char* value = nullptr;
      dbus_message_iter_get_basic(iter, &value);
      if(nullptr != value)
      {
         variantEntry.propData.setString(value);
         if(true == _tracesEnabled)
         {
            if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), value)); }
            else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), value)); }
         }
      }
      else
      {
         variantEntry.propData.setString("");
         if(true == _tracesEnabled)
         {
            if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), "<NULL>")); }
            else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), "<NULL>")); }
         }
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   // OBJECT_PATH (std::string)
   else if(0 == signature.compare("o"))
   {
      const char* value = nullptr;
      dbus_message_iter_get_basic(iter, &value);
      if(nullptr != value)
      {
         variantEntry.propData.setString(value);
         if(true == _tracesEnabled)
         {
            if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), value)); }
            else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), value)); }
         }
      }
      else
      {
         variantEntry.propData.setString("");
         if(true == _tracesEnabled)
         {
            if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), "<NULL>")); }
            else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), "<NULL>")); }
         }
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   // SIGNATURE (std::string)
   else if(0 == signature.compare("g"))
   {
      const char* value = nullptr;
      dbus_message_iter_get_basic(iter, &value);
      if(nullptr != value)
      {
         // not supported at the moment
         FW_NORMAL_ASSERT_ALWAYS();
         add = false;
         if(true == _tracesEnabled)
         {
            if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), value)); }
            else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), value)); }
         }
      }
      else
      {
         // not supported at the moment
         FW_NORMAL_ASSERT_ALWAYS();
         add = false;
         if(true == _tracesEnabled)
         {
            if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), "<NULL>")); }
            else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), "<NULL>")); }
         }
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   // UNIX_FD (uint32)
   else if(0 == signature.compare("h"))
   {
      dbus_uint32_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      // not supported at the moment
      FW_NORMAL_ASSERT_ALWAYS();
      add = false;
      if(true == _tracesEnabled)
      {
         if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value)); }
         else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%u", i, property.c_str(), value)); }
      }
   }
   // ARRAY of STRING (std::string)
   else if(0 == signature.compare("as"))
   {
      ::DBusMessageIter subIter;
      ::std::string subSignature;

      dbus_message_iter_recurse(iter, &subIter);
      char* sigPtr = dbus_message_iter_get_signature(&subIter);
      subSignature.assign(sigPtr);
      dbus_free(sigPtr);

      if(0 == subSignature.compare("s"))
      {
         ::std::vector<std::string> stringList;
         stringList.reserve(20);
         size_t j = 0;
         const char* value;
         bool next = true;
         while(true == next)
         {
            value = nullptr;
            dbus_message_iter_get_basic(&subIter, &value);
            if(nullptr != value)
            {
               stringList.push_back(value);
               if(true == _tracesEnabled)
               {
                  if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value[%u]=%s", i, property.c_str(), dictKey.c_str(), j, value)); }
                  else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value[%u]=%s", i, property.c_str(), j, value)); }
               }
            }
            else
            {
               stringList.push_back("");
               if(true == _tracesEnabled)
               {
                  if(true == isDict) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s : key=%20s value[%u]=%s", i, property.c_str(), dictKey.c_str(), j, "<NULL>")); }
                  else               { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value[%u]=%s", i, property.c_str(), j, "<NULL>")); }
               }
               FW_NORMAL_ASSERT_ALWAYS();
            }

            ++j;

            if(FALSE == dbus_message_iter_next(&subIter))
            {
               next = false;
            }
         }
         variantEntry.propData.setStringArray(stringList);
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
         add = false;
      }
   }
   // DICTIONARY (std::map)
   else if(0 == signature.compare("a{sv}"))
   {
      // do not parse any property now
      add = false;

      // if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT: signature=%s", signature.c_str())); }

      if(1 >= depth)
      {
         if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT: depth=%u (start)", depth)); }

         ::DBusMessageIter subIter;
         ::std::string subSignature;
         char* sigPtr;
         ::std::string dummy;

         // recurse into container value
         dbus_message_iter_recurse(iter, &subIter);

         size_t j = 0;
         bool next = true;
         while(true == next)
         {
            sigPtr = dbus_message_iter_get_signature(&subIter);
            subSignature.assign(sigPtr);
            dbus_free(sigPtr);

            if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT: [%u] subSignature=%s", j, subSignature.c_str())); }

            parseProperty(outProperties, interfaceList, &subIter, subSignature, property, i, (depth + 1), false, dummy);

            ++j;

            if(FALSE == dbus_message_iter_next(&subIter))
            {
               next = false;
            }
         }

         if(FALSE == dbus_message_iter_next(&subIter))
         {
            if(true == _tracesEnabled) { ETG_TRACE_USR4((" parseProperty(): DICT: no more fields (depth=%u)", depth)); }
         }
         else
         {
            sigPtr = dbus_message_iter_get_signature(&subIter);
            subSignature.assign(sigPtr);
            dbus_free(sigPtr);
            ETG_TRACE_ERR((" parseProperty(): DICT: more fields (depth=%u) subSignature=%s", depth, subSignature.c_str()));
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT: depth=%u (end)", depth)); }
      }
      else
      {
         ETG_TRACE_ERR((" parseProperty(): DICT: [%u] property=%20s value=%s", i, property.c_str(), "<DICT>"));
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   // DICTIONARY ENTRY
   else if(0 == signature.compare("{sv}"))
   {
      // do not parse any property now
      add = false;

      // if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT ENTRY: signature=%s", signature.c_str())); }

      if(1 >= depth)
      {
         if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT ENTRY: depth=%u (start)", depth)); }

         ::DBusMessageIter subIter;
         ::std::string subSignature;
         char* sigPtr;

         // recurse into container value
         dbus_message_iter_recurse(iter, &subIter);

         // get key
         sigPtr = dbus_message_iter_get_signature(&subIter);
         subSignature.assign(sigPtr);
         dbus_free(sigPtr);

         if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT ENTRY: key: subSignature=%s", subSignature.c_str())); }

         if(0 == subSignature.compare("s"))
         {
            const char* value = nullptr;
            dbus_message_iter_get_basic(&subIter, &value);
            if(nullptr != value)
            {
               if(TRUE == dbus_message_iter_next(&subIter))
               {
                  // get value
                  sigPtr = dbus_message_iter_get_signature(&subIter);
                  subSignature.assign(sigPtr);
                  dbus_free(sigPtr);
                  // if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT ENTRY: value: subSignature=%s", subSignature.c_str())); }

                  if(0 == subSignature.compare("v"))
                  {
                     ::DBusMessageIter valueIter;
                     ::std::string valueSignature;
                     char* sigValuePtr;
                     ::std::string key(value);

                     // recurse into container value
                     dbus_message_iter_recurse(&subIter, &valueIter);
                     sigValuePtr = dbus_message_iter_get_signature(&valueIter);
                     valueSignature.assign(sigValuePtr);
                     dbus_free(sigValuePtr);

                     if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT ENTRY: value: subSignature=%s", valueSignature.c_str())); }

                     parseProperty(outProperties, interfaceList, &valueIter, valueSignature, property, i, (depth + 1), true, key);
                  }
                  else
                  {
                     FW_NORMAL_ASSERT_ALWAYS();
                  }
               }
               else
               {
                  FW_NORMAL_ASSERT_ALWAYS();
               }
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(FALSE == dbus_message_iter_next(&subIter))
         {
            if(true == _tracesEnabled) { ETG_TRACE_USR4((" parseProperty(): DICT ENTRY: no more fields (depth=%u)", depth)); }
         }
         else
         {
            sigPtr = dbus_message_iter_get_signature(&subIter);
            subSignature.assign(sigPtr);
            dbus_free(sigPtr);
            ETG_TRACE_ERR((" parseProperty(): DICT ENTRY: more fields (depth=%u) subSignature=%s", depth, subSignature.c_str()));
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): DICT ENTRY: depth=%u (end)", depth)); }
      }
      else
      {
         ETG_TRACE_ERR((" parseProperty(): DICT ENTRY: [%u] property=%20s value=%s", i, property.c_str(), "<DICT>"));
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
   // invalid data (e.g. empty ARRAY)
   else if(0 == signature.compare(""))
   {
      // according DBUS specification an empty string indicates an invalid type
      // this was observed during testing for e.g. an empty string array
      // this means that the related property is empty
      // therefore add to property list to indicate an empty value
      // HINT: if property is not found related entry is not added to property list
      variantEntry.propData.setNone();
      if(true == _tracesEnabled) { ETG_TRACE_USR1((" parseProperty(): [%u] property=%20s value=%s", i, property.c_str(), "\"\"")); }
   }
   // others
   else
   {
      ETG_TRACE_ERR((" parseProperty(): [%u] property=%20s signature=\"%s\" not supported", i, property.c_str(), signature.c_str()));
      FW_NORMAL_ASSERT_ALWAYS();
      add = false;
   }

   if(true == add)
   {
      // get matching property
      bool found = false;

      for(size_t k = 0; (false == found) && (k < interfaceList.size()); k++)
      {
         if(nullptr != interfaceList[k])
         {
            const ::std::map< std::string, int >& propList = *interfaceList[k];

            ::std::map< std::string, int >::const_iterator it;

            if(true == isDict)
            {
               it = propList.find(dictKey);
            }
            else
            {
               it = propList.find(property);
            }

            if(propList.end() != it)
            {
               variantEntry.propEnum = it->second;
               found = true;
            }
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }
      }

      if(true == found)
      {
         outProperties.push_back(variantEntry);
      }
   }
}

#if 0
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:654, parseProperty(): [11] property=UUIDs value[10]=00001200-0000-1000-8000-00805f9b34fb
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:693, parseProperty(): DICT: depth=0 (start)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:711, parseProperty(): DICT: [0] subSignature={sv}
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:754, parseProperty(): DICT ENTRY: depth=1 (start)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:768, parseProperty(): DICT ENTRY: key: subSignature=s
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:797, parseProperty(): DICT ENTRY: value: subSignature=q
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:466, parseProperty(): [12] property=Version : key=Code value=1537
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:834, parseProperty(): DICT ENTRY: depth=1 (end)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:711, parseProperty(): DICT: [1] subSignature={sv}
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:754, parseProperty(): DICT ENTRY: depth=1 (start)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:768, parseProperty(): DICT ENTRY: key: subSignature=s
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:797, parseProperty(): DICT ENTRY: value: subSignature=q
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:466, parseProperty(): [12] property=Version : key=GeniviVersion value=49812
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:834, parseProperty(): DICT ENTRY: depth=1 (end)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:711, parseProperty(): DICT: [2] subSignature={sv}
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:754, parseProperty(): DICT ENTRY: depth=1 (start)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:768, parseProperty(): DICT ENTRY: key: subSignature=s
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:797, parseProperty(): DICT ENTRY: value: subSignature=q
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:466, parseProperty(): [12] property=Version : key=EvoVersion value=49776
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:834, parseProperty(): DICT ENTRY: depth=1 (end)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:736, parseProperty(): DICT: depth=0 (end)
[116780416]CCDBUSIF___,TRACE_LEVEL_USER1,Function:parseProperty,LINE:443, parseProperty(): [13] property=WBSMode value=1
#endif

} //ccdbusif

/** @} */
