/**
 * @file FileUtils.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the declaration of file utility functions used by PhoneCallManager
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCommon
 */

#ifndef FileUtils_h
#define FileUtils_h

#include <string>
#include <sys/stat.h>

namespace com
{
namespace bosch
{
namespace pmcommon
{

/**
 * This method is used to check the given path is a directory or not
 *
 * @param[in] dirPath - directory path
 * @param[out]
 * @param[in,out]
 *
 * @return bool : true - given path is a directory, false - given path is not a directory
 */
bool isDirectory(const std::string& dirPath);

/**
 * This method is used to check the existence of given directory
 *
 * @param[in] dirPath - directory path
 * @param[out]
 * @param[in,out]
 *
 * @return bool : true - directory exist, false - directory not exist
 */
bool checkDirectoryExist(const std::string& dirPath);

/**
 * This method is used to create the directory
 *
 * @param[in] dirPath - directory path
 * @param[in] dirPermissions - directory permissions
 *            Default : User & Group :- Read, Write and Search. for Others :- Read and Search
 * @param[out]
 * @param[in,out]
 *
 * @return int
 */
int createDirectory(const std::string& dirPath,
      const unsigned int dirPermissions = (S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH));

/**
 * This method is used to check the existence of given file
 *
 * @param[in] filePath - file path
 * @param[out]
 * @param[in,out]
 *
 * @return bool : true - file exist, false - file not exist
 */
bool checkFileExist(const std::string& filePath);

/**
 * This method is used to check the file permissions
 *
 * @param[in] filePath - file path
 * @param[out]
 * @param[in,out]
 *
 * @return int
 */
int checkFilePermission(const std::string& filePath);

/**
 * This method is used to check and adapt the file permissions
 *
 * @param[in] filePath - file path
 * @param[in] userName - user name
 * @param[in] groupName - group name
 * @param[in] filePermissions - file permissions
 * @param[out]
 * @param[in,out]
 *
 * @return int
 */
int checkAndAdaptFilePermission(const std::string& filePath, const std::string& userName,
      const std::string& groupName, const unsigned int filePermissions);

/**
 * This method is used to get the file name from given file path
 *
 * @param[in] filePath - file path
 * @param[out] fileName - file name
 * @param[in,out]
 *
 * @return bool
 */
bool getFileNameFromFilePath(const std::string& filePath, std::string& fileName);

} // namespace pmcommon
} // namespace bosch
} // namespace com

#endif // FileUtils_h
