/**
 * @file ITimer.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the declaration of the ITimer class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of Timer class.
 *
 * @ingroup PM Application- PM Common
 */

#ifndef ITimer_h
#define ITimer_h

#include "TimerData.h"

namespace com
{
namespace bosch
{
namespace pmcommon
{

/**
 * ITimer class definition
 */
template <class TUser, typename TTimerInfo, typename TDuration, typename TRepeatCount>
class ITimer
{
public:
   /**
    * Destructor of ITimer class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~ITimer(){}

   /**
    * Constructor of ITimer class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   ITimer() : _user(nullptr), _timerData()
   {
   }

   /**
    * This API is used to configure the Timer before starting it.
    *
    * @param[in]  TUser* - Pointer to the user class object
    *             TimerData - User Data for the timer such as interval duration
    *                callback has to be triggered
    * @param[out]
    * @param[in,out]
    *
    * @return true- configuring timer is successful; false- configuring timer is failure
    */
   virtual bool configure(TUser* const user,
         const TimerData<TTimerInfo, TDuration, TRepeatCount>& timerData) = 0;

   /**
    * This API is used to start the Timer after successful timer configuration.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return true- if timer is started successfully; false- timer not started successfully
    */
   virtual bool start() = 0;

   /**
    * This API is used to start the Timer along with the configuration values.
    *
    * @param[in]  TUser* - Pointer to the user class object
    *             TimerData - User Data for the timer such as interval duration
    *                callback has to be triggered
    * @param[out]
    * @param[in,out]
    *
    * @return true- if timer is started successfully; false- timer not started successfully
    */
   virtual bool start(TUser* const user, const TimerData<TTimerInfo, TDuration, TRepeatCount>& timerData) = 0;

   /**
    * This API is used to stop the Timer.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual void stop() = 0;

   /**
    * This API is used to know the active status of the timer.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return true- if the timer is active and running, false if the timer is not active
    */
   virtual bool isActive() = 0;

   /**
    * This API is used to know the stop status of the timer.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return true- if the timer is stopped, false if the timer is not stopped
    */
   virtual bool isStopped() = 0;

protected:
   /**
    * This function is used to get the "_user"
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return TUser* - _user
    */
   virtual TUser* getUser() = 0;

   /**
    * This function is used to get the "_timerData"
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return TimerData* - _timerData
    */
   virtual TimerData<TTimerInfo, TDuration, TRepeatCount> getTimerData() = 0;

   /**
    * This function is used to set the "_user" and "_timerData"
    *
    * @param[in]  TUser* user- User value to set _user
    *             TimerData& timerData- value to set _timerData
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual void configureTimer(TUser* const user,
         const TimerData<TTimerInfo, TDuration, TRepeatCount>& timerData) = 0;

   // User instantiating the timer
   TUser* _user;

   // Timer user data
   TimerData<TTimerInfo, TDuration, TRepeatCount> _timerData;
};

} // namespace pmcommon
} // namespace bosch
} // namespace com

#endif //ITimer_h
