/**
 * @file AsfTimer.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the declaration of the AsfTimer class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP request interfaces of Timer class.
 *
 * Instead of deriving from ITimerCallback the AsfTimer class has to be derived from the
 * ASF core class - TimerCallbackIF to receive the callback function - onExpired()
 *
 * Hint: There are no IPC synchronization mechanisms used within this class,
 * since the call back will be from ASF context only
 *
 * @ingroup PM Application- PM Common
 */

#ifndef AsfTimer_h
#define AsfTimer_h

#include "ITimer.h"
#include "asf/core/Timer.h"
#include "AsfTimerCallbackData.h"

// Setting the template types to ASF timer specific types
typedef asf::ms_t ms_t;
typedef unsigned int RepeatCount; // type of size_t

using namespace asf::core;

namespace com
{
namespace bosch
{
namespace pmcommon
{

/**
 * AsfTimer class definition
 */
template <class TUser, typename TTimerInfo, typename TDuration = ms_t, typename TRepeatCount = RepeatCount>
class AsfTimer final : public ITimer<TUser, TTimerInfo, TDuration, TRepeatCount>, public asf::core::TimerCallbackIF
{
public:
   /**
    * Constructor of AsfTimer class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   AsfTimer() : ITimer<TUser, TTimerInfo, TDuration, TRepeatCount>(),
   _asfTimerInst(nullptr)
   {

   }

   /**
    * Destructor of AsfTimer class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   ~AsfTimer()
   {
      stop();

      /*
       * TODO:
       * Comments in stop() interface of asf::core::Timer states that:
       * "It is not allowed to destruct a timer before isActive() returns false"
       * So need to check here.
       */

      if (_asfTimerInst)
      {
         delete _asfTimerInst;
         _asfTimerInst = nullptr;
      }

      ITimer<TUser, TTimerInfo, TDuration, TRepeatCount>::_user = nullptr;
   }

   /**
    * Copy constructor of AsfTimer class
    *
    * @param[in] ref - Reference to AsfTimer
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   AsfTimer(const AsfTimer& ref);

   /**
    * operator= overloaded function
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return Returns the object of the AsfTimer
    */
   AsfTimer& operator=(const AsfTimer& ref);

   /**
    * This API is used to configure the Timer before starting it.
    *
    * @param[in]  TUser* - Pointer to the user class object
    *             TimerData<TTimerInfo, TDuration, TRepeatCount> - User Data for the timer such as interval duration
    * @param[out]
    * @param[in,out]
    *
    * @return true- configuring timer is successful; false- configuring timer is failure
    */
   bool configure(TUser* const user, const TimerData<TTimerInfo, TDuration, TRepeatCount>& timerData) override
   {
      if (isUserAndTimerDataValid(user, timerData))
      {
         configureTimer(user, timerData);

         _asfTimerInst = new asf::core::Timer(*this, timerData._duration, timerData._repeatCount);
         return true;
      }

      return false;
   }

   /**
    * This API is used to start the Timer after successful timer configuration.
    *
    * This method should be invoked only after setting up the configuration successfully
    * by calling the configure() method.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return true- timer start() is posted ; false- timer start() is not posted
    */
   bool start() override
   {
      if (_asfTimerInst)
      {
         _asfTimerInst->start();
         return true;
      }

      return false;
   }

   /**
    * This API is used to start the Timer with the configuration values.
    *
    * @param[in]  TUser* - Pointer to the user class object
    *             TimerData<TTimerInfo, TDuration, TRepeatCount> - User Data for the timer such as interval duration
    * @param[out]
    * @param[in,out]
    *
    * @return true- timer start() is posted ; false- timer start() is not posted
    */
   bool start(TUser* const user, const TimerData<TTimerInfo, TDuration, TRepeatCount>& timerData) override
   {
      if (isUserAndTimerDataValid(user, timerData))
      {
         configureTimer(user, timerData);

         _asfTimerInst = new asf::core::Timer();
         _asfTimerInst->start(*this, getTimerData()._duration, getTimerData()._repeatCount);

         return true;
      }

      return false;
   }

   /**
    * This API is used to stop the Timer.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void stop() override
   {
      if (isActive())
      {
         _asfTimerInst->stop();
      }
   }

   /**
    * This API is used to know the active status of the timer.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return true- if the timer is active and running, false if the timer is not active
    */
   bool isActive() override
   {
      if (_asfTimerInst && _asfTimerInst->isActive())
      {
         return true;
      }

      return false;
   }

   /**
    * This API is used to know the active status of the timer.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return true- if the timer is active and running, false if the timer is not active
    */
   bool isStopped() override
   {
      if (_asfTimerInst && _asfTimerInst->isStopped())
      {
         return true;
      }

      return false;
   }

   /**
    * This function is inherited from asf::core::TimerCallbackIF and will be invoked on
    * expiry of the timer.
    *
    * @param[in]  asf::core::Timer& timer - Timer details with which the timer was started or instantiated
    *             ::boost::shared_ptr< asf::core::TimerPayload > data - Callback data from the asf::core::Timer
    *
    * @param[out]
    * @param[in,out]
    *
    */
   void onExpired(asf::core::Timer& timer, ::boost::shared_ptr< asf::core::TimerPayload > data) override
   {
      // The class instantiating ASFTimer must have a function- "void timerElapsed(TimerData, AsfTimerCallbackData)"
      // which will be called from here on Timer expiry.

      (void)(timer);
      AsfTimerCallbackData timerCallbackData;

      if (data.get())
      {
         timerCallbackData._repeatCount = data.get()->getRepeatCount();

         if (TimerPayload_Reason__Repeat == data.get()->getReason())
         {
            timerCallbackData._repeatReason = "Repeat";
         }
         else if (TimerPayload_Reason__Completed == data.get()->getReason())
         {
            timerCallbackData._repeatReason = "Completed";
         }
      }

      getUser()->timerElapsed(getTimerData(), timerCallbackData);
   }

private:
   /**
    * This function is used to get the "_user"
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return TUser* - _user
    */
   TUser* getUser() override
   {
      return (ITimer<TUser, TTimerInfo, TDuration, TRepeatCount>::_user);
   }

   /**
    * This function is used to get the "_timerData"
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return TimerData<TTimerInfo, TDuration, TRepeatCount> - _timerData
    */
   TimerData<TTimerInfo, TDuration, TRepeatCount> getTimerData() override
   {
      return (ITimer<TUser, TTimerInfo, TDuration, TRepeatCount>::_timerData);
   }

   /**
    * This function is used to set the "_user" and "_timerData"
    *
    * @param[in]  TUser* user- User value to set _user
    *             TimerData& timerData- value to set _timerData
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void configureTimer(TUser* const user,
         const TimerData<TTimerInfo, TDuration, TRepeatCount>& timerData) override
   {
      ITimer<TUser, TTimerInfo, TDuration, TRepeatCount>::_user = user;
      ITimer<TUser, TTimerInfo, TDuration, TRepeatCount>::_timerData = timerData;
   }

   /**
    * This function is used to check whether the given user and TimerData are valid.
    * Only if they are valid, timer shall be started.
    *
    * @param[in]  TUser* - Pointer to the user class object
    *             TimerData<TTimerInfo, TDuration, TRepeatCount>* timerData- TimerData pointer
    * @param[out]
    * @param[in,out]
    *
    * @return true- user and TimerData are valid; false- user or TimerData is invalid.
    */
   bool isUserAndTimerDataValid(const TUser* const user,
         const TimerData<TTimerInfo, TDuration, TRepeatCount>& timerData)
   {
      if (user)
      {
         if (DEFAULT_TIMER_DURATION != timerData._duration)
         {
            return true;
         }
      }

      return false;
   }

   // Instance of the asf::core::Timer
   asf::core::Timer* _asfTimerInst;
};

} // namespace pmcommon
} // namespace bosch
} // namespace com

#endif //AsfTimer_h
