/**
 * @file PmSingleton.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file defines the base class - 'PmSingleton' for all Singleton classes
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This class acts as the base class for the classes that opt to exhibit singleton pattern
 *
 * @ingroup PhoneCallManager
 */

#ifndef PmSingleton_h
#define PmSingleton_h

/*
 * This is Scott Meyers type of singleton pattern
 */
template <typename TClass>
class PmSingleton
{
public:

   /**
    * Returns the instance of the instantiated class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return  TClass& - object instantiated
    *
    */
   static TClass& getInstance()
   {
      if(!_instance)
      {
         if(_destroyed)  // check for the dead reference
         {
            /*
             * @attention: This code shall be executed only when there occurs a fault in coding!
             * Developers to take care of the destructor call sequence when destroying singleton objects
             */
            // TODO: Add a normal assert
            // ETG_ERROR(("Trying to refer the Static instance which is already destroyed"));
         }
         else
         {
            createInstance();
         }
      }

      return *_instance;
   }

   /**
    * Destructor of the PmSingleton class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~PmSingleton()
   {
      _instance = nullptr;
      _destroyed = true;
   }

protected:

   /**
    * Constructor of the PmSingleton class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   PmSingleton() = default;

   /**
    * Overloaded Copy constructor of the PmSingleton class
    *
    * @param[in] const PmSingleton& - object that has to be copied
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   PmSingleton(const PmSingleton& other) = delete; /**< Disallow copy */

   /**
    * Overloaded assignment operator of the PmSingleton class
    *
    * @param[in] const PmSingleton& - object that has to be copied
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   PmSingleton& operator=(const PmSingleton& rhs) = delete; /**< Disallow assignment */

   /**
    * Create the local static instance of the requested class type
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   static void createInstance()
   {
      static TClass instance;
      _instance = &instance;
   }

   static TClass* _instance;  /**< Pointer to the singleton instance */
   static bool _destroyed; /**< boolean variable to track the dead reference issue*/
};

// Initialize the static members of PmSingleton class

template <class TClass> TClass* PmSingleton <TClass>::_instance = nullptr;
template <class TClass> bool PmSingleton<TClass>::_destroyed = false;

#endif // PmSingleton_h
