/**
 * @file LoopbackController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the declaration of the LoopbackController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the interfaces for LoopbackController class.
 *          Similar to the Loopback messages offered by the CCA framework where the application can post a message
 *          within the application to itself. This mechanism is widely used to post messages from a foreign thread
 *          created within the application back to the entry thread. ASF also provides this option similar to CCA
 *          using a different terminology named LocalMessage.
 *          For more details refer section - 4.7. Local Messages in the ASF user guide.
 *
 *          Functionalities offered by this class are:
 *          i.    Interface to send the loopback Message
 *          ii.   Upon receiving the loopback message in the entry thread context, it is posted to the destination
 *                component as desired by the sending component.
 *                Destination component could be - IpcWrapper [OR] PM-Core [OR] PM-AudioManager
 *          iii.  The methods are thread safe and could be called from different thread context
 *
 * @ingroup PM Application- PM Common
 */

#ifndef LoopbackController_h
#define LoopbackController_h

#include "PmSingleton.h"
#include "LoopbackData.h"
#include "ILoopbackReceiver.h"
#include "PmAudioManagerRequestIf.h"
#include "Lock.h"

namespace com
{
namespace bosch
{
namespace pmcommon
{

/**
 * LoopbackController class definition
 */
class LoopbackController : public PmSingleton<LoopbackController>
{
public:
   /**
    * This API is used to post the loopback message to the component
    *
    * @param[in] loopbackData - pointer to the loopback message
    * @param[out]
    * @param[in,out]
    *
    * @return unsigned long long int - Acknowledgment token
    */
    ResponseToken postLoopbackMessage(LoopbackData* loopbackData)
    {
       ResponseToken act = 0;

       Locker lock(&_loopbackControllerLock);

       if(nullptr != _loopbackReceiver)
       {
          act = _loopbackReceiver->postLoopbackMessage(loopbackData);
       }

       return act;
    }

    /**
     * This method is invoked by ASF FW when the loopback message is received in the component
     *
     * @param[in] unsigned long long int - Acknowledgment token (Asynchronous completion Token)
     *            LoopbackData* - pointer to the loopback message
     * @param[out]
     * @param[in,out]
     *
     * @return void
     */
    void onLoopbackMessage(ResponseToken act, LoopbackData* loopbackData)
    {
       if(nullptr != loopbackData)
       {
          switch(loopbackData->getPmSubComponent())
          {
             case IpcWrapper:
                break;
             case PmCore:
                break;
             case PmAudioManager:
                pmaudiomanager::PmAudioManagerRequestIf::getInstance().onLoopbackMessage(act, loopbackData);
                break;
             default:
                break;
          }
       }
       else
       {
          // TODO: Add ETG_ERR for nullptr check
       }
    }

    void setLoopbackReceiver(ILoopbackReceiver* loopbackReceiver)
    {
       _loopbackReceiver = loopbackReceiver;
    }

private:
    friend class PmSingleton<LoopbackController>; /**< Make the LoopbackController class singleton */

    /**
     * Constructor of LoopbackController class
     *
     * @param[in]
     * @param[out]
     * @param[in,out]
     *
     * @return
     */
    LoopbackController()
    {
       _loopbackReceiver = nullptr;
    }

    /**
     * Destructor of LoopbackController class
     *
     * @param[in]
     * @param[out]
     * @param[in,out]
     *
     * @return
     */
    virtual ~LoopbackController()
    {
       _loopbackReceiver = nullptr;
    }

    ILoopbackReceiver* _loopbackReceiver; /**< pointer to LoopbackReceiver class */
    LockForever _loopbackControllerLock;  /**< Lock for synchronization, controller can be accessed from any thread */
};

} // namespace pmcommon
} // namespace bosch
} // namespace com

#endif //LoopbackController_h
