/**
 * @file JsonReader.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the JsonReader class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file reads, stores and prints the config values from PmConfig Json file.
 *
 * @ingroup PmCommon
 */

#include "JsonReader.h"
#include "PmConfiguration.h"
#include "PmDefaultConfiguration.h"
#include <yajl/yajl_parse.h>
#include <yajl/yajl_gen.h>
#include <yajl/yajl_tree.h>
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/JsonReader.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace com
{
namespace bosch
{
namespace pmcommon
{
   JsonReader::JsonReader()
   {
      ETG_TRACE_USR4(("JsonReader: Constructor"));
   }

   JsonReader::~JsonReader()
   {
      ETG_TRACE_USR4(("JsonReader: Destructor"));
   }

   bool JsonReader::readConfiguration(::std::string configFilePath)
   {
      if (configFilePath.empty())
      {
         ETG_TRACE_FATAL(("readConfiguration: configuration file name is empty"));
         ETG_TRACE_ERRMEM(("readConfiguration: configuration file name is empty"));
         return false;
      }

      const ::std::string configurationFile(configFilePath);
      ETG_TRACE_USR1(("readConfiguration: configurationFile = \"%s\"", configurationFile.c_str()));

      FILE* pFile = fopen(configurationFile.c_str(), "rb");
      if (nullptr == pFile)
      {
         ETG_TRACE_FATAL(("readConfiguration: open file %s failed (NULL pointer returned)",
               configurationFile.c_str()));
         ETG_TRACE_ERRMEM(("readConfiguration: open file %s failed (NULL pointer returned)",
               configurationFile.c_str()));
         return false;
      }

      //Get configuration File size
      fseek (pFile , 0 , SEEK_END);
      const size_t  bufferSize = ftell (pFile) + 1; // including NULL termination
      rewind (pFile);
      char* fileData = nullptr;
      try
      {
         fileData = new char[bufferSize];
      }
      catch(...)
      {
         fclose(pFile);
         return false;
      }
      if (nullptr == fileData)
      {
         ETG_TRACE_FATAL(("readConfiguration: memory allocation failed (NULL pointer returned)"));
         ETG_TRACE_ERRMEM(("readConfiguration: memory allocation failed (NULL pointer returned)"));
         fclose(pFile); // file open was successful, therefore close is necessary
         return false;
      }

      bool result = false;
      size_t rd = fread((void *) fileData, 1, bufferSize , pFile);
      const bool sizeOk = (0 < rd); // check if at least 1 byte was read
      const bool eofSet = (0 != feof(pFile)); // check if end-of-file indicator is set
      const bool errorSet = (0 != ferror(pFile)); // check if error indicator is set

      if(errorSet)
      {
         const int errNumber = errno;

         ETG_TRACE_FATAL(("readConfiguration: read file %60s failed: ERROR=%d (%s)", configurationFile.c_str(),
                        errNumber, strerror(errNumber)));
         ETG_TRACE_ERRMEM(("readConfiguration: read file %60s failed: ERROR=%d (%s)", configurationFile.c_str(),
               errNumber, strerror(errNumber)));
      }

      ETG_TRACE_USR1(("readConfiguration: read result: rd=%u sizeOk=%d eofSet=%d errorSet=%d", rd, sizeOk,
            eofSet, errorSet));

      // file read error handling
      if ((false == sizeOk) || (false == eofSet) || (true == errorSet))
      {
         ETG_TRACE_FATAL(("readConfiguration: could not read configuration file %s, using connectivity's built-in"
               " default configuration values", configurationFile.c_str()));
         ETG_TRACE_ERRMEM(("readConfiguration: could not read configuration file %s, using connectivity's built-in"
               " default configuration values", configurationFile.c_str()));
      }
      else
      {
         yajl_val node;
         char errbuf[256];

         // null plug buffers
         errbuf[0] = 0;
         fileData[rd] = '\0';

         // we have the whole config file in memory.  let's parse it ...
         node = yajl_tree_parse((const char *) fileData, errbuf, sizeof(errbuf));

         // parse error handling
         if (node == nullptr)
         {
            ETG_TRACE_FATAL(("readConfiguration: parsing content of configuration file %s failed,"
                  " using connectivity's built-in default configuration values", configurationFile.c_str()));
            ETG_TRACE_ERRMEM(("readConfiguration: parsing content of configuration file %s failed,"
                  " using connectivity's built-in default configuration values", configurationFile.c_str()));

            //if (strlen(errbuf))
            {
               ETG_TRACE_FATAL(("readConfiguration: yajl parser error message: %s", errbuf));
            }
         }
         else
         {
            auto& pmConfiguration = PmConfiguration::getInstance();
            const char * path_holdTimeAfterSCODisconnection[] = {"holdTimeAfterSCODisconnection", (const char *) nullptr };
            const char * path_holdTimeAfterEndingVRSession[] = {"holdTimeAfterEndingVRSession", (const char *) nullptr };
            const char * path_waitTimeForArlResponse[] = {"waitTimeForArlResponse", (const char *) nullptr };
            const char * path_ecnrDataSetForNBSPhoneCall[] = {"ecnrDataSetForNBSPhoneCall", (const char *) nullptr };
            const char * path_ecnrDataSetForWBSPhoneCall[] = {"ecnrDataSetForWBSPhoneCall", (const char *) nullptr };
            const char * path_ecnrDataSetForNBSVR[] = {"ecnrDataSetForNBSVR", (const char *) nullptr };
            const char * path_ecnrDataSetForWBSVR[] = {"ecnrDataSetForWBSVR", (const char *) nullptr };
            const char * path_ecnrDataSetForNBSSIRI[] = {"ecnrDataSetForNBSSIRI", (const char *) nullptr };
            const char * path_ecnrDataSetForWBSSIRI[] = {"ecnrDataSetForWBSSIRI", (const char *) nullptr };
            const char * path_autoRejectThirdIncomingCall[] = {"autoRejectThirdIncomingCall", (const char *) nullptr };
            const char * path_phoneSubstateSetForHandsetMode[] = {"phoneSubstateSetForHandsetMode", (const char *) nullptr };
            const char * path_maxNumActiveDevices[] = {"maxNumActiveDevices", (const char *) nullptr };
            const char * path_supportedTypeForSIRI[] = {"supportedTypeForSIRI", (const char *) nullptr};
            const char * path_waitTimeForAudioPlayerResponse[] = {"waitTimeForAudioPlayerResponse", (const char *) nullptr};
            const char * path_remoteVolumeControl[] = {"remoteVolumeControl", (const char *) nullptr};
            const char * path_rvcResponseTimeout[] = {"rvcResponseTimeout", (const char *) nullptr};
            const char * path_minAMVolumeLevel[] = {"minAMVolumeLevel", (const char *) nullptr};
            const char * path_maxAMVolumeLevel[] = {"maxAMVolumeLevel", (const char *) nullptr};
            const char * path_gamVoiceRecSourceID[] = {"gamVoiceRecSourceID", (const char *) nullptr};
            const char * path_gamDefaultPhoneSourceXID[] = {"gamDefaultPhoneSourceXID", (const char *) nullptr};
            const char * path_gamDefaultPhoneSourceOffsetID[] = {"gamDefaultPhoneSourceOffsetID", (const char *) nullptr};
            const char * path_gamDefaultPhoneSourceID[] = {"gamDefaultPhoneSourceID", (const char *) nullptr};

            yajl_val value = yajl_tree_get(node, path_holdTimeAfterSCODisconnection, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setHoldTimeAfterSCODisconnection((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_holdTimeAfterSCODisconnection[0],
                     pmConfiguration.getHoldTimeAfterSCODisconnection()));
            }

            value = yajl_tree_get(node, path_holdTimeAfterEndingVRSession, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setHoldTimeAfterEndingVRSession((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_holdTimeAfterEndingVRSession[0],
                     pmConfiguration.getHoldTimeAfterEndingVRSession()));
            }

            value = yajl_tree_get(node, path_waitTimeForArlResponse, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setWaitTimeForArlResponse((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_waitTimeForArlResponse[0],
                     pmConfiguration.getWaitTimeForArlResponse()));
            }

            value = yajl_tree_get(node, path_ecnrDataSetForNBSPhoneCall, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setEcnrDataSetForNBSPhoneCall((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_ecnrDataSetForNBSPhoneCall[0],
                     pmConfiguration.getEcnrDataSetForNBSPhoneCall()));
            }

            value = yajl_tree_get(node, path_ecnrDataSetForWBSPhoneCall, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setEcnrDataSetForWBSPhoneCall((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_ecnrDataSetForWBSPhoneCall[0],
                     pmConfiguration.getEcnrDataSetForWBSPhoneCall()));
            }

            value = yajl_tree_get(node, path_ecnrDataSetForNBSVR, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setEcnrDataSetForNBSVR((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_ecnrDataSetForNBSVR[0],
                     pmConfiguration.getEcnrDataSetForNBSVR()));
            }

            value = yajl_tree_get(node, path_ecnrDataSetForWBSVR, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setEcnrDataSetForWBSVR((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_ecnrDataSetForWBSVR[0],
                     pmConfiguration.getEcnrDataSetForWBSVR()));
            }

            value = yajl_tree_get(node, path_ecnrDataSetForNBSSIRI, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setEcnrDataSetForNBSSIRI((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_ecnrDataSetForNBSSIRI[0],
                     pmConfiguration.getEcnrDataSetForNBSSIRI()));
            }

            value = yajl_tree_get(node, path_ecnrDataSetForWBSSIRI, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setEcnrDataSetForWBSSIRI((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_ecnrDataSetForWBSSIRI[0],
                     pmConfiguration.getEcnrDataSetForWBSSIRI()));
            }

            value = yajl_tree_get(node, path_autoRejectThirdIncomingCall, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setAutoRejectThirdIncomingCall((bool)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_autoRejectThirdIncomingCall[0],
                     pmConfiguration.getAutoRejectThirdIncomingCall()));
            }

            value = yajl_tree_get(node, path_phoneSubstateSetForHandsetMode, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setPhoneSubstateSetForHandsetMode((uint8_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_phoneSubstateSetForHandsetMode[0],
                     pmConfiguration.getPhoneSubstateSetForHandsetMode()));
            }

            value = yajl_tree_get(node, path_maxNumActiveDevices, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setMaxNumActiveDevices((uint8_t)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_maxNumActiveDevices[0],
                     pmConfiguration.getMaxNumActiveDevices()));
            }

            value = yajl_tree_get(node, path_supportedTypeForSIRI, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setSupportedTypeForSIRI((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_supportedTypeForSIRI[0],
                        pmConfiguration.getSupportedTypeForSIRI()));
            }

            value = yajl_tree_get(node, path_waitTimeForAudioPlayerResponse, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setWaitTimeForAudioPlayerResponse((int)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_waitTimeForAudioPlayerResponse[0],
                        pmConfiguration.getWaitTimeForAudioPlayerResponse()));
            }

            value = yajl_tree_get(node, path_remoteVolumeControl, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setRemoteVolumeControl((bool)YAJL_GET_INTEGER(value));
            }
            else
            {
               ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                     " configuration value (= %d)", path_remoteVolumeControl[0],
                     pmConfiguration.getRemoteVolumeControl()));
            }

            value = yajl_tree_get(node, path_rvcResponseTimeout, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
               pmConfiguration.setRvcResponseTimeout((int)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_rvcResponseTimeout[0],
                        pmConfiguration.getRvcResponseTimeout()));
            }

            value = yajl_tree_get(node, path_minAMVolumeLevel, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setMinAMVolumeLevel((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_minAMVolumeLevel[0],
                        pmConfiguration.getMinAMVolumeLevel()));
            }

            value = yajl_tree_get(node, path_maxAMVolumeLevel, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setMaxAMVolumeLevel((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_maxAMVolumeLevel[0],
                        pmConfiguration.getMaxAMVolumeLevel()));
            }

            value = yajl_tree_get(node, path_gamVoiceRecSourceID, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setGAMVoiceRecSourceID((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_gamVoiceRecSourceID[0],
                        pmConfiguration.getGAMVoiceRecSourceID()));
            }

            value = yajl_tree_get(node, path_gamDefaultPhoneSourceXID, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setGAMDefaultPhoneSourceXID((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_gamDefaultPhoneSourceXID[0],
                        pmConfiguration.getGAMDefaultPhoneSourceXID()));
            }

            value = yajl_tree_get(node, path_gamDefaultPhoneSourceOffsetID, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setGAMDefaultPhoneSourceOffsetID((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_gamDefaultPhoneSourceOffsetID[0],
                        pmConfiguration.getGAMDefaultPhoneSourceOffsetID()));
            }

            value = yajl_tree_get(node, path_gamDefaultPhoneSourceID, yajl_t_number);

            if (true == YAJL_IS_INTEGER(value))
            {
                pmConfiguration.setGAMDefaultPhoneSourceID((uint16_t)YAJL_GET_INTEGER(value));
            }
            else
            {
                ETG_TRACE_ERR(("readConfiguration: parameter %30s does not exist (or) incorrect format, using default"
                        " configuration value (= %d)", path_gamDefaultPhoneSourceID[0],
                        pmConfiguration.getGAMDefaultPhoneSourceID()));
            }

            yajl_tree_free(node);

            result = true;
         }
      }

      delete[] fileData;
      fclose(pFile);

      return result;
   }

   void JsonReader::traceConfiguration()
   {
      ETG_TRACE_USR1(("tracePmConfiguration() entered"));

      auto& pmConfiguration = PmConfiguration::getInstance();

      ETG_TRACE_USR1(("tracePmConfiguration: HoldTimeAfterSCODisconnection = %d",
            pmConfiguration.getHoldTimeAfterSCODisconnection()));

      ETG_TRACE_USR1(("tracePmConfiguration: HoldTimeAfterEndingVRSession = %d",
            pmConfiguration.getHoldTimeAfterEndingVRSession()));

      ETG_TRACE_USR1(("tracePmConfiguration: WaitTimeForArlResponse = %d",
            pmConfiguration.getWaitTimeForArlResponse()));

      ETG_TRACE_USR1(("tracePmConfiguration: EcnrDataSetForNBSPhoneCall = %d",
            pmConfiguration.getEcnrDataSetForNBSPhoneCall()));

      ETG_TRACE_USR1(("tracePmConfiguration: EcnrDataSetForWBSPhoneCall = %d",
            pmConfiguration.getEcnrDataSetForWBSPhoneCall()));

      ETG_TRACE_USR1(("tracePmConfiguration: EcnrDataSetForNBSVR = %d",
            pmConfiguration.getEcnrDataSetForNBSVR()));

      ETG_TRACE_USR1(("tracePmConfiguration: EcnrDataSetForWBSVR = %d",
            pmConfiguration.getEcnrDataSetForWBSVR()));

      ETG_TRACE_USR1(("tracePmConfiguration: EcnrDataSetForNBSSIRI = %d",
            pmConfiguration.getEcnrDataSetForNBSSIRI()));

      ETG_TRACE_USR1(("tracePmConfiguration: EcnrDataSetForWBSSIRI = %d",
            pmConfiguration.getEcnrDataSetForWBSSIRI()));

      ETG_TRACE_USR1(("tracePmConfiguration: AutoRejectThirdIncomingCall = %d",
            pmConfiguration.getAutoRejectThirdIncomingCall()));

      ETG_TRACE_USR1(("tracePmConfiguration: PhoneSubstateSetForHandsetMode = %d",
            pmConfiguration.getPhoneSubstateSetForHandsetMode()));

      ETG_TRACE_USR1(("tracePmConfiguration: MaxNumActiveDevices = %d",
            pmConfiguration.getMaxNumActiveDevices()));

      ETG_TRACE_USR1(("tracePmConfiguration: SupportedTypeForSIRI = %d",
            pmConfiguration.getSupportedTypeForSIRI()));

      ETG_TRACE_USR1(("tracePmConfiguration: WaitTimeForAudioPlayerResponse = %d",
            pmConfiguration.getWaitTimeForAudioPlayerResponse()));

      ETG_TRACE_USR1(("tracePmConfiguration: RemoteVolumeControl = %d",
            pmConfiguration.getRemoteVolumeControl()));

      ETG_TRACE_USR1(("tracePmConfiguration: RvcResponseTimeout = %d",
            pmConfiguration.getRvcResponseTimeout()));

      ETG_TRACE_USR1(("tracePmConfiguration: MinAMVolumeLevel = %d",
            pmConfiguration.getMinAMVolumeLevel()));

      ETG_TRACE_USR1(("tracePmConfiguration: MaxAMVolumeLevel = %d",
            pmConfiguration.getMaxAMVolumeLevel()));

      ETG_TRACE_USR1(("tracePmConfiguration: GAMVoiceRecSourceID = %d",
            pmConfiguration.getGAMVoiceRecSourceID()));

      ETG_TRACE_USR1(("tracePmConfiguration: GAMDefaultPhoneSourceXID = %d",
            pmConfiguration.getGAMDefaultPhoneSourceXID()));

      ETG_TRACE_USR1(("tracePmConfiguration: GAMDefaultPhoneSourceOffsetID = %d",
            pmConfiguration.getGAMDefaultPhoneSourceOffsetID()));

      ETG_TRACE_USR1(("tracePmConfiguration: GAMDefaultPhoneSourceID = %d",
            pmConfiguration.getGAMDefaultPhoneSourceID()));

   }

} // namespace pmcommon
} // namespace bosch
} // namespace com
