/**
 * @file PmUtils.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of utility functions used by PhoneCallManager
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details.
 *
 * @ingroup
 */

#include "PmUtils.h"
#include "PmUnistd.h"

namespace com
{
namespace bosch
{
namespace pmcommon
{

#define PM_DIAL_NUMBER_MAX_LEN_IN_CHARS 85    // 85 to support Evolution stack
#define WAITINGMODE_TONE_PATH "/var/opt/bosch/static/connectivity/tones/waitingmodetones/"
#define COPY_ONE_TIME 1

void validateDialNumber(const std::string& telephoneNumber, std::string &validTelephoneNumber)
{
   if(telephoneNumber.empty())
   {
      return;
   }

   // Phone number check:
   // Extract only "valid" characters of given phone number (see GIS-327-3259 Dialing String Characters),
   // valid characters are: 1 2 3 4 5 6 7 8 9 0 * # + A B C and W; (Note: All are ASCII characters)
   unsigned long u32SrcIdx  = 0;
   unsigned long u32DestIdx = 0;
   unsigned long u32SrcLen  = static_cast<unsigned long>(telephoneNumber.length());
   unsigned char cPhoneNumChar = ' ';

   for( u32SrcIdx = 0; u32SrcIdx < u32SrcLen; u32SrcIdx++)
   {
      cPhoneNumChar = telephoneNumber[u32SrcIdx];

      // Copying is stopped if PM_DIAL_NUMBER_MAX_LEN_IN_CHARS characters are available in validTelephoneNumber
      if(u32DestIdx < PM_DIAL_NUMBER_MAX_LEN_IN_CHARS)
      {
         // First we check if highest bit is NOT set to 1.
         // If so, we have an single ASCII character (and not an UTF-8 multi-byte character).
         if( (cPhoneNumChar & 0x80) != 0x80 )
         {
            // We have an single ASCII character and can continue with more detailed checks:

            // Check for digit, i.e. range: 0x30 ('0') to 0x39 ('9')
            if(((cPhoneNumChar >= 0x30) && (cPhoneNumChar <= 0x39))

                  // Check for: '*' (0x2A), '#' (0x23), '+' (0x2B)and ',' (0x2C)
                  || ((cPhoneNumChar == 0x2A) || (cPhoneNumChar == 0x23)
                        || (cPhoneNumChar == 0x2B) || (cPhoneNumChar == 0x2C))

                        // Check for 'A', 'B', 'C', i.e range: 0x41 to 0x43
                        ||((cPhoneNumChar >= 0x41) && (cPhoneNumChar <= 0x43))

                        // Check for 'W' (0x57)
                        || (cPhoneNumChar == 0x57)
            )
            {
               // Copy 'cPhoneNumChar' to valid phone number
               validTelephoneNumber.append(COPY_ONE_TIME, cPhoneNumChar);
               u32DestIdx++;
            }
         }
      }  // end of: if(u32DestIdx < PM_DIAL_NUMBER_MAX_LEN_IN_CHARS)
      else
      {
         break;
         // Maximum number of digits in destination 'validTelephoneNumber' is reached.
         // No need to run the for loop any longer.
      }
   }
}

void validateWaitingModeFilePath(const std::string& filePath, std::string& completeFilePath)
{
   if(0 == filePath.length())
      return;

   if (std::string::npos == filePath.find("/"))
   {
      completeFilePath = WAITINGMODE_TONE_PATH + filePath;
   }
   else
   {
      completeFilePath = filePath;
   }

   if((access(completeFilePath.c_str(), F_OK)) != -1)
   {
      // File is present in the path
   }
   else
   {
      completeFilePath = "";
   }
}

} // namespace pmcommon
} // namespace bosch
} // namespace com
