/**
 * @file PmComponent.cpp
 *
 * @swcomponent PhoneCallManager 
 *
 * @brief This file contains the definition of the PmComponent class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This is the main file of phone call manager.
 *          It makes the component as an ASF based component.
 *
 * @ingroup PhoneCallManager
 */

#include "PmComponent.h"
#include "LoopbackController.h"
#include "PhoneCallManager.h"
#include "IpcWrapper.h"
#include "PmAppTrace.h"

#include <systemd/sd-daemon.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_PM_APPLICATION
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmComponent.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_PM_APPLICATION
#endif
#endif

namespace com
{
namespace bosch
{

PmComponent::PmComponent():BaseComponent(), _lcmAppComponent(/*busname */"com.bosch.PmApp",
      "/org/genivi/NodeStateManager/LifeCycleConsumer/PmApp",
      *this)
{
   // TODO: Check if it is really required to call this macro to open the ETG trace logging mechanism.
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ET_TRACE_OPEN;
#endif
   // TODO: All the Instrumentation code is added for testing purpose only, will be removed soon.
   ETG_TRACE_USR4(("Instrumentation purpose :PmComponent::PmComponent() entered"));
   ETG_TRACE_ERRMEM (("Instrumentation purpose PmComponent::PmComponent() entered"));

   // Instantiate the singleton instance of phone call manager
   PhoneCallManager::getInstance();
   pm_ipc_wrapper::IpcWrapper::getInstance().setPmComponent(this);
   ETG_TRACE_ERRMEM(("Instrumentation purpose : PmComponent Before setAppStartReady"));
   ETG_TRACE_USR4(("Instrumentation purpose :PmComponent Before setAppStartReady"));

   _lcmAppComponent.setAppStartReady();

   ETG_TRACE_ERRMEM(("Instrumentation purpose : PmComponent After setAppStartReady"));
   ETG_TRACE_USR4(("Instrumentation purpose :PmComponent After setAppStartReady"));
   ETG_TRACE_USR4(("PmComponent::PmComponent() exit"));
}

PmComponent::~PmComponent()
{
   ETG_TRACE_USR4(("PmComponent::~PmComponent() entered"));

   // TODO: check if it really required to close the ETG Trace 
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ET_TRACE_CLOSE;
#endif
}

void PmComponent::onLocalMessage(::boost::shared_ptr< PayloadTuple1 < pmcommon::LoopbackData*> >& payload)
{
   ETG_TRACE_USR4(("PmComponent::onLocalMessage() entered"));

   if(nullptr != payload)
   {
      pmcommon::LoopbackController::getInstance().onLoopbackMessage(payload->getAct(), payload->get0());
   }
   else
   {
      ETG_TRACE_ERR(("Received LocalMessage payload is null"));
   }
}

void PmComponent::onLoadPersistency()
{
   // TODO: All the Instrumentation code is added for testing purpose only, will be removed soon.
   //ETG_TRACE_USR4(("PmComponent::onLoadPersistency() entered"));
   // Start time measurement
   struct timespec time1 = { 0, 0 };
   struct timespec time2 = { 0, 0 };
   struct timespec diff_time = { 0, 0 };
   clock_gettime(CLOCK_MONOTONIC, &time1);


   ETG_TRACE_FATAL ((" Instrumentation purpose -PM_APP_STATE_CHANGE- onLoadPersistency() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));
   // EM trace added temporarily for tracking of State change issue:
   ETG_TRACE_ERRMEM((" Instrumentation purpose -PM_APP_STATE_CHANGE- onLoadPersistency() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));

   pm_ipc_wrapper::IpcWrapper::getInstance().onSystemStateChanged(pmcore::PM_SPM_SYSTEM_ON);
   PhoneCallManager::getInstance().getPmAppCcaServer()->setServiceAvailable();

   // Evaluation of needed time
   clock_gettime(CLOCK_MONOTONIC, &time2);
   if ((time2.tv_nsec - time1.tv_nsec) < 0)
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec - 1;
      diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
   }
   else
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec;
      diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
   }
   ETG_TRACE_FATAL ((" Instrumentation purpose -PM_APP_STATE_CHANGE- onLoadPersistency() exited, needed time= [%5d] msec ",
         diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));

   // EM trace added temporarily for tracking of State change issue:
   ETG_TRACE_ERRMEM((" Instrumentation purpose -PM_APP_STATE_CHANGE- onLoadPersistency() exited, needed time= [%5lu] msec ",
         diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));
}

void PmComponent::onSavePersistency()
{
   // TODO: All the Instrumentation code is added for testing purpose only, will be removed soon.
   //ETG_TRACE_USR4(("PmComponent::onSavePersistency() entered"));
   // Start time measurement
   struct timespec time1 = { 0, 0 };
   struct timespec time2 = { 0, 0 };
   struct timespec diff_time = { 0, 0 };
   clock_gettime(CLOCK_MONOTONIC, &time1);


   ETG_TRACE_FATAL ((" Instrumentation purpose -PM_APP_STATE_CHANGE- onSavePersistency() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));
   // EM trace added temporarily for tracking of State change issue:
   ETG_TRACE_ERRMEM((" Instrumentation purpose -PM_APP_STATE_CHANGE- onSavePersistency() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));

   pm_ipc_wrapper::IpcWrapper::getInstance().onSystemStateChanged(pmcore::PM_SPM_SYSTEM_OFF);

   // Evaluation of needed time
   clock_gettime(CLOCK_MONOTONIC, &time2);
   if ((time2.tv_nsec - time1.tv_nsec) < 0)
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec - 1;
      diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
   }
   else
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec;
      diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
   }
   ETG_TRACE_FATAL ((" Instrumentation purpose -PM_APP_STATE_CHANGE- onSavePersistency() exited, needed time= [%5d] msec ",
         diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));

   // EM trace added temporarily for tracking of State change issue:
   ETG_TRACE_ERRMEM((" Instrumentation purpose -PM_APP_STATE_CHANGE- onSavePersistency() exited, needed time= [%5lu] msec ",
         diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));
}

_NsmErrorStatus_e PmComponent::OnAppModeChange(_Nsm_Shutdown_Type_e newAppMode, _Nsm_Shutdown_Type_e oldAppMode)
{
   // TODO: All the Instrumentation code is added for testing purpose only, will be removed soon.
   ETG_TRACE_USR4(("PmComponent::OnAppModeChange() entered with oldAppMode: %u and newAppMode: %u",
         ETG_CENUM(_Nsm_Shutdown_Type_e, oldAppMode), ETG_CENUM(_Nsm_Shutdown_Type_e, newAppMode)));
   ETG_TRACE_ERRMEM(("Instrumentation purpose : PmComponent::OnAppModeChange() entered with oldAppMode: %u and newAppMode: %u",
         ETG_CENUM(_Nsm_Shutdown_Type_e, oldAppMode), ETG_CENUM(_Nsm_Shutdown_Type_e, newAppMode)));

   return _NsmErrorStatus_e__NsmErrorStatus_Ok;
}

bool PmComponent::OnAppWatchdog()
{
   ETG_TRACE_USR4(("PmComponent::OnAppWatchdog() entered"));
   return true;
}

void PmComponent::OnLcmRegistered()
{
   // TODO: All the Instrumentation code is added for testing purpose only, will be removed soon.
   ETG_TRACE_USR4(("Instrumentation purpose PmComponent::OnLcmRegistered() entered"));
   ETG_TRACE_ERRMEM (("Instrumentation purpose PmComponent::OnLcmRegistered() entered"));
}

} // namespace bosch
} // namespace com
