/**
 * @file YakinduSmAudioSessionIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmAudioSessionIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of audio session
 *
 * @ingroup PmAudioManager
 */

#include "YakinduSmAudioSessionIf.h"
#include "AmMainController.h"
#include "AudioSessionHandler.h"
#include "AmIfMessageRequest.h"
#include "AmIfMessageResult.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/YakinduSmAudioSessionIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
   YakinduSmAudioSessionIf::YakinduSmAudioSessionIf():_sessionIdToSessionHandlerMap()
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf: Constructor"));
   }

   YakinduSmAudioSessionIf::~YakinduSmAudioSessionIf()
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf: Destructor"));

      clearSessionIdToSessionHandlerMap();
   }

   void YakinduSmAudioSessionIf::prepareAudioRouteRequest(
         IN std::shared_ptr<AmIfMessage_PrepareAudioRouteRequest> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::prepareAudioRouteRequest() entered"));

      AmResult amResult(AM_RESULT_ERR_GENERAL, "");

      if(AM_SESSION_ID_DEFAULT == amIfMessage->getAmSessionId())
      {
         static AmSessionId amSessionId = 0x01; // = 0; TODO fix this later
         //amSessionId++;

         ETG_TRACE_USR4(("YakinduSmAudioSessionIf::prepareAudioRouteRequest: SessionID created : %d", amSessionId));

         AudioSessionHandler* audioSessionHandler = new AudioSessionHandler(amSessionId);
         if(nullptr != audioSessionHandler)
         {
            amIfMessage->setAmSessionId(amSessionId);

            _sessionIdToSessionHandlerMap.insert(std::pair<AmSessionId, AudioSessionHandler*>(amSessionId,
                  audioSessionHandler));

            audioSessionHandler->prepareAudioRouteRequest(amIfMessage);

            amResult._amResultCode = AM_RESULT_OK;
         }
         else
         {
            ETG_TRACE_ERR(("YakinduSmAudioSessionIf::prepareAudioRouteRequest: audioSessionHandler is null"));
         }
      }
      else
      {
         ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
               amIfMessage->getAmSessionId());

         if(it != _sessionIdToSessionHandlerMap.end())
         {
            it->second->prepareAudioRouteRequest(amIfMessage);

            amResult._amResultCode = AM_RESULT_OK;
         }
         else
         {
            ETG_TRACE_ERR(("YakinduSmAudioSessionIf:: SM not exist for SessionId %d", amIfMessage->getAmSessionId()));
         }
      }

      AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().prepareAudioRouteResponse(
            amIfMessage->getAmSessionId(), amResult);
   }

   void YakinduSmAudioSessionIf::playAudioFileRequest(IN std::shared_ptr<AmIfMessage_PlayAudioFileRequest> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::playAudioFileRequest() entered"));

      AmResult amResult(AM_RESULT_ERR_GENERAL, "");

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->playAudioFileRequest(amIfMessage);

         amResult._amResultCode = AM_RESULT_OK;
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::playAudioFileRequest: AmSessionId not found"));
      }

      AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().playAudioResponse(
            amIfMessage->getAmSessionId(), amResult);
   }

   void YakinduSmAudioSessionIf::playHfAudioRequest(IN std::shared_ptr<AmIfMessage_PlayHfAudioRequest> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::playHfAudioRequest() entered"));

      AmResult amResult(AM_RESULT_ERR_GENERAL, "");

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->playHfAudioRequest(amIfMessage);

         amResult._amResultCode = AM_RESULT_OK;
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::playHfAudioRequest: AmSessionId not found"));
      }

      AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().playAudioResponse(
            amIfMessage->getAmSessionId(), amResult);
   }

   void YakinduSmAudioSessionIf::stopAudioRequest(IN std::shared_ptr<AmIfMessage_StopAudioRequest> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::stopAudioRequest() entered"));

      AmResult amResult(AM_RESULT_ERR_GENERAL, "");

      AmSessionId amSessionId = amIfMessage->getAmSessionId();
      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(amSessionId);

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->stopAudioRequest(amIfMessage);

         amResult._amResultCode = AM_RESULT_OK;

         //On receiving stopAudio() before receiving the response for bSetAudioRouteRequest(TRUE)[i.e. bOnallocate()],
         //SESSION_REMOVED event should be sent to PmCore, which is required to remove the SessionId in PMCore.
         removeSessionHandlerFromMap(amSessionId);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::stopAudioRequest: AmSessionId not found"));
      }

      AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().stopAudioResponse(
            amIfMessage->getAmSessionId(), amResult);
   }

   void YakinduSmAudioSessionIf::pauseAudioRequest(IN std::shared_ptr<AmIfMessage_PauseAudioRequest> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::pauseAudioRequest() entered"));

      AmResult amResult(AM_RESULT_ERR_GENERAL, "");

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->pauseAudioRequest(amIfMessage);

         amResult._amResultCode = AM_RESULT_OK;
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::pauseAudioRequest: AmSessionId not found"));
      }

      AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().pauseAudioResponse(
            amIfMessage->getAmSessionId(), amResult);
   }

   void YakinduSmAudioSessionIf::setMicMuteStateRequest(IN std::shared_ptr<AmIfMessage_SetMicMuteStateRequest> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::setMicMuteStateRequest() entered"));

      AmResult amResult(AM_RESULT_ERR_GENERAL, "");

      // TODO: Later when Session ID related changes are done use the sessionId
      // from the amIfMessage instead of hardcoding
      const AmSessionId sessionId = 0x01;

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(sessionId);

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->setMicMuteStateRequest(amIfMessage);
         amResult._amResultCode = AM_RESULT_OK;
      }
      else
      {
         // Indicates that there is no active session available
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::setMicMuteStateRequest: AmSessionId not found"));
      }

      AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().setMicMuteStateResponse(
            amIfMessage->getAmSessionId(), amResult);
   }

   void YakinduSmAudioSessionIf::ecnrInitializeResult(
         IN std::shared_ptr<AmIfMessage_EcnrInitializeResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::ecnrInitializeResult() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->ecnrInitializeResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::ecnrInitializeResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::ecnrDestroyResult(IN std::shared_ptr<AmIfMessage_EcnrDestroyResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::ecnrDestroyResult() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->ecnrDestroyResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::ecnrDestroyResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::ecnrSetConfigurationResult(
         IN std::shared_ptr<AmIfMessage_EcnrSetConfigurationResult> amIfMessage)
   {
      ETG_TRACE_USR4(("ecnrSetConfigurationResult() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->ecnrSetConfigurationResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::ecnrSetConfigurationResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::ecnrSetSendMuteSwitchResult(
         IN std::shared_ptr<AmIfMessage_EcnrSetSendMuteSwitchResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::ecnrSetSendMuteSwitchResult() entered"));

      // TODO: Later when Session ID related changes are done use the sessionId
      // from the amIfMessage instead of hardcoding
      const AmSessionId sessionId = 0x01;

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(sessionId);

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->ecnrSetSendMuteSwitchResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::ecnrSetSendMuteSwitchResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::ecnrStartAudioResult(IN std::shared_ptr<AmIfMessage_EcnrStartAudioResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::ecnrStartAudioResult() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->ecnrStartAudioResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::ecnrStartAudioResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::ecnrStopAudioResult(IN std::shared_ptr<AmIfMessage_EcnrStopAudioResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::ecnrStopAudioResult() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->ecnrStopAudioResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::ecnrStopAudioResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::ecnrGetVersionResult(IN std::shared_ptr<AmIfMessage_EcnrGetVersionResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::ecnrGetVersionResult() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->ecnrGetVersionResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::ecnrGetVersionResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::requestAudioRouteResult(
         IN std::shared_ptr<AmIfMessage_RequestAudioRouteResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::requestAudioRouteResult() entered"));

      AmSessionId sessionId = amIfMessage->getAmSessionId();
      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(sessionId);

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->requestAudioRouteResult(amIfMessage);

         //If there is no response from ARL for source allocation request, then after timeout
         //SESSION_REMOVED event should be sent to PmCore, which is required to remove the SessionId in PMCore.
         removeSessionHandlerFromMap(sessionId);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::requestAudioRouteResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::releaseAudioRouteResult(
         IN std::shared_ptr<AmIfMessage_ReleaseAudioRouteResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::releaseAudioRouteResult() entered"));

      AmSessionId sessionId = amIfMessage->getAmSessionId();
      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(sessionId);

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         if(nullptr != it->second)
         {
            it->second->releaseAudioRouteResult(amIfMessage);

            //If all available channels are released, then SESSION_REMOVED event should be sent to PmCore.
            //This is required to remove the SessionId in PMCore.
            removeSessionHandlerFromMap(sessionId);
         }
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::releaseAudioRouteResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::sourceActivityResult(
         IN std::shared_ptr<AmIfMessage_SourceActivityResult> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::sourceActivityResult() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->sourceActivityResult(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::sourceActivityResult: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::playbackStatus(IN std::shared_ptr<AmIfMessage_PlaybackStatus> amIfMessage)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::playbackStatus() entered"));

      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(
            amIfMessage->getAmSessionId());

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         it->second->playbackStatus(amIfMessage);
      }
      else
      {
         ETG_TRACE_ERR(("YakinduSmAudioSessionIf::playbackStatus: AmSessionId not found"));
      }
   }

   void YakinduSmAudioSessionIf::removeSessionHandlerFromMap(const AmSessionId amSessionId)
   {
      ETG_TRACE_USR4(("YakinduSmAudioSessionIf::removeSessionHandlerFromMap() entered"));
      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it = _sessionIdToSessionHandlerMap.find(amSessionId);

      if(it != _sessionIdToSessionHandlerMap.end())
      {
         if(nullptr != it->second)
         {
            if(true != it->second->isAnyAudioChannelActive())
            {
               delete it->second;

               _sessionIdToSessionHandlerMap.erase(it);

               AmEventDetails eventInfo(SESSION_REMOVED, "");

               AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().amNotificationEvent(amSessionId,
                     AM_UNKNOWN, STOPPED, eventInfo);
            }
         }
      }
   }

   void YakinduSmAudioSessionIf::clearSessionIdToSessionHandlerMap()
   {
      ::std::map<AmSessionId, AudioSessionHandler*>::iterator it;
      for(it = _sessionIdToSessionHandlerMap.begin(); it != _sessionIdToSessionHandlerMap.end(); it++)
      {
         if(nullptr != it->second)
         {
            delete it->second;
         }
      }
      _sessionIdToSessionHandlerMap.clear();
   }
}
