/**
 * @file VoiceRecognitionAudioHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VoiceRecognitionAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice recognition audio
 *
 * @ingroup PmAudioManager
 */

#include "VoiceRecAudioHandler.h"
#include "AmMainController.h"
#include "AudioSessionHandler.h"
#include "ArlWrapper.h"
#include "EcnrWrapper.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/VoiceRecAudioHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
   VoiceRecAudioHandler::VoiceRecAudioHandler()
   {
      _smAudioWrapperReqIf = nullptr;
      _smEcnrRequestIf = nullptr;
      _smVoiceRecAudioSmHandlerIf = nullptr;
      _audioSessionHandler = nullptr;
      _smTraceObserverIf = nullptr;
   }

   VoiceRecAudioHandler::VoiceRecAudioHandler(AudioSessionHandler* audioSessionHandler)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler: Parameterized Constructor"));

      _smAudioWrapperReqIf = new VoiceRecAudioHandler::SmAudioWrapperReqIf();

      if(nullptr != _smAudioWrapperReqIf)
      {
         setSCI_AudioWrapperReqIf_OCB(_smAudioWrapperReqIf);
      }

      _smEcnrRequestIf = new VoiceRecAudioHandler::SmEcnrRequestIf(*this);

      if(nullptr != _smEcnrRequestIf)
      {
         setSCI_EcnrRequest_OCB(_smEcnrRequestIf);
      }

      _smVoiceRecAudioSmHandlerIf = new VoiceRecAudioHandler::SmVoiceRecAudioSmHandlerIf(*this);

      if(nullptr != _smVoiceRecAudioSmHandlerIf)
      {
         setSCI_VoiceRecAudioSmHandler_OCB(_smVoiceRecAudioSmHandlerIf);
      }

      _audioSessionHandler = audioSessionHandler;

      _smTraceObserverIf = new VoiceRecAudioHandler::SmTraceObserverIf<VoiceRecAudioSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      //initialize the state machine
      init();

      enter();
   }

   VoiceRecAudioHandler::~VoiceRecAudioHandler()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler: Destructor"));

      if(nullptr != _smAudioWrapperReqIf)
      {
         delete _smAudioWrapperReqIf;
         _smAudioWrapperReqIf = nullptr;
      }

      if(nullptr != _smEcnrRequestIf)
      {
         delete _smEcnrRequestIf;
         _smEcnrRequestIf = nullptr;
      }

      if(nullptr != _smVoiceRecAudioSmHandlerIf)
      {
         delete _smVoiceRecAudioSmHandlerIf;
         _smVoiceRecAudioSmHandlerIf = nullptr;
      }

      _audioSessionHandler = nullptr;

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   void VoiceRecAudioHandler::prepareVoiceRecAudio()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:prepareVoiceRecAudio"));

      getSCI_AmClientRequest()->raise_pREPARE_VOICEREC_CHANNEL();
   }

   void VoiceRecAudioHandler::pauseVoiceRecAudio()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:pauseVoiceRecAudio"));

      getSCI_AmClientRequest()->raise_pAUSE_VOICEREC_AUDIO();
   }

   void VoiceRecAudioHandler::stopVoiceRecAudio()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:stopVoiceRecAudio"));

      getSCI_AmClientRequest()->raise_sTOP_VOICEREC_AUDIO();
   }

   void VoiceRecAudioHandler::playHfAudio(const SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:playHfAudio : %u", ETG_CENUM(SamplingType, samplingType)));

      getSCI_AmClientRequest()->raise_sTREAM_VOICEREC_AUDIO(samplingType);
   }

   void VoiceRecAudioHandler::onAudioManagerResponse(const AudioChannelStatus status)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:onAudioManagerResponse: %u",
            ETG_CENUM(AudioChannelStatus, status)));

      switch(status)
      {
         case CHANNEL_GRANTED:
            getSCI_AudioWrapperResponse()->raise_vOICEREC_AUDIO_CHANNEL_GRANTED();
            break;
         case CHANNEL_DENIED:
            getSCI_AudioWrapperResponse()->raise_vOICEREC_AUDIO_CHANNEL_DENIED();
            break;
         case CHANNEL_DEALLOCATED:
         default:
            getSCI_AudioWrapperResponse()->raise_vOICEREC_AUDIO_CHANNEL_DEALLOCATED();
            break;
      }
   }

   void VoiceRecAudioHandler::onSourceActivityResult(const SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:onSourceActivityResult: SrcActivity : %u",
            ETG_CENUM(SrcActivity, srcActivity)));

      switch(srcActivity)
      {
         case SRC_ACTIVITY_OFF:
            getSCI_AudioWrapperResponse()->raise_vOICEREC_AUDIO_SA_OFF();
            break;
         case SRC_ACTIVITY_ON:
            getSCI_AudioWrapperResponse()->raise_vOICEREC_AUDIO_SA_ON();
            break;
         case SRC_ACTIVITY_PAUSE:
            getSCI_AudioWrapperResponse()->raise_vOICEREC_AUDIO_SA_PAUSE();
            break;
         default:
            getSCI_AudioWrapperResponse()->raise_vOICEREC_AUDIO_SA_OFF();
            break;
      }
   }

   void VoiceRecAudioHandler::onEcnrResponse(const EcnrReponse response, const AmResultMessage& msg)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:onEcnrResponse: %u", ETG_CENUM(EcnrReponse, response)));

      switch(response)
      {
         case ECNR_INIT_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_INIT_FAILURE(msg);
            break;
         case ECNR_INIT_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_INIT_SUCCESS();
            break;
         case ECNR_DESTROY_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_DESTROY_FAILURE(msg);
            break;
         case ECNR_DESTROY_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_DESTROY_SUCCESS();
            break;
         case ECNR_SET_CONFIGURATION_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_SET_CONFIGURATION_FAILURE(msg);
            break;
         case ECNR_SET_CONFIGURATION_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_SET_CONFIGURATION_SUCCESS();
            break;
         case ECNR_START_AUDIO_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_START_FAILURE(msg);
            break;
         case ECNR_START_AUDIO_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_START_SUCCESS();
            break;
         case ECNR_STOP_AUDIO_FAILURE:
            getSCI_EcnrResponse()->raise_eCNR_STOP_FAILURE(msg);
            break;
         case ECNR_STOP_AUDIO_SUCCESS:
            getSCI_EcnrResponse()->raise_eCNR_STOP_SUCCESS();
            break;
         default:
            break;
      }
   }

   void VoiceRecAudioHandler::SmAudioWrapperReqIf::sendAllocateRequest()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().requestAudioRoute(AM_VOICERECOGNITION);
   }

   void VoiceRecAudioHandler::SmAudioWrapperReqIf::sendDeAllocateRequest()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendDeAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().releaseAudioRoute(AM_VOICERECOGNITION);
   }

   void VoiceRecAudioHandler::SmAudioWrapperReqIf::sendSrcActivityResult(SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendSrcActivityResult: SrcActivity : %u",
            ETG_CENUM(SrcActivity, srcActivity)));

      (void)AmMainController::getInstance().getArlWrapper().sendSourceActivityResult(AM_VOICERECOGNITION, srcActivity);
   }

   AmResultCode VoiceRecAudioHandler::SmEcnrRequestIf::sendEcnrInit(SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendEcnrInit: SamplingType : %u",
            ETG_CENUM(SamplingType, samplingType)));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceRecAudioHandler._audioSessionHandler)
      {
         auto& pmConfiguration = com::bosch::pmcommon::PmConfiguration::getInstance();
         EcnrConfigurationId configId = pmConfiguration.getEcnrDataSetForWBSVR();

         switch (samplingType)
         {
            case NARROWBAND:
               configId = pmConfiguration.getEcnrDataSetForNBSVR();
               break;
            case NARROWBAND_SIRI:
               configId = pmConfiguration.getEcnrDataSetForNBSSIRI();
               break;
            case WIDEBAND_SIRI:
               configId = pmConfiguration.getEcnrDataSetForWBSSIRI();
               break;
            default:
               ETG_TRACE_USR4(("Default case"));
         }

         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().initialize(
               _voiceRecAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE, configId))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceRecAudioHandler::SmEcnrRequestIf::sendEcnrSetConfiguration(SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendEcnrSetConfiguration:  SamplingType : %u",
            ETG_CENUM(SamplingType, samplingType)));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceRecAudioHandler._audioSessionHandler)
      {
         auto& pmConfiguration = com::bosch::pmcommon::PmConfiguration::getInstance();
         EcnrConfigurationId configId = pmConfiguration.getEcnrDataSetForWBSVR();

         switch (samplingType)
         {
            case NARROWBAND:
               configId = pmConfiguration.getEcnrDataSetForNBSVR();
               break;
            case NARROWBAND_SIRI:
               configId = pmConfiguration.getEcnrDataSetForNBSSIRI();
               break;
            case WIDEBAND_SIRI:
               configId = pmConfiguration.getEcnrDataSetForWBSSIRI();
               break;
            default:
               ETG_TRACE_USR4(("Default case"));
         }

         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().setConfiguration(
               _voiceRecAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE, configId))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceRecAudioHandler::SmEcnrRequestIf::sendEcnrStart()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendEcnrStart"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceRecAudioHandler._audioSessionHandler)
      {
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().startAudio(
               _voiceRecAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceRecAudioHandler::SmEcnrRequestIf::sendEcnrStop()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendEcnrStop"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceRecAudioHandler._audioSessionHandler)
      {
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().stopAudio(
               _voiceRecAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode VoiceRecAudioHandler::SmEcnrRequestIf::sendEcnrDestroy()
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:sendEcnrDestroy"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _voiceRecAudioHandler._audioSessionHandler)
      {
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().destroy(
               _voiceRecAudioHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   void VoiceRecAudioHandler::SmVoiceRecAudioSmHandlerIf::updateSamplingType(SamplingType samplingType)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:updateSamplingType : %u", ETG_CENUM(SamplingType, samplingType)));

      _voiceRecAudioHandler.getSCI_VoiceRecAudioSmHandler()->set_samplingType(samplingType);
   }

   void VoiceRecAudioHandler::SmVoiceRecAudioSmHandlerIf::updateStatus(RouteState routeState,
         AmResultMessage msg)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler:updateStatus : RouteState : %u, Message : %50s",
            ETG_CENUM(RouteState, routeState), msg.c_str()));

      if(nullptr != _voiceRecAudioHandler._audioSessionHandler)
      {
         _voiceRecAudioHandler._audioSessionHandler->updateStatus(AM_VOICERECOGNITION, routeState, msg);
      }
   }

   template<typename T>
   void VoiceRecAudioHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler::stateEntered()::%u", ETG_CENUM(VoiceRecAudioSmStates, state)));
   }

   template<typename T>
   void VoiceRecAudioHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("VoiceRecAudioHandler::stateExited()::%u", ETG_CENUM(VoiceRecAudioSmStates, state)));
   }
}
