/**
 * @file VehicleRingtoneAudioHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VehicleRingtoneAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of vehicle ringtone audio
 *
 * @ingroup PmAudioManager
 */

#include "VehicleRingtoneAudioHandler.h"
#include "AmMainController.h"
#include "AudioSessionHandler.h"
#include "ArlWrapper.h"
#include "AudioPlayerWrapper.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/VehicleRingtoneAudioHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
   VehicleRingtoneAudioHandler::VehicleRingtoneAudioHandler()
   {
      _smAudioWrapperReqIf = nullptr;
      _smAudioPlayerReqIf = nullptr;
      _smVehicleRingtoneSmHandlerIf = nullptr;
      _audioSessionHandler = nullptr;
      _smTraceObserverIf = nullptr;
   }

   VehicleRingtoneAudioHandler::VehicleRingtoneAudioHandler(AudioSessionHandler* audioSessionHandler)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: Constructor"));

      _smAudioWrapperReqIf = new VehicleRingtoneAudioHandler::SmAudioWrapperReqIf();

      if(nullptr != _smAudioWrapperReqIf)
      {
         setSCI_AudioWrapperReqIf_OCB(_smAudioWrapperReqIf);
      }

      _smAudioPlayerReqIf = new VehicleRingtoneAudioHandler::SmAudioPlayerReqIf(*this);

      if(nullptr != _smAudioPlayerReqIf)
      {
         setSCI_AudioPlayerReqIf_OCB(_smAudioPlayerReqIf);
      }

      _smVehicleRingtoneSmHandlerIf = new VehicleRingtoneAudioHandler::SmVehicleRingtoneSmHandlerIf(*this);

      if(nullptr != _smVehicleRingtoneSmHandlerIf)
      {
         setSCI_VehicleRingtoneSmHandler_OCB(_smVehicleRingtoneSmHandlerIf);
      }

      _audioSessionHandler = audioSessionHandler;

      _smTraceObserverIf = new VehicleRingtoneAudioHandler::SmTraceObserverIf<VehicleRingtoneAudioSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      //initialize the state machine
      init();

      enter();
   }

   VehicleRingtoneAudioHandler::~VehicleRingtoneAudioHandler()
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: Destructor"));

      if(nullptr != _smAudioWrapperReqIf)
      {
         delete _smAudioWrapperReqIf;
         _smAudioWrapperReqIf = nullptr;
      }

      if(nullptr != _smAudioPlayerReqIf)
      {
         delete _smAudioPlayerReqIf;
         _smAudioPlayerReqIf = nullptr;
      }

      if(nullptr != _smVehicleRingtoneSmHandlerIf)
      {
         delete _smVehicleRingtoneSmHandlerIf;
         _smVehicleRingtoneSmHandlerIf = nullptr;
      }

      _audioSessionHandler = nullptr;

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   void VehicleRingtoneAudioHandler::prepareRingtoneAudio()
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: prepareRingtoneAudio"));

      getSCI_AmClientRequest()->raise_pREPARE_OUTBAND_RINGTONE_CHANNEL();
   }

   void VehicleRingtoneAudioHandler::pauseRingtoneAudio()
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: pauseRingtoneAudio"));

      getSCI_AmClientRequest()->raise_pAUSE_OUTBAND_RINGTONE();
   }

   void VehicleRingtoneAudioHandler::stopRingtoneAudio()
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: stopRingtoneAudio"));

      getSCI_AmClientRequest()->raise_sTOP_OUTBAND_RINGTONE();
   }

   void VehicleRingtoneAudioHandler::playAudioFile(IN std::shared_ptr<AmIfMessage_PlayAudioFileRequest> amIfMessage)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: playAudioFile"));

      getSCI_VehicleRingtoneSmHandler()->set_ringtoneFilePath(amIfMessage->getFilePath());
      getSCI_VehicleRingtoneSmHandler()->set_ringtonePlayCount(amIfMessage->getPlayCount());

      getSCI_AmClientRequest()->raise_pLAY_OUTBAND_RINGTONE();
   }

   void VehicleRingtoneAudioHandler::onSourceActivityResult(const SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: onSourceActivityResult"));

      switch(srcActivity)
      {
         case SRC_ACTIVITY_OFF:
            getSCI_AudioWrapperResponse()->raise_oUTBAND_RINGTONE_SA_OFF();
            break;
         case SRC_ACTIVITY_ON:
            getSCI_AudioWrapperResponse()->raise_oUTBAND_RINGTONE_SA_ON();
            break;
         case SRC_ACTIVITY_PAUSE:
            getSCI_AudioWrapperResponse()->raise_oUTBAND_RINGTONE_SA_PAUSE();
            break;
         default:
            getSCI_AudioWrapperResponse()->raise_oUTBAND_RINGTONE_SA_OFF();
            break;
      }
   }

   void VehicleRingtoneAudioHandler::onAudioManagerResponse(const AudioChannelStatus status,
         const AudioSink& audioSink)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler:onAudioManagerResponse : %u",
            ETG_CENUM(AudioChannelStatus, status)));

      switch(status)
      {
         case CHANNEL_GRANTED:
            getSCI_AudioWrapperResponse()->raise_oUTBAND_RINGTONE_CHANNEL_GRANTED(audioSink);
            break;
         case CHANNEL_DENIED:
            getSCI_AudioWrapperResponse()->raise_oUTBAND_RINGTONE_CHANNEL_DENIED();
            break;
         case CHANNEL_DEALLOCATED: //intentional fall through
         default:
            getSCI_AudioWrapperResponse()->raise_oUTBAND_RINGTONE_CHANNEL_DEALLOCATED();
            break;
      }
   }

   void VehicleRingtoneAudioHandler::onAudioPlayerResponse(const PlayStatus playStatus,
         const AudioPlayerErrorInfo& errorInfo)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler::onAudioPlayerResponse : PlayStatus: %u, errorInfo: %50s",
            ETG_CENUM(PlayStatus, playStatus), errorInfo.c_str()));

      switch(playStatus)
      {
         case PlayStatus::PLAYING:
            getSCI_AudioPlayerResponse()->raise_pLAYING_SUCCESS();
            break;
         case PlayStatus::STOPPED:
            getSCI_AudioPlayerResponse()->raise_sTOP_SUCCESS();
            break;
         case PlayStatus::PLAYBACK_ERROR:
            getSCI_AudioPlayerResponse()->raise_pLAYING_ERROR(errorInfo);  // Fixme: The type in SM defined event - PLAYING_ERROR
            break;
         case PlayStatus::STOP_PLAYBACK_ERROR:
            getSCI_AudioPlayerResponse()->raise_sTOP_FAILURE(errorInfo);   // Fixme: The type in SM defined event - STOP_FAILURE
            break;
         default:
            break;
      }
   }

   void VehicleRingtoneAudioHandler::SmAudioWrapperReqIf::sendAllocateRequest()
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: sendAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().requestAudioRoute(AM_VEHICLERINGTONE);
   }

   void VehicleRingtoneAudioHandler::SmAudioWrapperReqIf::sendDeAllocateRequest()
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: sendDeAllocateRequest"));

      (void)AmMainController::getInstance().getArlWrapper().releaseAudioRoute(AM_VEHICLERINGTONE);
   }

   void VehicleRingtoneAudioHandler::SmAudioWrapperReqIf::sendSrcActivityResult(SrcActivity srcActivity)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler: sendSrcActivityResult"));

      (void)AmMainController::getInstance().getArlWrapper().sendSourceActivityResult(AM_VEHICLERINGTONE, srcActivity);
   }

   AmResultCode VehicleRingtoneAudioHandler::SmAudioPlayerReqIf::setAlsaDeviceName(AudioSink sinkName,
         AudioSource srcName)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler:setAlsaDeviceName: Sink : %50s, Source : %50s",
            sinkName.c_str(), srcName.c_str()));

      return AmMainController::getInstance().getAudioPlayerWrapper().setAlsaDeviceRequest(
            _vehicleRingtoneAudioHandler._audioSessionHandler->getAmSessionId(), sinkName, srcName, RING_TONE);
   }

   AmResultCode VehicleRingtoneAudioHandler::SmAudioPlayerReqIf::sendPlayFileRequest(FilePath filePath,
         PlayCount playCount)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler:sendPlayFileRequest : File=\"%80s\", PlayCount=%d",
            filePath.c_str(), playCount));

      return AmMainController::getInstance().getAudioPlayerWrapper().playAudioFileRequest(
            _vehicleRingtoneAudioHandler. _audioSessionHandler->getAmSessionId(), filePath, playCount);
   }

   AmResultCode VehicleRingtoneAudioHandler::SmAudioPlayerReqIf::stopPlayback()
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler:stopPlayback"));

      return AmMainController::getInstance().getAudioPlayerWrapper().stopPlaybackRequest(
            _vehicleRingtoneAudioHandler._audioSessionHandler->getAmSessionId());
   }

   void VehicleRingtoneAudioHandler::SmVehicleRingtoneSmHandlerIf::updateStatus(RouteState routeState,
         AmResultMessage msg)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler:updateStatus : RouteState : %u, Message : %50s",
            ETG_CENUM(RouteState, routeState), msg.c_str()));

      if(nullptr != _vehicleRingtoneAudioHandler._audioSessionHandler)
      {
         _vehicleRingtoneAudioHandler._audioSessionHandler->updateStatus(AM_VEHICLERINGTONE, routeState, msg);
      }
   }

   void VehicleRingtoneAudioHandler::SmVehicleRingtoneSmHandlerIf::updateSinkName(AudioSink sinkName)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler:updateSinkName : \"%80s\"", sinkName.c_str()));

      _vehicleRingtoneAudioHandler.getSCI_VehicleRingtoneSmHandler()->set_sinkName(sinkName);
   }

   template<typename T>
   void VehicleRingtoneAudioHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler::stateEntered()::%u",
            ETG_CENUM(VehicleRingtoneAudioSmStates, state)));
   }

   template<typename T>
   void VehicleRingtoneAudioHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("VehicleRingtoneAudioHandler::stateExited()::%u",
            ETG_CENUM(VehicleRingtoneAudioSmStates, state)));
   }
}
