/**
 * @file MuteUnmuteHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the MuteUnmuteHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of Mute Unmute
 *
 * @ingroup PmAudioManager
 */

#include "MuteUnmuteHandler.h"
#include "AmMainController.h"
#include "AudioSessionHandler.h"
#include "EcnrWrapper.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/MuteUnmuteHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

#define MIC_MUTE_FAILURE "Mic mute failure"
#define MIC_MUTED "Mic Muted"
#define MIC_DEMUTE_FAILURE "Mic demute failure"
#define MIC_DEMUTED "Mic Demuted"

namespace pmaudiomanager
{
   MuteUnmuteHandler::MuteUnmuteHandler()
   {
      _smEcnrRequestIf = nullptr;
      _smMuteUnmuteSmHandlerIf = nullptr;
      _audioSessionHandler = nullptr;
      _smTraceObserverIf = nullptr;
   }

   MuteUnmuteHandler::MuteUnmuteHandler(AudioSessionHandler* audioSessionHandler)
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler: Constructor"));

      _smEcnrRequestIf = new MuteUnmuteHandler::SmEcnrRequestIf(*this);

      if(nullptr != _smEcnrRequestIf)
      {
         setSCI_EcnrRequest_OCB(_smEcnrRequestIf);
      }

      _smMuteUnmuteSmHandlerIf = new MuteUnmuteHandler::MuteUnmuteSmHandlerIf(*this);

      if(nullptr != _smMuteUnmuteSmHandlerIf)
      {
         setSCI_SmHandler_OCB(_smMuteUnmuteSmHandlerIf);
      }

      _audioSessionHandler = audioSessionHandler;

      _smTraceObserverIf = new MuteUnmuteHandler::SmTraceObserverIf<MuteUnmuteSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      //initialize the state machine
      init();

      enter();
   }

   MuteUnmuteHandler::~MuteUnmuteHandler()
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler: Destructor"));

      if(nullptr != _smEcnrRequestIf)
      {
         delete _smEcnrRequestIf;
         _smEcnrRequestIf = nullptr;
      }

      if(nullptr != _smMuteUnmuteSmHandlerIf)
      {
         delete _smMuteUnmuteSmHandlerIf;
         _smMuteUnmuteSmHandlerIf = nullptr;
      }

      _audioSessionHandler = nullptr;

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   void MuteUnmuteHandler::muteAudio()
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler:muteAudio"));

      getSCI_AmClientRequest()->raise_mUTE_AUDIO();
   }

   void MuteUnmuteHandler::deMuteAudio()
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler:deMuteAudio"));

      getSCI_AmClientRequest()->raise_dEMUTE_AUDIO();
   }

   void MuteUnmuteHandler::onEcnrResponse(const EcnrReponse response, const AmResultMessage& /*msg*/)
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler:onEcnrResponse: %u", ETG_CENUM(EcnrReponse, response)));

      switch(response)
      {
         case ECNR_MUTE_DEMUTE_SUCCESS:
            getSCI_EcnrResponse()->raise_rEQUEST_SUCCESS();
            break;
         case ECNR_MUTE_DEMUTE_FAILURE:
            getSCI_EcnrResponse()->raise_rEQUEST_FAILURE();
            break;
         default:
            break;
      }
   }

   AmResultCode MuteUnmuteHandler::SmEcnrRequestIf::sendEcnrMuteRequest()
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler:sendEcnrMuteRequest"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _MuteUnmuteHandler._audioSessionHandler)
      {
         EcnrSendMuteSwitch mute = 0x01;
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().setSendMuteSwitch(
               _MuteUnmuteHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE, mute))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   AmResultCode MuteUnmuteHandler::SmEcnrRequestIf::sendEcnrDeMuteRequest()
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler:sendEcnrDeMuteRequest"));

      AmResultCode amResult = AM_RESULT_ERR_GENERAL;

      if(nullptr != _MuteUnmuteHandler._audioSessionHandler)
      {
    	 EcnrSendMuteSwitch deMute = 0x00;
         if(ECNR_OK == AmMainController::getInstance().getEcnrWrapper().setSendMuteSwitch(
               _MuteUnmuteHandler._audioSessionHandler->getAmSessionId(), ECNR_APPID_PHONE, deMute))
         {
            amResult = AM_RESULT_OK;
         }
      }

      return amResult;
   }

   void MuteUnmuteHandler::MuteUnmuteSmHandlerIf::updateStatus(AmResultMessage msg)
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler:updateStatus : Message : %50s", msg.c_str()));

      if(nullptr != _MuteUnmuteHandler._audioSessionHandler)
      {
         _MuteUnmuteHandler._audioSessionHandler->updateMuteUnmuteStatus(msg);
      }
   }

   template<typename T>
   void MuteUnmuteHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler::stateEntered()::%u", ETG_CENUM(MuteUnmuteSmStates, state)));
   }

   template<typename T>
   void MuteUnmuteHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("MuteUnmuteHandler::stateExited()::%u", ETG_CENUM(MuteUnmuteSmStates, state)));
   }
}
