/**
 * @file AudioSessionHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the AudioSessionHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of audio session
 *
 * @ingroup PmAudioManager
 */

#include "AudioSessionHandler.h"
#include "VoiceRecAudioHandler.h"
#include "VoiceCallAudioHandler.h"
#include "VehicleRingtoneAudioHandler.h"
#include "WaitingModeToneAudioHandler.h"
#include "MuteUnmuteHandler.h"
#include "AmIfMessageRequest.h"
#include "AmIfMessageResult.h"
#include "AmMainController.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AudioSessionHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

#define MIC_MUTE_FAILURE "Mic mute failure"
#define MIC_MUTED "Mic Muted"
#define MIC_DEMUTE_FAILURE "Mic demute failure"
#define MIC_DEMUTED "Mic Demuted"

namespace pmaudiomanager
{
   AudioSessionHandler::AudioSessionHandler() : _audioChannelToRouteStateMap()
   {
      ETG_TRACE_USR4(("AudioSessionHandler: Constructor"));

      _amSessionId = AM_SESSION_ID_DEFAULT;
      _voiceCallAudioHandler = nullptr;
      _vehicleRingtoneAudioHandler = nullptr;
      _voiceRecAudioHandler = nullptr;
      _waitingModeToneAudioHandler = nullptr;
      _muteUnmuteHandler = nullptr;
   }

   AudioSessionHandler::AudioSessionHandler(IN const AmSessionId amSessionId)
   {
      ETG_TRACE_USR4(("AudioSessionHandler:Created for AmSessionId : %d", amSessionId));

      _amSessionId = amSessionId;
      _voiceCallAudioHandler = new VoiceCallAudioHandler(this);
      _vehicleRingtoneAudioHandler = new VehicleRingtoneAudioHandler(this);
      _voiceRecAudioHandler = new VoiceRecAudioHandler(this);
      _waitingModeToneAudioHandler = new WaitingModeToneAudioHandler(this);
      _muteUnmuteHandler = new MuteUnmuteHandler(this);

      _audioChannelToRouteStateMap.clear();
      setRouteState(AM_PHONEAUDIO, NOT_ALLOCATED);
      setRouteState(AM_VEHICLERINGTONE, NOT_ALLOCATED);
      setRouteState(AM_WAITINGMODE, NOT_ALLOCATED);
      setRouteState(AM_VOICERECOGNITION, NOT_ALLOCATED);
   }

   AudioSessionHandler::~AudioSessionHandler()
   {
      ETG_TRACE_USR4(("AudioSessionHandler: Destructor"));

      _amSessionId = AM_SESSION_ID_DEFAULT;
      _audioChannelToRouteStateMap.clear();

      if(nullptr != _voiceCallAudioHandler)
      {
         delete _voiceCallAudioHandler;
         _voiceCallAudioHandler = nullptr;
      }

      if(nullptr != _vehicleRingtoneAudioHandler)
      {
         delete _vehicleRingtoneAudioHandler;
         _vehicleRingtoneAudioHandler = nullptr;
      }

      if(nullptr != _voiceRecAudioHandler)
      {
         delete _voiceRecAudioHandler;
         _voiceRecAudioHandler = nullptr;
      }

      if(nullptr != _waitingModeToneAudioHandler)
      {
         delete _waitingModeToneAudioHandler;
         _waitingModeToneAudioHandler = nullptr;
      }

      if(nullptr != _muteUnmuteHandler)
      {
         delete _muteUnmuteHandler;
         _muteUnmuteHandler = nullptr;
      }
   }

   AmSessionId AudioSessionHandler::getAmSessionId() const
   {
      return _amSessionId;
   }

   void AudioSessionHandler::prepareAudioRouteRequest(
         IN std::shared_ptr<AmIfMessage_PrepareAudioRouteRequest> amIfMessage)
   {
      ::std::map<AudioChannel, RouteState>::iterator it = _audioChannelToRouteStateMap.find(
            amIfMessage->getAudioChannel());

      if(it != _audioChannelToRouteStateMap.end())
      {
         switch(it->first)
         {
            case AM_PHONEAUDIO:
            {
               if(nullptr != _voiceCallAudioHandler)
               {
                  _voiceCallAudioHandler->prepareVoiceCallAudio();
               }
            }
            break;
            case AM_VEHICLERINGTONE:
            {
               if(nullptr != _vehicleRingtoneAudioHandler)
               {
                  _vehicleRingtoneAudioHandler->prepareRingtoneAudio();
               }
            }
            break;
            case AM_WAITINGMODE:
            {
               if(nullptr != _waitingModeToneAudioHandler)
               {
                  _waitingModeToneAudioHandler->prepareWaitingModeToneAudio();
               }
            }
            break;
            case AM_VOICERECOGNITION:
            {
               if(nullptr != _voiceRecAudioHandler)
               {
                  _voiceRecAudioHandler->prepareVoiceRecAudio();
               }
            }
            break;
            default:
            {
               //error, something wrong
               ETG_TRACE_ERR(("prepareAudioRouteRequest: Invalid Audio Source"));
            }
            break;
         }
      }
   }

   void AudioSessionHandler::playAudioFileRequest(IN std::shared_ptr<AmIfMessage_PlayAudioFileRequest> amIfMessage)
   {
      std::string filePath = amIfMessage->getFilePath();

      if(RING_TONE == amIfMessage->getToneType())
      {
         RouteState routeState = getRouteState(AM_VEHICLERINGTONE);

         if((READY_TO_STREAM == routeState) || (STREAMING_PAUSED == routeState))
         {
            if(nullptr != _vehicleRingtoneAudioHandler)
            {
               _vehicleRingtoneAudioHandler->playAudioFile(amIfMessage);
            }
         }
      }
      else if(WAITINGMODE_TONE == amIfMessage->getToneType())
      {
         RouteState routeState = getRouteState(AM_WAITINGMODE);

         if((READY_TO_STREAM == routeState) || (STREAMING_PAUSED == routeState))
         {
            if(nullptr != _waitingModeToneAudioHandler)
            {
               _waitingModeToneAudioHandler->playAudioFile(filePath);
            }
         }
      }
   }

   void AudioSessionHandler::playHfAudioRequest(IN std::shared_ptr<AmIfMessage_PlayHfAudioRequest> amIfMessage)
   {
      ::std::map<AudioChannel, RouteState>::iterator it;
      for(it = _audioChannelToRouteStateMap.begin(); it != _audioChannelToRouteStateMap.end(); it++)
      {
         // Condition "STREAMING" is also added to handle scenario where SM waits for ecnrStop response on
         // PAUSE_AUDIO request from STREAMING state.
         if((READY_TO_STREAM == it->second) || (STREAMING_PAUSED == it->second) || (STREAMING == it->second))
         {
            if(AM_PHONEAUDIO == it->first)
            {
               if(nullptr != _voiceCallAudioHandler)
               {
                  _voiceCallAudioHandler->playHfAudio(amIfMessage->getSamplingType());
               }
            }
            else if(AM_VOICERECOGNITION == it->first)
            {
               if(nullptr != _voiceRecAudioHandler)
               {
                  _voiceRecAudioHandler->playHfAudio(amIfMessage->getSamplingType());
               }
            }
            else
            {
               //error, something wrong
               ETG_TRACE_ERR(("playHfAudioRequest: Received for Invalid Audio Source"));
            }
            break;
         }
      }
   }

   void AudioSessionHandler::stopAudioRequest(IN std::shared_ptr<AmIfMessage_StopAudioRequest> amIfMessage)
   {
      (void)(amIfMessage);

      ::std::map<AudioChannel, RouteState>::iterator it;
      for(it = _audioChannelToRouteStateMap.begin(); it != _audioChannelToRouteStateMap.end(); it++)
      {
         if((NOT_ALLOCATED != it->second) && (DEALLOCATING != it->second))
         {
            switch(it->first)
            {
               case AM_PHONEAUDIO:
               {
                  if(nullptr != _voiceCallAudioHandler)
                  {
                     _voiceCallAudioHandler->stopVoiceCallAudio();
                  }
               }
               break;
               case AM_VEHICLERINGTONE:
               {
                  if(nullptr != _vehicleRingtoneAudioHandler)
                  {
                     _vehicleRingtoneAudioHandler->stopRingtoneAudio();
                  }
               }
               break;
               case AM_WAITINGMODE:
               {
                  if(nullptr != _waitingModeToneAudioHandler)
                  {
                     _waitingModeToneAudioHandler->stopWaitingModeToneAudio();
                  }
               }
               break;
               case AM_VOICERECOGNITION:
               {
                  if(nullptr != _voiceRecAudioHandler)
                  {
                     _voiceRecAudioHandler->stopVoiceRecAudio();
                  }
               }
               break;
               default:
               {
                  //error, something wrong
                  ETG_TRACE_ERR(("stopAudioRequest: Invalid Audio Source"));
               }
               break;
            }
         }
      }
   }

   void AudioSessionHandler::pauseAudioRequest(IN std::shared_ptr<AmIfMessage_PauseAudioRequest> amIfMessage)
   {
      (void)(amIfMessage);

      ::std::map<AudioChannel, RouteState>::iterator it;
      for(it = _audioChannelToRouteStateMap.begin(); it != _audioChannelToRouteStateMap.end(); it++)
      {
         if((NOT_ALLOCATED != it->second) && (DEALLOCATING != it->second))
         {
            switch(it->first)
            {
               case AM_PHONEAUDIO:
               {
                  if(nullptr != _voiceCallAudioHandler)
                  {
                     _voiceCallAudioHandler->pauseVoiceCallAudio();
                  }
               }
               break;
               case AM_VEHICLERINGTONE:
               {
                  if(nullptr != _vehicleRingtoneAudioHandler)
                  {
                     _vehicleRingtoneAudioHandler->pauseRingtoneAudio();
                  }
               }
               break;
               case AM_WAITINGMODE:
               {
                  if(nullptr != _waitingModeToneAudioHandler)
                  {
                     _waitingModeToneAudioHandler->pauseWaitingModeToneAudio();
                  }
               }
               break;
               case AM_VOICERECOGNITION:
               {
                  if(nullptr != _voiceRecAudioHandler)
                  {
                     _voiceRecAudioHandler->pauseVoiceRecAudio();
                  }
               }
               break;
               default:
               {
                  //error, something wrong
                  ETG_TRACE_ERR(("pauseAudioRequest: Invalid Audio Source"));
               }
               break;
            }
         }
      }
   }

   void AudioSessionHandler::setMicMuteStateRequest(IN std::shared_ptr<AmIfMessage_SetMicMuteStateRequest> amIfMessage)
   {
      if((nullptr != amIfMessage) && (nullptr != _muteUnmuteHandler))
      {
         if(0x01 == amIfMessage->getMuteState())
         {
            _muteUnmuteHandler->muteAudio();
         }
         else
         {
            _muteUnmuteHandler->deMuteAudio();
         }
      }
   }

   void AudioSessionHandler::ecnrInitializeResult(IN std::shared_ptr<AmIfMessage_EcnrInitializeResult> amIfMessage)
   {
      RouteState routeState = getRouteState(AM_PHONEAUDIO);

      // TODO: not a proper way to determine the EcnrInitialize result for a SM
      if(READY_TO_STREAM == routeState)
      {
         if(nullptr != _voiceCallAudioHandler)
         {
            if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
            {
               _voiceCallAudioHandler->onEcnrResponse(ECNR_INIT_SUCCESS, amIfMessage->getAmResult()._amResultMessage);
            }
            else
            {
               _voiceCallAudioHandler->onEcnrResponse(ECNR_INIT_FAILURE, amIfMessage->getAmResult()._amResultMessage);
            }
         }
      }
      else
      {
         if(nullptr != _voiceRecAudioHandler)
         {
            if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
            {
               _voiceRecAudioHandler->onEcnrResponse(ECNR_INIT_SUCCESS, amIfMessage->getAmResult()._amResultMessage);
            }
            else
            {
               _voiceRecAudioHandler->onEcnrResponse(ECNR_INIT_FAILURE, amIfMessage->getAmResult()._amResultMessage);
            }
         }
      }
   }

   void AudioSessionHandler::ecnrDestroyResult(IN std::shared_ptr<AmIfMessage_EcnrDestroyResult> amIfMessage)
   {
      if(nullptr != _voiceCallAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_DESTROY_SUCCESS, amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_DESTROY_FAILURE, amIfMessage->getAmResult()._amResultMessage);
         }
      }

      if(nullptr != _voiceRecAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_DESTROY_SUCCESS, amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_DESTROY_FAILURE, amIfMessage->getAmResult()._amResultMessage);
         }
      }
   }

   void AudioSessionHandler::ecnrSetConfigurationResult(
         IN std::shared_ptr<AmIfMessage_EcnrSetConfigurationResult> amIfMessage)
   {
      if(nullptr != _voiceCallAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_SET_CONFIGURATION_SUCCESS,
                  amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_SET_CONFIGURATION_FAILURE,
                  amIfMessage->getAmResult()._amResultMessage);
         }
      }

      if(nullptr != _voiceRecAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_SET_CONFIGURATION_SUCCESS,
                  amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_SET_CONFIGURATION_FAILURE,
                  amIfMessage->getAmResult()._amResultMessage);
         }
      }
   }

   void AudioSessionHandler::ecnrSetSendMuteSwitchResult(
         IN std::shared_ptr<AmIfMessage_EcnrSetSendMuteSwitchResult> amIfMessage)
   {
      if(nullptr != _muteUnmuteHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _muteUnmuteHandler->onEcnrResponse(ECNR_MUTE_DEMUTE_SUCCESS,
                  amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _muteUnmuteHandler->onEcnrResponse(ECNR_MUTE_DEMUTE_FAILURE,
                  amIfMessage->getAmResult()._amResultMessage);
         }
      }
   }

   void AudioSessionHandler::ecnrStartAudioResult(IN std::shared_ptr<AmIfMessage_EcnrStartAudioResult> amIfMessage)
   {
      if(nullptr != _voiceCallAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_START_AUDIO_SUCCESS,
                  amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_START_AUDIO_FAILURE,
                  amIfMessage->getAmResult()._amResultMessage);
         }
      }

      if(nullptr != _voiceRecAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_START_AUDIO_SUCCESS,
                  amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_START_AUDIO_FAILURE,
                  amIfMessage->getAmResult()._amResultMessage);
         }
      }
   }

   void AudioSessionHandler::ecnrStopAudioResult(IN std::shared_ptr<AmIfMessage_EcnrStopAudioResult> amIfMessage)
   {
      if(nullptr != _voiceCallAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_STOP_AUDIO_SUCCESS,
                  amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceCallAudioHandler->onEcnrResponse(ECNR_STOP_AUDIO_FAILURE,
                  amIfMessage->getAmResult()._amResultMessage);
         }
      }

      if(nullptr != _voiceRecAudioHandler)
      {
         if(AM_RESULT_OK == amIfMessage->getAmResult()._amResultCode)
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_STOP_AUDIO_SUCCESS,
                  amIfMessage->getAmResult()._amResultMessage);
         }
         else
         {
            _voiceRecAudioHandler->onEcnrResponse(ECNR_STOP_AUDIO_FAILURE,
                  amIfMessage->getAmResult()._amResultMessage);
         }
      }
   }

   void AudioSessionHandler::ecnrGetVersionResult(IN std::shared_ptr<AmIfMessage_EcnrGetVersionResult> amIfMessage)
   {
      (void)(amIfMessage);

      //TODO: implement if required
   }

   void AudioSessionHandler::requestAudioRouteResult(
         IN std::shared_ptr<AmIfMessage_RequestAudioRouteResult> amIfMessage)
   {
      AudioChannelStatus channelStatus = CHANNEL_GRANTED;

      if(AM_RESULT_ERR_TIMEOUT == amIfMessage->getAmResult()._amResultCode)
      {
         channelStatus = CHANNEL_DENIED;
      }

      switch(amIfMessage->getAudioChannel())
      {
         case AM_VEHICLERINGTONE:
         {
            if(nullptr != _vehicleRingtoneAudioHandler)
            {
               AudioDeviceList audioSink = amIfMessage->getAudioSink();

               if((CHANNEL_GRANTED == channelStatus) && (0 < audioSink.size()))
               {
                  _vehicleRingtoneAudioHandler->onAudioManagerResponse(channelStatus, audioSink[0]);
               }
               else
               {
                  _vehicleRingtoneAudioHandler->onAudioManagerResponse(channelStatus);
               }
            }
         }
         break;
         case AM_PHONEAUDIO:
         case AM_ALERTTONE:
         {
            if(nullptr != _voiceCallAudioHandler)
            {
               _voiceCallAudioHandler->onAudioManagerResponse(channelStatus);
            }
         }
         break;
         case AM_WAITINGMODE:
         {
            if(nullptr != _waitingModeToneAudioHandler)
            {
               AudioDeviceList audioSink = amIfMessage->getAudioSink();

               if((CHANNEL_GRANTED == channelStatus) && (0 < audioSink.size()))
               {
                  _waitingModeToneAudioHandler->onAudioManagerResponse(channelStatus, audioSink[0]);
               }
               else
               {
                  _waitingModeToneAudioHandler->onAudioManagerResponse(channelStatus);
               }
            }
         }
         break;
         case AM_VOICERECOGNITION:
         {
            if(nullptr != _voiceRecAudioHandler)
            {
               _voiceRecAudioHandler->onAudioManagerResponse(channelStatus);
            }
         }
         break;
         default:
         {
            //error, something wrong
            ETG_TRACE_ERR(("requestAudioRouteResult: Invalid Audio Source"));
         }
         break;
      }
   }

   void AudioSessionHandler::releaseAudioRouteResult(
         IN std::shared_ptr<AmIfMessage_ReleaseAudioRouteResult> amIfMessage)
   {
      switch(amIfMessage->getAudioChannel())
      {
         case AM_VEHICLERINGTONE:
         {
            if(nullptr != _vehicleRingtoneAudioHandler)
            {
               _vehicleRingtoneAudioHandler->onAudioManagerResponse(CHANNEL_DEALLOCATED);
            }
         }
         break;
         case AM_PHONEAUDIO:
         case AM_ALERTTONE:
         {
            if(nullptr != _voiceCallAudioHandler)
            {
               _voiceCallAudioHandler->onAudioManagerResponse(CHANNEL_DEALLOCATED);
            }
         }
         break;
         case AM_WAITINGMODE:
         {
            if(nullptr != _waitingModeToneAudioHandler)
            {
               _waitingModeToneAudioHandler->onAudioManagerResponse(CHANNEL_DEALLOCATED);
            }
         }
         break;
         case AM_VOICERECOGNITION:
         {
            if(nullptr != _voiceRecAudioHandler)
            {
               _voiceRecAudioHandler->onAudioManagerResponse(CHANNEL_DEALLOCATED);
            }
         }
         break;
         default:
         {
            //error, something wrong
            ETG_TRACE_ERR(("releaseAudioRouteResult: Invalid Audio Source"));
         }
         break;
      }
   }

   void AudioSessionHandler::sourceActivityResult(IN std::shared_ptr<AmIfMessage_SourceActivityResult> amIfMessage)
   {
      switch(amIfMessage->getAudioChannel())
      {
         case AM_VEHICLERINGTONE:
         {
            if(nullptr != _vehicleRingtoneAudioHandler)
            {
               _vehicleRingtoneAudioHandler->onSourceActivityResult(amIfMessage->getSrcActivity());
            }
         }
         break;
         case AM_PHONEAUDIO:
         case AM_ALERTTONE:
         {
            if(nullptr != _voiceCallAudioHandler)
            {
               _voiceCallAudioHandler->onSourceActivityResult(amIfMessage->getSrcActivity());
            }
         }
         break;
         case AM_WAITINGMODE:
         {
            if(nullptr != _waitingModeToneAudioHandler)
            {
               _waitingModeToneAudioHandler->onSourceActivityResult(amIfMessage->getSrcActivity());
            }
         }
         break;
         case AM_VOICERECOGNITION:
         {
            if(nullptr != _voiceRecAudioHandler)
            {
               _voiceRecAudioHandler->onSourceActivityResult(amIfMessage->getSrcActivity());
            }
         }
         break;
         default:
         {
            //error, something wrong
            ETG_TRACE_ERR(("sourceActivityResult: Invalid Audio Source"));
         }
         break;
      }
   }

   void AudioSessionHandler::playbackStatus(IN std::shared_ptr<AmIfMessage_PlaybackStatus> amIfMessage)
   {
      if (RING_TONE == amIfMessage->getToneType())
      {
         if(_vehicleRingtoneAudioHandler)
         {
            _vehicleRingtoneAudioHandler->onAudioPlayerResponse(amIfMessage->getPlayStatus());
         }
      }
      else if (WAITINGMODE_TONE == amIfMessage->getToneType())
      {
         if(_waitingModeToneAudioHandler)
         {
            _waitingModeToneAudioHandler->onAudioPlayerResponse(amIfMessage->getPlayStatus());
         }
      }
   }

   void AudioSessionHandler::updateStatus(const AudioChannel channelId, const RouteState routeState,
         const AmResultMessage& msg)
   {
      RouteState prevRouteState = getRouteState(channelId);

      AmEventDetails eventInfo(AUDIO_STATE_CHANGE, msg);
      StreamingState streamingState = StreamingStateEnumType::STOPPED;
      bool updateNotificationEvent = true;

      switch(routeState)
      {
         case NOT_ALLOCATED:
         {
            if((0 == msg.compare("Source Switch")) && (false == isOtherAudioChannelActive(channelId)))
            {
               eventInfo._message = "streaming stopped due to external trigger";
            }
         }
         break;
         case PREPARING:
            //No need to inform this intermediate state
            updateNotificationEvent = false;
            break;
         case READY_TO_STREAM:
            streamingState = StreamingStateEnumType::NOT_PLAYING;
            break;
         case STREAMING:
            streamingState = StreamingStateEnumType::PLAYING;
            break;
         case STREAMING_PAUSED:
            streamingState = StreamingStateEnumType::PAUSED;
            break;
         case DEALLOCATING:
            //No need to inform this intermediate state
            updateNotificationEvent = false;
            break;
         case FAILURE:
            eventInfo._event = REQUEST_FAILURE;
            break;
         default:
            //something wrong, no need to update
            updateNotificationEvent = false;
            break;
      }

      if ((PREPARING == prevRouteState) && (FAILURE == routeState))
      {
         //Channel allocation failed, update the route state as NOT_ALLOCATED
         setRouteState(channelId, NOT_ALLOCATED);
      }
      else if(FAILURE != routeState)
      {
         //Update the route state of the received channel in STL map
         setRouteState(channelId, routeState);
      }

      //The intermediate routing states are for the handling in AudioManager.
      //Hence it is not required to update these states to PmCore.
      if(updateNotificationEvent)
      {
         AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().amNotificationEvent(getAmSessionId(),
               channelId, streamingState, eventInfo);
      }
   }

   void AudioSessionHandler::updateMuteUnmuteStatus(IN const AmResultMessage& msg)
   {
      ETG_TRACE_USR4(("AudioSessionHandler: updateMuteUnmuteStatus"));

      AudioChannel channelId = getActiveChannel();
      bool updateAmClients = true;

      if(AM_UNKNOWN != channelId)
      {
         StreamingState streamingState(StreamingStateEnumType::PLAYING);
         AmEventDetails eventInfo(AUDIO_STATE_CHANGE, msg);

         if(MIC_MUTED == msg)
         {
            streamingState = StreamingStateEnumType::MUTED;
         }
         else if(MIC_DEMUTED == msg)
         {
            streamingState = StreamingStateEnumType::DEMUTED;
         }
         else if(MIC_MUTE_FAILURE == msg)
         {
            eventInfo._event = REQUEST_FAILURE;
            eventInfo._message = "SetMicMute-" + msg;
         }
         else if (MIC_DEMUTE_FAILURE == msg)
         {
            eventInfo._event = REQUEST_FAILURE;
            eventInfo._message = "SetMicUnmute-" + msg;
         }
         else
         {
            ETG_TRACE_ERR(("AudioSessionHandler: updateMuteUnmuteStatus: Unknown status MSG from SM"));
            updateAmClients = false;
         }
         if(updateAmClients)
         {
            AmMainController::getInstance().getPmAudioManagerCallbackIfWrapper().amNotificationEvent(getAmSessionId(),
                  channelId, streamingState, eventInfo);
         }
      }
      else
      {
         ETG_TRACE_ERR(("AudioSessionHandler: updateMuteUnmuteStatus: Unknown channel skip Mute/Unmute status"));
      }
   }

   void AudioSessionHandler::setRouteState(const AudioChannel channelId, const RouteState routeState)
   {
      ::std::map<AudioChannel, RouteState>::iterator it = _audioChannelToRouteStateMap.find(channelId);
      if(it != _audioChannelToRouteStateMap.end())
      {
         //Update the route state for the corresponding channel
         it->second = routeState;
      }
      else
      {
         //AudioChannel is inserted into the STL map.
         _audioChannelToRouteStateMap.insert(::std::pair<AudioChannel, RouteState>(channelId, routeState));
      }
   }

   RouteState AudioSessionHandler::getRouteState(const AudioChannel channelId)
   {
      RouteState routeState = NOT_ALLOCATED;

      ::std::map<AudioChannel, RouteState>::iterator it = _audioChannelToRouteStateMap.find(channelId);
      if(it != _audioChannelToRouteStateMap.end())
      {
         routeState = it->second;
      }

      return routeState;
   }

   bool AudioSessionHandler::isAnyAudioChannelActive()
   {
      bool isActive = false;

      ::std::map<AudioChannel, RouteState>::iterator it;
      for(it = _audioChannelToRouteStateMap.begin(); it != _audioChannelToRouteStateMap.end(); it++)
      {
         if(NOT_ALLOCATED != it->second)
         {
            isActive = true;
            break;
         }
      }

      return isActive;
   }

   bool AudioSessionHandler::isOtherAudioChannelActive(const AudioChannel channel)
   {
      bool isActive = false;

      ::std::map<AudioChannel, RouteState>::iterator it;
      for(it = _audioChannelToRouteStateMap.begin(); it != _audioChannelToRouteStateMap.end(); it++)
      {
         if((NOT_ALLOCATED != it->second) && (channel != it->first))
         {
            isActive = true;
            break;
         }
      }

      return isActive;
   }

   AudioChannel AudioSessionHandler::getActiveChannel()
   {
      AudioChannel channel(AM_UNKNOWN);
      ::std::map<AudioChannel, RouteState>::iterator it;
      for(it = _audioChannelToRouteStateMap.begin(); it != _audioChannelToRouteStateMap.end(); it++)
      {
         if(NOT_ALLOCATED != it->second)
         {
            channel = it->first;
            break;
         }
      }
      return channel;
   }
}
