/**
 * @file VoiceRecAudioHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VoiceRecAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice recognition audio
 *
 * @ingroup PmAudioManager
 */

#ifndef VoiceRecAudioHandler_h
#define VoiceRecAudioHandler_h

#include "AmIfMessageResult.h"
#include "../src-gen/VoiceRecAudioSm.h"

namespace pmaudiomanager
{
   //class forward declaration
   class AudioSessionHandler;

   class VoiceRecAudioHandler final: public VoiceRecAudioSm
   {
   public:
      /**
       * Parameterized Constructor of VoiceRecAudioHandler class
       *
       * @param[in] audioSessionHandler - pointer of AudioSessionHandler
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VoiceRecAudioHandler(IN AudioSessionHandler* audioSessionHandler);

      /**
       * Destructor of VoiceRecAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~VoiceRecAudioHandler();

      /**
       * This method is used to notify the SM to prepare VoiceRecognition AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void prepareVoiceRecAudio();

      /**
       * This method is used to notify the SM to pause VoiceRecognition AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void pauseVoiceRecAudio();

      /**
       * This method is used to notify the SM to stop VoiceRecognition AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopVoiceRecAudio();

      /**
       * This method is used to notify the SM to play audio
       *
       * @param[in] samplingType - sampling type
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playHfAudio(const SamplingType samplingType);

      /**
       * This method is used to notify the audio source activity changes to SM
       *
       * @param[in] srcActivity - source activity
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSourceActivityResult(const SrcActivity srcActivity);

      /**
       * This method is used to notify the audio channel allocation/deallocation status to SM
       *
       * @param[in] status - audio channel status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerResponse(const AudioChannelStatus status);

      /**
       * This method is used to notify the Ecnr response to SM
       *
       * @param[in] response - ecnr response
       * @param[in] msg - result message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onEcnrResponse(const EcnrReponse response, const AmResultMessage& msg);

   private:
      /**
       * Constructor of VoiceRecAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VoiceRecAudioHandler();

      /**
       * SmAudioWrapperReqIf class inherits the functions of SCI_AudioWrapperReqIf_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioWrapperReqIf final: public SCI_AudioWrapperReqIf_OCB
      {
      public:
         SmAudioWrapperReqIf(){}

         ~SmAudioWrapperReqIf(){}

         virtual void sendAllocateRequest() override;

         virtual void sendDeAllocateRequest() override;

         virtual void sendSrcActivityResult(SrcActivity srcActivity) override;
      };

      /**
       * SmEcnrRequestIf class inherits the functions of SCI_EcnrRequest_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmEcnrRequestIf final: public SCI_EcnrRequest_OCB
      {
      public:
         SmEcnrRequestIf(VoiceRecAudioHandler& voiceRecAudioHandler):
            _voiceRecAudioHandler(voiceRecAudioHandler){}

         ~SmEcnrRequestIf(){}

         virtual AmResultCode sendEcnrInit(SamplingType samplingType) override;

         virtual AmResultCode sendEcnrSetConfiguration(SamplingType samplingType) override;

         virtual AmResultCode sendEcnrStart() override;

         virtual AmResultCode sendEcnrStop() override;

         virtual AmResultCode sendEcnrDestroy() override;

      private:
         VoiceRecAudioHandler& _voiceRecAudioHandler;
      };

      /**
       * SmVoiceRecAudioSmHandlerIf class inherits the functions of SCI_VoiceRecAudioSmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmVoiceRecAudioSmHandlerIf final: public SCI_VoiceRecAudioSmHandler_OCB
      {
      public:
         SmVoiceRecAudioSmHandlerIf(VoiceRecAudioHandler& voiceRecAudioHandler):
            _voiceRecAudioHandler(voiceRecAudioHandler){}

         ~SmVoiceRecAudioSmHandlerIf(){}

         virtual void updateSamplingType(SamplingType samplingType) override;

         virtual void updateStatus(RouteState routeState, AmResultMessage msg) override;

      private:
         VoiceRecAudioHandler& _voiceRecAudioHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final : public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(VoiceRecAudioHandler& voiceRecAudioHandler):
            _voiceRecAudioHandler(voiceRecAudioHandler){}

         virtual ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         VoiceRecAudioHandler& _voiceRecAudioHandler;
      };

      AudioSessionHandler*         _audioSessionHandler;        /**< pointer to AudioSessionHandler */
      SmAudioWrapperReqIf*         _smAudioWrapperReqIf;        /**< pointer to SmAudioWrapperReqIf */
      SmEcnrRequestIf*             _smEcnrRequestIf;            /**< pointer to SmEcnrRequestIf */
      SmVoiceRecAudioSmHandlerIf*  _smVoiceRecAudioSmHandlerIf; /**< pointer to SmVoiceRecAudioSmHandlerIf */
      SmTraceObserverIf<VoiceRecAudioSmStates>* _smTraceObserverIf; /**< pointer for TraceObserver interface */
   };
}

#endif //VoiceCallAudioHandler_h
