/**
 * @file VoiceCallAudioHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VoiceCallAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call audio
 *
 * @ingroup PmAudioManager
 */

#ifndef VoiceCallAudioHandler_h
#define VoiceCallAudioHandler_h

#include "AmIfMessageResult.h"
#include "../src-gen/VoiceCallAudioSm.h"

namespace pmaudiomanager
{
   //class forward declaration
   class AudioSessionHandler;

   class VoiceCallAudioHandler final: public VoiceCallAudioSm
   {
   public:
      /**
       * Parameterized Constructor of VoiceCallAudioHandler class
       *
       * @param[in] audioSessionHandler - pointer of AudioSessionHandler
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VoiceCallAudioHandler(IN AudioSessionHandler* audioSessionHandler);

      /**
       * Destructor of VoiceCallAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~VoiceCallAudioHandler();

      /**
       * This method is used to notify the SM to prepare VoiceCall AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void prepareVoiceCallAudio();

      /**
       * This method is used to notify the SM to pause VoiceCall AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void pauseVoiceCallAudio();

      /**
       * This method is used to notify the SM to stop VoiceCall AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopVoiceCallAudio();

      /**
       * This method is used to notify the SM to play audio
       *
       * @param[in] samplingType - sampling type
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playHfAudio(const SamplingType samplingType);

      /**
       * This method is used to notify the audio source activity changes to SM
       *
       * @param[in] srcActivity - source activity
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSourceActivityResult(const SrcActivity srcActivity);

      /**
       * This method is used to notify the audio channel allocation/deallocation status to SM
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerResponse(const AudioChannelStatus status);

      /**
       * This method is used to notify the Ecnr response to SM
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onEcnrResponse(const EcnrReponse response, const AmResultMessage& msg);

   private:
      /**
       * Constructor of VoiceCallAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VoiceCallAudioHandler();

      /**
       * SmAudioWrapperReqIf class inherits the functions of SCI_AudioWrapperReqIf_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioWrapperReqIf final: public SCI_AudioWrapperReqIf_OCB
      {
      public:
         SmAudioWrapperReqIf(){}

         ~SmAudioWrapperReqIf(){}

         virtual void sendAllocateRequest() override;

         virtual void sendDeAllocateRequest() override;

         virtual void sendSrcActivityResult(SrcActivity srcActivity) override;
      };

      /**
       * SmEcnrRequestIf class inherits the functions of SCI_EcnrRequest_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmEcnrRequestIf final: public SCI_EcnrRequest_OCB
      {
      public:
         SmEcnrRequestIf(VoiceCallAudioHandler& voiceCallAudioHandler):
            _voiceCallAudioHandler(voiceCallAudioHandler){}

         ~SmEcnrRequestIf(){}

         virtual AmResultCode sendEcnrInit(SamplingType samplingType) override;

         virtual AmResultCode sendEcnrSetConfiguration(SamplingType samplingType) override;

         virtual AmResultCode sendEcnrStart() override;

         virtual AmResultCode sendEcnrStop() override;

         virtual AmResultCode sendEcnrDestroy() override;

      private:
         VoiceCallAudioHandler& _voiceCallAudioHandler;
      };

      /**
       * SmVoiceCallAudioSmHandlerIf class inherits the functions of SCI_VoiceCallAudioSmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmVoiceCallAudioSmHandlerIf final: public SCI_VoiceCallAudioSmHandler_OCB
      {
      public:
         SmVoiceCallAudioSmHandlerIf(VoiceCallAudioHandler& voiceCallAudioHandler):
            _voiceCallAudioHandler(voiceCallAudioHandler){}

         ~SmVoiceCallAudioSmHandlerIf(){}

         virtual void updateSamplingType(SamplingType samplingType) override;

         virtual void updateStatus(RouteState routeState, AmResultMessage msg) override;

      private:
         VoiceCallAudioHandler& _voiceCallAudioHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final : public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(VoiceCallAudioHandler& voiceCallAudioHandler):
            _voiceCallAudioHandler(voiceCallAudioHandler){}

         virtual ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         VoiceCallAudioHandler& _voiceCallAudioHandler;
      };


      AudioSessionHandler*         _audioSessionHandler;         /**< pointer to AudioSessionHandler */
      SmAudioWrapperReqIf*         _smAudioWrapperReqIf;         /**< pointer to SmAudioWrapperReqIf */
      SmEcnrRequestIf*             _smEcnrRequestIf;             /**< pointer to SmEcnrRequestIf */
      SmVoiceCallAudioSmHandlerIf* _smVoiceCallAudioSmHandlerIf; /**< pointer to SmVoiceCallAudioSmHandlerIf */
      SmTraceObserverIf<VoiceCallAudioSmStates>* _smTraceObserverIf; /**< pointer for TraceObserver interface */
   };
}

#endif //VoiceCallAudioHandler_h
