/**
 * @file VehicleRingtoneAudioHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VehicleRingtoneAudioHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of vehicle ringtone audio
 *
 * @ingroup PmAudioManager
 */

#ifndef VehicleRingtoneAudioHandler_h
#define VehicleRingtoneAudioHandler_h

#include "AmIfMessageRequest.h"
#include "../src-gen/VehicleRingtoneAudioSm.h"

namespace pmaudiomanager
{
   //class forward declaration
   class AudioSessionHandler;

   class VehicleRingtoneAudioHandler final: public VehicleRingtoneAudioSm
   {
   public:
      /**
       * Parameterized Constructor of VehicleRingtoneAudioHandler class
       *
       * @param[in] audioSessionHandler - pointer of AudioSessionHandler
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VehicleRingtoneAudioHandler(AudioSessionHandler* audioSessionHandler);

      /**
       * Destructor of VehicleRingtoneAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~VehicleRingtoneAudioHandler();

      /**
       * This method is used to notify the SM to prepare VehicleRingtone AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void prepareRingtoneAudio();

      /**
       * This method is used to notify the SM to pause VehicleRingtone AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void pauseRingtoneAudio();

      /**
       * This method is used to notify the SM to stop VehicleRingtone AudioChannel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopRingtoneAudio();

      /**
       * This method is used to notify the SM to play given audio file
       *
       * @param[in] amIfMessage - shared ptr for PlayAudioFileRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playAudioFile(IN std::shared_ptr<AmIfMessage_PlayAudioFileRequest> amIfMessage);

      /**
       * This method is used to notify the audio source activity changes to SM
       *
       * @param[in] srcActivity - source activity
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onSourceActivityResult(const SrcActivity srcActivity);

      /**
       * This method is used to notify the audio channel allocation/deallocation status to SM
       *
       * @param[in] status - audio channel allocation/deallocation status
       * @param[in] audioSink - audio sink device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerResponse(const AudioChannelStatus status, const AudioSink& audioSink = "");

      /**
       * This method is used to notify the AudioPlayer response to SM
       *
       * @param[in] playStatus - playback status
       * @param[in] errorInfo  - error info (default - empty string)
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioPlayerResponse(const PlayStatus playStatus, const AudioPlayerErrorInfo& errorInfo = "");

   private:
      /**
       * Constructor of VehicleRingtoneAudioHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VehicleRingtoneAudioHandler();

      /**
       * SmAudioWrapperReqIf class inherits the functions of SCI_AudioWrapperReqIf_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioWrapperReqIf final: public SCI_AudioWrapperReqIf_OCB
      {
      public:
         SmAudioWrapperReqIf(){}

         ~SmAudioWrapperReqIf(){}

         virtual void sendAllocateRequest() override;

         virtual void sendDeAllocateRequest() override;

         virtual void sendSrcActivityResult(SrcActivity srcActivity) override;
      };

      /**
       * SmAudioPlayerReqIf class inherits the functions of SCI_AudioPlayerReqIf_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioPlayerReqIf final: public SCI_AudioPlayerReqIf_OCB
      {
      public:
         SmAudioPlayerReqIf(VehicleRingtoneAudioHandler& vehicleRingtoneAudioHandler) :
            _vehicleRingtoneAudioHandler(vehicleRingtoneAudioHandler) {}

         ~SmAudioPlayerReqIf(){}

         virtual AmResultCode setAlsaDeviceName(AudioSink sinkName, AudioSource srcName) override;

         virtual AmResultCode sendPlayFileRequest(FilePath filePath, PlayCount playCount) override;

         virtual AmResultCode stopPlayback() override;

      private:
         VehicleRingtoneAudioHandler& _vehicleRingtoneAudioHandler;
      };

      /**
       * SmVehicleRingtoneSmHandlerIf class inherits the functions of SCI_VehicleRingtoneSmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmVehicleRingtoneSmHandlerIf final: public SCI_VehicleRingtoneSmHandler_OCB
      {
      public:
         SmVehicleRingtoneSmHandlerIf(VehicleRingtoneAudioHandler& vehicleRingtoneAudioHandler):
            _vehicleRingtoneAudioHandler(vehicleRingtoneAudioHandler){}

         ~SmVehicleRingtoneSmHandlerIf(){}

         virtual void updateStatus(RouteState routeState, AmResultMessage msg) override;

         virtual void updateSinkName(AudioSink sinkName) override;

      private:
         VehicleRingtoneAudioHandler& _vehicleRingtoneAudioHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final: public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(VehicleRingtoneAudioHandler& vehicleRingtoneAudioHandler):
           _vehicleRingtoneAudioHandler(vehicleRingtoneAudioHandler){}

         virtual ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         VehicleRingtoneAudioHandler& _vehicleRingtoneAudioHandler;
      };

      AudioSessionHandler*          _audioSessionHandler;          /**< pointer to AudioSessionHandler */
      SmAudioWrapperReqIf*          _smAudioWrapperReqIf;          /**< pointer to SmAudioWrapperReqIf */
      SmAudioPlayerReqIf*           _smAudioPlayerReqIf;           /**< pointer to SmAudioPlayerReqIf */
      SmVehicleRingtoneSmHandlerIf* _smVehicleRingtoneSmHandlerIf; /**< pointer to SmVehicleRingtoneSmHandlerIf */
      SmTraceObserverIf<VehicleRingtoneAudioSmStates>* _smTraceObserverIf; /**< pointer for TraceObserver interface */
   };
}

#endif //VehicleRingtoneAudioHandler_h
