/**
 * @file AudioSessionHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the AudioSessionHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of audio session
 *
 * @ingroup PmAudioManager
 */

#ifndef AudioSessionHandler_h
#define AudioSessionHandler_h

#include "PmAudioManagerTypesInternal.h"

namespace pmaudiomanager
{
   //class forward declarations
   class VoiceCallAudioHandler;
   class VehicleRingtoneAudioHandler;
   class VoiceRecAudioHandler;
   class WaitingModeToneAudioHandler;
   class MuteUnmuteHandler;

   //Method Requests
   class AmIfMessage_PrepareAudioRouteRequest;
   class AmIfMessage_PlayAudioFileRequest;
   class AmIfMessage_PlayHfAudioRequest;
   class AmIfMessage_StopAudioRequest;
   class AmIfMessage_PauseAudioRequest;
   class AmIfMessage_SetMicMuteStateRequest;

   //Method Results
   class AmIfMessage_EcnrInitializeResult;
   class AmIfMessage_EcnrDestroyResult;
   class AmIfMessage_EcnrSetConfigurationResult;
   class AmIfMessage_EcnrSetSendMuteSwitchResult;
   class AmIfMessage_EcnrStartAudioResult;
   class AmIfMessage_EcnrStopAudioResult;
   class AmIfMessage_EcnrGetVersionResult;
   class AmIfMessage_RequestAudioRouteResult;
   class AmIfMessage_ReleaseAudioRouteResult;
   class AmIfMessage_SourceActivityResult;
   class AmIfMessage_PlaybackStatus;

   class AudioSessionHandler
   {
   public:
      /**
       * Parameterized Constructor of AudioSessionHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      AudioSessionHandler(IN const AmSessionId amSessionId);

      /**
       * Copy constructor of AudioSessionHandler class
       *
       * @param[in] ref - Reference to AudioSessionHandler
       * @param[out]
       * @param[in,out]
       *
       * @return
       *
       */
      AudioSessionHandler(const AudioSessionHandler& ref);

      /**
       * operator= overloaded function
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return Returns the object of the AudioSessionHandler
       *
       */
      AudioSessionHandler& operator=(const AudioSessionHandler& ref);

      /**
       * Destructor of AudioSessionHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~AudioSessionHandler();

      /**
       * This method is used to get the AmSessionId of the SM instnace
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return AmSessionId
       */
      AmSessionId getAmSessionId() const;

      /**
       * This method is used to handle PrepareAudioRoute Request
       *
       * @param[in] amIfMessage - shared pointer to PrepareAudioRouteRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void prepareAudioRouteRequest(IN std::shared_ptr<AmIfMessage_PrepareAudioRouteRequest> amIfMessage);

      /**
       * This method is used to handle PlayAudioFile Request
       *
       * @param[in] amIfMessage - shared pointer to PlayAudioFileRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playAudioFileRequest(IN std::shared_ptr<AmIfMessage_PlayAudioFileRequest> amIfMessage);

      /**
       * This method is used to handle PlayHfAudio Request
       *
       * @param[in] amIfMessage - shared pointer to PlayHfAudioRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playHfAudioRequest(IN std::shared_ptr<AmIfMessage_PlayHfAudioRequest> amIfMessage);

      /**
       * This method is used to handle StopAudio Request
       *
       * @param[in] amIfMessage - shared pointer to StopAudioRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopAudioRequest(IN std::shared_ptr<AmIfMessage_StopAudioRequest> amIfMessage);

      /**
       * This method is used to handle PauseAudio Request
       *
       * @param[in] amIfMessage - shared pointer to PauseAudioRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void pauseAudioRequest(IN std::shared_ptr<AmIfMessage_PauseAudioRequest> amIfMessage);

      /**
       * This method is used to handle set Mic Mute State Request
       *
       * @param[in] amIfMessage - shared pointer to set Mic Mute State Request
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setMicMuteStateRequest(IN std::shared_ptr<AmIfMessage_SetMicMuteStateRequest> amIfMessage);

      /**
       * This method is used to handle EcnrInitialize Result
       *
       * @param[in] amIfMessage - shared pointer to EcnrInitializeResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ecnrInitializeResult(IN std::shared_ptr<AmIfMessage_EcnrInitializeResult> amIfMessage);

      /**
       * This method is used to handle EcnrDestroy Result
       *
       * @param[in] amIfMessage - shared pointer to EcnrDestroyResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ecnrDestroyResult(IN std::shared_ptr<AmIfMessage_EcnrDestroyResult> amIfMessage);

      /**
       * This method is used to handle EcnrSetConfiguration Result
       *
       * @param[in] amIfMessage - shared pointer to EcnrSetConfigurationResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ecnrSetConfigurationResult(IN std::shared_ptr<AmIfMessage_EcnrSetConfigurationResult> amIfMessage);

      /**
       * This method is used to handle EcnrSetSendMuteSwitch Result
       *
       * @param[in] amIfMessage - shared pointer to EcnrSetSendMuteSwitchResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ecnrSetSendMuteSwitchResult(IN std::shared_ptr<AmIfMessage_EcnrSetSendMuteSwitchResult> amIfMessage);

      /**
       * This method is used to handle EcnrStartAudio Result
       *
       * @param[in] amIfMessage - shared pointer to EcnrStartAudioResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ecnrStartAudioResult(IN std::shared_ptr<AmIfMessage_EcnrStartAudioResult> amIfMessage);

      /**
       * This method is used to handle EcnrStopAudio Result
       *
       * @param[in] amIfMessage - shared pointer to EcnrStopAudioResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ecnrStopAudioResult(IN std::shared_ptr<AmIfMessage_EcnrStopAudioResult> amIfMessage);

      /**
       * This method is used to handle EcnrGetVersion Result
       *
       * @param[in] amIfMessage - shared pointer to EcnrGetVersionResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void ecnrGetVersionResult(IN std::shared_ptr<AmIfMessage_EcnrGetVersionResult> amIfMessage);

      /**
       * This method is used to handle RequestAudioRoute Result
       *
       * @param[in] amIfMessage - shared pointer to RequestAudioRouteResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void requestAudioRouteResult(IN std::shared_ptr<AmIfMessage_RequestAudioRouteResult> amIfMessage);

      /**
       * This method is used to handle ReleaseAudioRoute Result
       *
       * @param[in] amIfMessage - shared pointer to ReleaseAudioRouteResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void releaseAudioRouteResult(IN std::shared_ptr<AmIfMessage_ReleaseAudioRouteResult> amIfMessage);

      /**
       * This method is used to handle SourceActivity Result
       *
       * @param[in] amIfMessage - shared pointer to SourceActivityResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void sourceActivityResult(IN std::shared_ptr<AmIfMessage_SourceActivityResult> amIfMessage);

      /**
       * This method is used to handle audio player Playback Status
       *
       * @param[in] amIfMessage - shared pointer to PlaybackStatus
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playbackStatus(IN std::shared_ptr<AmIfMessage_PlaybackStatus> amIfMessage);

      /**
       * This method is used to notify the audio request failure or state change to PmAudioManagerWrapper
       *
       * @param[in] channelId - audio channel id
       * @param[in] routeState - audio routing status
       * @param[in] msg - status message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateStatus(const AudioChannel channelId, const RouteState routeState, const AmResultMessage& msg);

      /**
       * This method is used to notify the audio request failure or state change to PmAudioManagerWrapper
       *
       * @param[in] channelId - audio channel id
       * @param[in] routeState - audio routing status
       * @param[in] msg - status message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateMuteUnmuteStatus(IN const AmResultMessage& msg);

      /**
       * This method is used to set the RouteState of AudioChannel
       *
       * @param[in] channelId - audio channel id
       * @param[in] routeState - audio routing status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setRouteState(const AudioChannel channelId, const RouteState routeState);

      /**
       * This method is used to get the RouteState of AudioChannel
       *
       * @param[in] channelId - audio channel id
       * @param[out]
       * @param[in,out]
       *
       * @return RouteState - audio routing status
       */
      RouteState getRouteState(const AudioChannel channelId);

      /**
       * This method is used to check if any audio channel is active in session handler
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool isAnyAudioChannelActive();

      /**
       * This method is used to check if other audio channel is active in session handler
       *
       * @param[in] channel - audio channel
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool isOtherAudioChannelActive(const AudioChannel channel);

      /**
       * This method is used to get the active channel
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return AudioChannel - channel Id
       */
      AudioChannel getActiveChannel();

   private:
      /**
       * Constructor of AudioSessionHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      AudioSessionHandler();

      AmSessionId                    _amSessionId;                 /**< am session id */
      VoiceCallAudioHandler*         _voiceCallAudioHandler;       /**< pointer to VoiceCallAudiohandler */
      VehicleRingtoneAudioHandler*   _vehicleRingtoneAudioHandler; /**< pointer to VehicleRingtoneAudiohandler */
      VoiceRecAudioHandler*          _voiceRecAudioHandler;        /**< pointer to VoiceRecAudiohandler */
      WaitingModeToneAudioHandler*   _waitingModeToneAudioHandler; /**< pointer to WaitingModeToneAudioHandler */
      MuteUnmuteHandler*             _muteUnmuteHandler;           /**< pointer to MuteUnmuteHandler */
      ::std::map<AudioChannel, RouteState> _audioChannelToRouteStateMap; /**< mapping between channel and routing status */
   };
}

#endif //AudioSessionHandler_h
