/**
 * @file AmMainController.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the AmMainController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "AmMainController.h"
#include "AmSessionController.h"
#include "ArlWrapper.h"
#include "EcnrWrapper.h"
#include "GeniviAmWrapper.h"
#include "AudioPlayerWrapper.h"
#include "AmIfMessageRequest.h"
#include "AmIfMessageResult.h"
#include "ahl_fw.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AmMainController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager {

AmMainController::AmMainController()
{
   ETG_TRACE_USR4(("AmMainController(): Constructor"));

   _amSessionController = new AmSessionController;
   _ecnrWrapper = new EcnrWrapper;
   _geniviAmWrapper = new GeniviAmWrapper;
   _audioPlayerWrapper = new AudioPlayerWrapper;

   //the following members will be created later
   _arlWrapper = nullptr;
   _pmAudioManagerCallbackIfWrapper = nullptr;
}

AmMainController::~AmMainController()
{
   ETG_TRACE_USR4(("AmMainController(): Destructor"));

   if(nullptr != _amSessionController)
   {
      delete _amSessionController;
      _amSessionController = nullptr;
   }

   if(nullptr != _ecnrWrapper)
   {
      delete _ecnrWrapper;
      _ecnrWrapper = nullptr;
   }

   if(nullptr != _geniviAmWrapper)
   {
      delete _geniviAmWrapper;
      _geniviAmWrapper = nullptr;
   }

   if(nullptr != _arlWrapper)
   {
      delete _arlWrapper;
      _arlWrapper = nullptr;
   }

   if(nullptr != _audioPlayerWrapper)
   {
      delete _audioPlayerWrapper;
      _audioPlayerWrapper = nullptr;
   }

   if (nullptr != _pmAudioManagerCallbackIfWrapper)
   {
      delete _pmAudioManagerCallbackIfWrapper;
      _pmAudioManagerCallbackIfWrapper = nullptr;
   }
}

void AmMainController::setPmApplication(ahl_tclBaseOneThreadApp* mainApplication)
{
   ETG_TRACE_USR4(("setPmApplication() entered"));

   if(nullptr != mainApplication)
   {
      _arlWrapper = new ArlWrapper(mainApplication);
   }
   else
   {
      ETG_TRACE_ERR(("setPmApplication: mainApplication is null"));
   }
}

AmResult AmMainController::registerCallback(const IPmAudioManagerCallbackIf* pmAudioManagerCallbackIf)
{
   ETG_TRACE_USR1(("registerCallback: pmAudioManagerCallbackIf = 0x%p", (void *) pmAudioManagerCallbackIf));
   AmResult amResult(AM_RESULT_OK, "");

   if(nullptr != pmAudioManagerCallbackIf)
   {
      _pmAudioManagerCallbackIfWrapper = new PmAudioManagerCallbackIfWrapper(
            const_cast<IPmAudioManagerCallbackIf *>(pmAudioManagerCallbackIf));

      if(nullptr == _pmAudioManagerCallbackIfWrapper)
      {
         ETG_TRACE_FATAL(("registerCallback: _pmAudioManagerCallbackIfWrapper is 0"));
         amResult._amResultCode = AM_RESULT_ERR_GENERAL;
      }
      else
      {
         ETG_TRACE_USR1(("registering PmAudioManagerCallbackIfWrapper: _pmAudioManagerCallbackIfWrapper = 0x%p",
               (void *) _pmAudioManagerCallbackIfWrapper));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("registerCallback: pmAudioManagerCallbackIf is 0"));
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   return amResult;
}

PmAudioManagerCallbackIfWrapper& AmMainController::getPmAudioManagerCallbackIfWrapper(void)
{
   return *(_pmAudioManagerCallbackIfWrapper);
}

ArlWrapper& AmMainController::getArlWrapper(void)
{
   return *(_arlWrapper);
}

EcnrWrapper& AmMainController::getEcnrWrapper(void)
{
   return *(_ecnrWrapper);
}

GeniviAmWrapper& AmMainController::getGeniviAmWrapper(void)
{
   return *(_geniviAmWrapper);
}

AudioPlayerWrapper& AmMainController::getAudioPlayerWrapper(void)
{
   return *(_audioPlayerWrapper);
}

AmResult AmMainController::handleAmIfMessage(IN std::shared_ptr<AmIfMessage>& amIfMessage)
{
   AmResult amResult(AM_RESULT_OK, "");

   if(nullptr == amIfMessage)
   {
      ETG_TRACE_USR4(("handleAmIfMessage: amIfMessage is null"));
      amResult._amResultCode = AM_RESULT_ERR_INVALID_PARAMETER;
      return amResult;
   }

   ETG_TRACE_USR1(("handleAmIfMessage: amIfMessage = 0x%p", amIfMessage.get()));

   AmIfMsgId msgId = amIfMessage->getMessageId();

   switch(msgId)
   {
      case AM_IF_MSG_ID_PREPARE_AUDIO_ROUTE:
         handleAmIfMessage_PrepareAudioRouteRequest(
               std::static_pointer_cast<AmIfMessage_PrepareAudioRouteRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_PLAY_AUDIO_FILE:
         handleAmIfMessage_PlayAudioFileRequest(
               std::static_pointer_cast<AmIfMessage_PlayAudioFileRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_PLAY_HF_AUDIO:
         handleAmIfMessage_PlayHfAudioRequest(
               std::static_pointer_cast<AmIfMessage_PlayHfAudioRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_STOP_AUDIO:
         handleAmIfMessage_StopAudioRequest(
               std::static_pointer_cast<AmIfMessage_StopAudioRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_PAUSE_AUDIO:
         handleAmIfMessage_PauseAudioRequest(
               std::static_pointer_cast<AmIfMessage_PauseAudioRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_SET_MIC_MUTE_STATE:
         handleAmIfMessage_SetMicMuteStateRequest(
               std::static_pointer_cast<AmIfMessage_SetMicMuteStateRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_GET_AUDIO_SINK_VOLUME:
         handleAmIfMessage_GetAudioSinkVolumeRequest(
               std::static_pointer_cast<AmIfMessage_GetAudioSinkVolumeRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_SET_AUDIO_SINK_VOLUME:
         handleAmIfMessage_SetAudioSinkVolumeRequest(
               std::static_pointer_cast<AmIfMessage_SetAudioSinkVolumeRequest>(amIfMessage));
         break;
      case AM_IF_MSG_ID_ECNR_INITIALIZE_RESULT:
         handleAmIfMessage_EcnrInitializeResult(
               std::static_pointer_cast<AmIfMessage_EcnrInitializeResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_ECNR_DESTROY_RESULT:
         handleAmIfMessage_EcnrDestroyResult(
               std::static_pointer_cast<AmIfMessage_EcnrDestroyResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_ECNR_SET_CONFIGURATION_RESULT:
         handleAmIfMessage_EcnrSetConfigurationResult(
               std::static_pointer_cast<AmIfMessage_EcnrSetConfigurationResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_ECNR_SET_SEND_MUTE_SWITCH_RESULT:
         handleAmIfMessage_EcnrSetSendMuteSwitchResult(
               std::static_pointer_cast<AmIfMessage_EcnrSetSendMuteSwitchResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_ECNR_START_AUDIO_RESULT:
         handleAmIfMessage_EcnrStartAudioResult(
               std::static_pointer_cast<AmIfMessage_EcnrStartAudioResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_ECNR_STOP_AUDIO_RESULT:
         handleAmIfMessage_EcnrStopAudioResult(
               std::static_pointer_cast<AmIfMessage_EcnrStopAudioResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_ECNR_GET_VERSION_RESULT:
         handleAmIfMessage_EcnrGetVersionResult(
               std::static_pointer_cast<AmIfMessage_EcnrGetVersionResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_REQUEST_AUDIO_ROUTE_RESULT:
         handleAmIfMessage_RequestAudioRouteResult(
               std::static_pointer_cast<AmIfMessage_RequestAudioRouteResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_RELEASE_AUDIO_ROUTE_RESULT:
         handleAmIfMessage_ReleaseAudioRouteResult(
               std::static_pointer_cast<AmIfMessage_ReleaseAudioRouteResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_SOURCE_ACTIVITY_RESULT:
         handleAmIfMessage_SourceActivityResult(
               std::static_pointer_cast<AmIfMessage_SourceActivityResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_REQUEST_MUTE_ACTION_RESULT:
         handleAmIfMessage_RequestMuteActionResult(
               std::static_pointer_cast<AmIfMessage_RequestMuteActionResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_PLAYBACK_STATUS:
         handleAmIfMessage_PlaybackStatus(
               std::static_pointer_cast<AmIfMessage_PlaybackStatus>(amIfMessage));
         break;
      case AM_IF_MSG_ID_GAM_GET_LIST_MAIN_SINK_SND_PROP_RESULT:
         handleAmIfMessage_GetListMainSinkSoundPropertiesResult(
               std::static_pointer_cast<AmIfMessage_GetListMainSinkSoundPropertiesResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_GAM_SET_MAIN_SINK_SND_PROP_RESULT:
         handleAmIfMessage_SetMainSinkSoundPropertyResult(
               std::static_pointer_cast<AmIfMessage_SetMainSinkSoundPropertyResult>(amIfMessage));
         break;
      case AM_IF_MSG_ID_GAM_UPDATE_AUDIO_SINK_VOL_LIST:
         handleAmIfMessage_UpdateAudioSinkVolumeList(
               std::static_pointer_cast<AmIfMessage_UpdateAudioSinkVolumeList>(amIfMessage));
         break;
      default:
         ETG_TRACE_ERR(("handleAmIfMessage: AmIfMsgId is not valid"));
         amResult._amResultCode = AM_RESULT_ERR_INVALID_PARAMETER;
         break;
   }

   return amResult;
}

void AmMainController::handleAmIfMessage_PrepareAudioRouteRequest(
      IN std::shared_ptr<AmIfMessage_PrepareAudioRouteRequest> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->prepareAudioRouteRequest(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_PlayAudioFileRequest(
      IN std::shared_ptr<AmIfMessage_PlayAudioFileRequest> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->playAudioFileRequest(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_PlayHfAudioRequest(
      IN std::shared_ptr<AmIfMessage_PlayHfAudioRequest> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->playHfAudioRequest(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_StopAudioRequest(
      IN std::shared_ptr<AmIfMessage_StopAudioRequest> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->stopAudioRequest(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_PauseAudioRequest(
      IN std::shared_ptr<AmIfMessage_PauseAudioRequest> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->pauseAudioRequest(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_SetMicMuteStateRequest(
      IN std::shared_ptr<AmIfMessage_SetMicMuteStateRequest> amIfMessage)
{
   //This request should be handled directly from here, no need to send the request to SM
   if(nullptr != _amSessionController)
   {
      // EcnrSendMuteSwitch ecnrSendMuteSwitch = amIfMessage->getMuteState() ? 1 : 0;
      // _ecnrWrapper->setSendMuteSwitch(amIfMessage->getAmSessionId(), ECNR_APPID_PHONE, ecnrSendMuteSwitch);
      _amSessionController->setMicMuteStateRequest(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_GetAudioSinkVolumeRequest(
         IN std::shared_ptr<AmIfMessage_GetAudioSinkVolumeRequest> amIfMessage)
{
   if(nullptr != _geniviAmWrapper)
   {
      GeniviAMSinkId sinkId = amIfMessage->getSinkId();
      _geniviAmWrapper->getMainSinkSoundPropertiesList(amIfMessage->getAmSessionId(), sinkId);
   }
}

void AmMainController::handleAmIfMessage_SetAudioSinkVolumeRequest(
         IN std::shared_ptr<AmIfMessage_SetAudioSinkVolumeRequest> amIfMessage)
{
   if(nullptr != _geniviAmWrapper)
   {
      GeniviAMSinkId sinkId = amIfMessage->getSinkId();
      AmPropertyType2VolumeMap propertyMap= amIfMessage->getProperty();
      GeniviAMPropertyType propertyType = propertyMap.begin()->first;
      GeniviAMVolumeLevel volumeLevel = propertyMap.begin()->second;
      _geniviAmWrapper->setMainSinkSoundProperty(amIfMessage->getAmSessionId(), sinkId, propertyType, volumeLevel);
   }
}

void AmMainController::handleAmIfMessage_EcnrInitializeResult(
      IN std::shared_ptr<AmIfMessage_EcnrInitializeResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->ecnrInitializeResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_EcnrDestroyResult(
      IN std::shared_ptr<AmIfMessage_EcnrDestroyResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->ecnrDestroyResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_EcnrSetConfigurationResult(
      IN std::shared_ptr<AmIfMessage_EcnrSetConfigurationResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->ecnrSetConfigurationResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_EcnrSetSendMuteSwitchResult(
      IN std::shared_ptr<AmIfMessage_EcnrSetSendMuteSwitchResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->ecnrSetSendMuteSwitchResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_EcnrStartAudioResult(
      IN std::shared_ptr<AmIfMessage_EcnrStartAudioResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->ecnrStartAudioResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_EcnrStopAudioResult(
      IN std::shared_ptr<AmIfMessage_EcnrStopAudioResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->ecnrStopAudioResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_EcnrGetVersionResult(
      IN std::shared_ptr<AmIfMessage_EcnrGetVersionResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->ecnrGetVersionResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_RequestAudioRouteResult(
      IN std::shared_ptr<AmIfMessage_RequestAudioRouteResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->requestAudioRouteResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_ReleaseAudioRouteResult(
      IN std::shared_ptr<AmIfMessage_ReleaseAudioRouteResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->releaseAudioRouteResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_SourceActivityResult(
      IN std::shared_ptr<AmIfMessage_SourceActivityResult> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->sourceActivityResult(amIfMessage);
   }
}

void AmMainController::handleAmIfMessage_RequestMuteActionResult(
      IN std::shared_ptr<AmIfMessage_RequestMuteActionResult> amIfMessage)
{
   //This request should be handled directly from here, no need to send the request to SM
   if(nullptr != _pmAudioManagerCallbackIfWrapper)
   {
      _pmAudioManagerCallbackIfWrapper->setMicMuteStateResponse(amIfMessage->getAmSessionId(),
            amIfMessage->getAmResult());
   }
}

void AmMainController::handleAmIfMessage_GetListMainSinkSoundPropertiesResult(
      IN std::shared_ptr<AmIfMessage_GetListMainSinkSoundPropertiesResult> amIfMessage)
{
   if(nullptr != _pmAudioManagerCallbackIfWrapper)
   {
      _pmAudioManagerCallbackIfWrapper->getAudioSinkVolumeResponse(amIfMessage->getAmSessionId(),
            amIfMessage->getAmResult());
   }
}

void AmMainController::handleAmIfMessage_SetMainSinkSoundPropertyResult(
      IN std::shared_ptr<AmIfMessage_SetMainSinkSoundPropertyResult> amIfMessage)
{
   if(nullptr != _pmAudioManagerCallbackIfWrapper)
   {
      _pmAudioManagerCallbackIfWrapper->setAudioSinkVolumeResponse(amIfMessage->getAmSessionId(),
            amIfMessage->getAmResult());
   }
}

void AmMainController::handleAmIfMessage_UpdateAudioSinkVolumeList(
         IN std::shared_ptr<AmIfMessage_UpdateAudioSinkVolumeList> amIfMessage)
{
   if(nullptr != _pmAudioManagerCallbackIfWrapper)
   {
      _pmAudioManagerCallbackIfWrapper->updateAudioSinkVolumeList(amIfMessage->getAmSessionId(),
            amIfMessage->getAmResult(), amIfMessage->getSinkId(), amIfMessage->getPropertyMap());
   }
}

void AmMainController::handleAmIfMessage_PlaybackStatus(
      IN std::shared_ptr<AmIfMessage_PlaybackStatus> amIfMessage)
{
   if(nullptr != _amSessionController)
   {
      _amSessionController->playbackStatus(amIfMessage);
   }
}

} //pmaudiomanager
