/**
 * @file AmMainController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the AmMainController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#ifndef AmMainController_h
#define AmMainController_h

#include "PmSingleton.h"
#include "PmAudioManagerIfTypes.h"
#include "PmAudioManagerCallbackIfWrapper.h"

//class forward declarations
class ahl_tclBaseOneThreadApp;

namespace pmaudiomanager {

//class forward declarations
class ArlWrapper;
class EcnrWrapper;
class GeniviAmWrapper;
class AudioPlayerWrapper;
class AmSessionController;

class AmIfMessage;

//Method Requests
class AmIfMessage_PrepareAudioRouteRequest;
class AmIfMessage_PlayAudioFileRequest;
class AmIfMessage_PlayHfAudioRequest;
class AmIfMessage_StopAudioRequest;
class AmIfMessage_PauseAudioRequest;
class AmIfMessage_SetMicMuteStateRequest;
class AmIfMessage_GetAudioSinkVolumeRequest;
class AmIfMessage_SetAudioSinkVolumeRequest;

//Method Results
class AmIfMessage_EcnrInitializeResult;
class AmIfMessage_EcnrDestroyResult;
class AmIfMessage_EcnrSetConfigurationResult;
class AmIfMessage_EcnrSetSendMuteSwitchResult;
class AmIfMessage_EcnrStartAudioResult;
class AmIfMessage_EcnrStopAudioResult;
class AmIfMessage_EcnrGetVersionResult;
class AmIfMessage_RequestAudioRouteResult;
class AmIfMessage_ReleaseAudioRouteResult;
class AmIfMessage_SourceActivityResult;
class AmIfMessage_RequestMuteActionResult;
class AmIfMessage_PlaybackStatus;
class AmIfMessage_GetListMainSinkSoundPropertiesResult;
class AmIfMessage_SetMainSinkSoundPropertyResult;
class AmIfMessage_UpdateAudioSinkVolumeList;

/**
 * AmMainController class.
 */
class AmMainController : public PmSingleton <AmMainController>
{
public:
   /**
    * This method is used to set the pm application.
    *
    * @param[in] mainApplication - Pointer to the AhlApp class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setPmApplication(ahl_tclBaseOneThreadApp* mainApplication);

   /**
    * This method is used to register with the CallbackIfWrapper.
    *
    * @param[in] pmAudioManagerCallbackIf - Pointer to the IPmAudioManagerCallbackIf class
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult (AmResultCode, AmResultMessage)
    */
   AmResult registerCallback(const IPmAudioManagerCallbackIf* pmAudioManagerCallbackIf);

   /**
    * This method is used to get the Arl Wrapper
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return ArlWrapper - Reference to the ArlWrapper
    */
   ArlWrapper& getArlWrapper(void);

   /**
    * This method is used to get the Ecnr Wrapper
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrWrapper - Reference to the EcnrWrapper
    */
   EcnrWrapper& getEcnrWrapper(void);

   /**
    * This method is used to get the GeniviAm Wrapper
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return GeniviAmWrapper - Reference to the GeniviAmWrapper
    */
   GeniviAmWrapper& getGeniviAmWrapper(void);

   /**
    * This method is used to get the AudioPlayer Wrapper
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return AudioPlayerWrapper - Reference to the AudioPlayerWrapper
    */
   AudioPlayerWrapper& getAudioPlayerWrapper(void);

   /**
    * This method is used to get the PmAudioManager Callback Wrapper
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return PmAudioManagerCallbackIfWrapper - Reference to the PmAudioManagerCallbackIfWrapper
    */
   PmAudioManagerCallbackIfWrapper& getPmAudioManagerCallbackIfWrapper(void);

   /**
    * This method is used to handle the incoming request and result messages to the AmMainController.
    *
    * @param[in] amIfMessage - Pointer to the AmIfMessage class
    * @param[out]
    * @param[in,out]
    *
    * @return AmResult
    */
   AmResult handleAmIfMessage(IN std::shared_ptr<AmIfMessage>& amIfMessage);

private:
   friend class PmSingleton<AmMainController>;

   /**
    * Constructor of AmMainController class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   AmMainController();

   /**
    * Destructor of AmMainController class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   virtual ~AmMainController();

   /**
    * Copy constructor => must not be used.
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * return
    */
   AmMainController(IN const AmMainController& ref);

   /**
    * Assignment operator => must not be used.
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * @return reference to AmMainController
    */
   AmMainController& operator=(IN const AmMainController& ref);

   /**
    * This method is used to handle PrepareAudioRoute Request
    *
    * @param[in] amIfMessage - shared pointer to PrepareAudioRouteRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_PrepareAudioRouteRequest(
         IN std::shared_ptr<AmIfMessage_PrepareAudioRouteRequest> amIfMessage);

   /**
    * This method is used to handle PlayAudioFile Request
    *
    * @param[in] amIfMessage - shared pointer to PlayAudioFileRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_PlayAudioFileRequest(IN std::shared_ptr<AmIfMessage_PlayAudioFileRequest> amIfMessage);

   /**
    * This method is used to handle PlayHfAudio Request
    *
    * @param[in] amIfMessage - shared pointer to PlayHfAudioRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_PlayHfAudioRequest(IN std::shared_ptr<AmIfMessage_PlayHfAudioRequest> amIfMessage);

   /**
    * This method is used to handle StopAudio Request
    *
    * @param[in] amIfMessage - shared pointer to StopAudioRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_StopAudioRequest(IN std::shared_ptr<AmIfMessage_StopAudioRequest> amIfMessage);

   /**
    * This method is used to handle PauseAudio Request
    *
    * @param[in] amIfMessage - shared pointer to PauseAudioRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_PauseAudioRequest(IN std::shared_ptr<AmIfMessage_PauseAudioRequest> amIfMessage);

   /**
    * This method is used to handle GetAudioSinkVolume Request
    *
    * @param[in] amIfMessage - shared pointer to PauseAudioRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_GetAudioSinkVolumeRequest(
         IN std::shared_ptr<AmIfMessage_GetAudioSinkVolumeRequest> amIfMessage);

   /**
    * This method is used to handle SetAudioSinkVolume Request
    *
    * @param[in] amIfMessage - shared pointer to PauseAudioRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_SetAudioSinkVolumeRequest(
         IN std::shared_ptr<AmIfMessage_SetAudioSinkVolumeRequest> amIfMessage);

   /**
    * This method is used to handle SetMicMuteState Request
    *
    * @param[in] amIfMessage - shared pointer to SetMicMuteStateRequest
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_SetMicMuteStateRequest(
         IN std::shared_ptr<AmIfMessage_SetMicMuteStateRequest> amIfMessage);

   /**
    * This method is used to handle EcnrInitialize Result
    *
    * @param[in] amIfMessage - shared pointer to EcnrInitializeResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_EcnrInitializeResult(
         IN std::shared_ptr<AmIfMessage_EcnrInitializeResult> amIfMessage);

   /**
    * This method is used to handle EcnrDestroy Result
    *
    * @param[in] amIfMessage - shared pointer to EcnrDestroyResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_EcnrDestroyResult(IN std::shared_ptr<AmIfMessage_EcnrDestroyResult> amIfMessage);

   /**
    * This method is used to handle EcnrSetConfiguration Result
    *
    * @param[in] amIfMessage - shared pointer to EcnrSetConfigurationResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_EcnrSetConfigurationResult(
         IN std::shared_ptr<AmIfMessage_EcnrSetConfigurationResult> amIfMessage);

   /**
    * This method is used to handle EcnrSetSendMuteSwitch Result
    *
    * @param[in] amIfMessage - shared pointer to EcnrSetSendMuteSwitchResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_EcnrSetSendMuteSwitchResult(
         IN std::shared_ptr<AmIfMessage_EcnrSetSendMuteSwitchResult> amIfMessage);

   /**
    * This method is used to handle EcnrStartAudio Result
    *
    * @param[in] amIfMessage - shared pointer to EcnrStartAudioResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_EcnrStartAudioResult(
         IN std::shared_ptr<AmIfMessage_EcnrStartAudioResult> amIfMessage);

   /**
    * This method is used to handle EcnrStopAudio Result
    *
    * @param[in] amIfMessage - shared pointer to EcnrStopAudioResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_EcnrStopAudioResult(
         IN std::shared_ptr<AmIfMessage_EcnrStopAudioResult> amIfMessage);

   /**
    * This method is used to handle EcnrGetVersion Result
    *
    * @param[in] amIfMessage - shared pointer to EcnrGetVersionResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_EcnrGetVersionResult(
         IN std::shared_ptr<AmIfMessage_EcnrGetVersionResult> amIfMessage);

   /**
    * This method is used to handle RequestAudioRoute Result
    *
    * @param[in] amIfMessage - shared pointer to RequestAudioRouteResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_RequestAudioRouteResult(
         IN std::shared_ptr<AmIfMessage_RequestAudioRouteResult> amIfMessage);

   /**
    * This method is used to handle ReleaseAudioRoute Result
    *
    * @param[in] amIfMessage - shared pointer to ReleaseAudioRouteResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_ReleaseAudioRouteResult(
         IN std::shared_ptr<AmIfMessage_ReleaseAudioRouteResult> amIfMessage);

   /**
    * This method is used to handle SourceActivity Result
    *
    * @param[in] amIfMessage - shared pointer to SourceActivityResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_SourceActivityResult(
         IN std::shared_ptr<AmIfMessage_SourceActivityResult> amIfMessage);

   /**
    * This method is used to handle RequestMuteAction Result
    *
    * @param[in] amIfMessage - shared pointer to RequestMuteActionResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_RequestMuteActionResult(
         IN std::shared_ptr<AmIfMessage_RequestMuteActionResult> amIfMessage);

   /**
    * This method is used to handle GetListMainSinkSoundProperties Result
    *
    * @param[in] amIfMessage - shared pointer to GetListMainSinkSoundPropertiesResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_GetListMainSinkSoundPropertiesResult(
         IN std::shared_ptr<AmIfMessage_GetListMainSinkSoundPropertiesResult> amIfMessage);

   /**
    * This method is used to handle SetMainSinkSoundProperty Result
    *
    * @param[in] amIfMessage - shared pointer to SetMainSinkSoundPropertyResult
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_SetMainSinkSoundPropertyResult(
         IN std::shared_ptr<AmIfMessage_SetMainSinkSoundPropertyResult> amIfMessage);

   /**
    * This method is used to handle UpdateMainSinkSoundProperty Signal
    *
    * @param[in] amIfMessage - shared pointer to UpdateMainSinkSoundProperty
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_UpdateAudioSinkVolumeList(
         IN std::shared_ptr<AmIfMessage_UpdateAudioSinkVolumeList> amIfMessage);

   /**
    * This method is used to handle audio player Playback Status
    *
    * @param[in] amIfMessage - shared pointer to PlaybackStatus
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void handleAmIfMessage_PlaybackStatus(
         IN std::shared_ptr<AmIfMessage_PlaybackStatus> amIfMessage);

   AmSessionController*             _amSessionController;             /**< Instance of AmSession Controller */
   ArlWrapper*                      _arlWrapper;                      /**< Instance of Arl Wrapper */
   EcnrWrapper*                     _ecnrWrapper;                     /**< Instance of Ecnr Wrapper */
   GeniviAmWrapper*                 _geniviAmWrapper;                 /**< Instance of GeniviAm Wrapper */
   AudioPlayerWrapper*              _audioPlayerWrapper;              /**< Instance of AudioPlayer Wrapper */
   PmAudioManagerCallbackIfWrapper* _pmAudioManagerCallbackIfWrapper; /**< Instance of PmAudioManagerCallbackIf Wrapper */
};

} //pmaudiomanager

#endif // AmMainController_h
