/**
 * @file PmAudioManagerCallbackIfWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmAudioManagerCallbackIfWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CPP callback interfaces of PmAudioManager.
 *
 * @ingroup PmAudioManager
 */


#ifndef PmAudioManagerCallbackIfWrapper_h
#define PmAudioManagerCallbackIfWrapper_h

#include "PmAudioManagerIfTypes.h"

namespace pmaudiomanager
{

/**
 * forward declaration
 */
class IPmAudioManagerCallbackIf;

class PmAudioManagerCallbackIfWrapper
{
public:

   /**
    * Constructor of the interface class PmAudioManagerCallbackIfWrapper
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   PmAudioManagerCallbackIfWrapper(IPmAudioManagerCallbackIf* iPmAudioManagerCallbackIfWrapper);

   /**
    * destructor of the interface class PmAudioManagerCallbackIfWrapper
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   ~PmAudioManagerCallbackIfWrapper();

   /**
    * Response interface for the prepareAudioRoute request.
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void prepareAudioRouteResponse(IN const AmSessionId amSessionId, IN const AmResult amResult);

   /**
    * Response interface for the playAudio request
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void playAudioResponse(IN const AmSessionId amSessionId, IN const AmResult amResult);

   /**
    * Response for stopAudio interface.
    * Response will be sent as success once the request is considered for processing.
    * AmSessionId will become invalid once the stopAudio is processed
    * i.e. when the amNotificationEvent is sent with the StreamingState set as STOPPED
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void stopAudioResponse(IN const AmSessionId amSessionId, IN const AmResult amResult);

   /**
    * This API is used to notify:
    *    1. Audio streaming / playing status
    *    2. Error response when the acquisition (or) release of AudioChannel failed.
    *    3. Update when the audio is stopped / paused due to a trigger from ARL (Audio Routing Library)
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AudioChannel - Prepare the audio route for the requested audio channel.
    *            StreamingState - Gives the Streaming state for the corresponding Audio session
    *            AmEventDetails - Gives the details of the event
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void amNotificationEvent(IN const AmSessionId amSessionId, IN const AudioChannel audioChannelId,
         IN const StreamingState streamingState, IN const AmEventDetails eventInfo);

   /**
    * Response for setMicMuteState interface. The final Mute state is sent through the amNotificationEvent method.
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void setMicMuteStateResponse(IN const AmSessionId amSessionId, IN const AmResult amResult);

   /**
    * Response for pauseAudio interface.
    * Response will be sent as success once the request is considered for processing.
    * AmSessionId will become invalid once the stopAudio is processed
    * i.e. when the amNotificationEvent is sent with the StreamingState set as NOT_PLAYING
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void pauseAudioResponse(IN const AmSessionId amSessionId, IN const AmResult amResult);

   /**
    * Response for getAudioSinkVolume interface. The proprty list is sent through the updateAudioSinkVolumeList interface.
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void getAudioSinkVolumeResponse(IN const AmSessionId amSessionId, IN const AmResult amResult);

   /**
    * Response for setAudioSinkVolume interface.
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void setAudioSinkVolumeResponse(IN const AmSessionId amSessionId, IN const AmResult amResult);

   /**
    * Invokes updateAudioSinkVolumeList Interface.
    *
    * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
    *                if the request is successful, then the new session ID will be created by PM AM
    *                for the corresponding audio session and will update it.
    *            AmResult - Result of the operation
    *            GeniviAMSinkId - Sink ID
    *            AmPropertyType2VolumeMap - Property List
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void updateAudioSinkVolumeList(IN const AmSessionId amSessionId, IN const AmResult amResult,
         IN const GeniviAMSinkId sinkId, IN const AmPropertyType2VolumeMap propertyList);

private:
   IPmAudioManagerCallbackIf* _pmAudioManagerCallbackIf; /**< pointer to IPmAudioManagerCallbackIf class */
};

} /* namespace pmaudiomanager */

#endif /* PmAudioManagerCallbackIfWrapper_h */
