/**
 * @file GeniviAmWrapper.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file used to request the GeniviAm methods and response from GeniviAm is forwarded to AmMainController
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "GeniviAmWrapper.h"
#include "GeniviAmCallBackIf.h"
#include "GeniviAmRequestIf.h"
#include "AmIfMessageCreator.h"
#include "AmMainController.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/GeniviAmWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
GeniviAmWrapper::GeniviAmWrapper() : GeniviAmWrapper(new(std::nothrow) GeniviAmCallBackIf(this), new(std::nothrow) GeniviAmRequestIf())
{
    ETG_TRACE_USR4(("GeniviAmWrapper(): Constructor"));
}

GeniviAmWrapper::GeniviAmWrapper(IGeniviAmCallBackIf* geniviAmCallbackIf, IGeniviAmRequestIf* geniviamRequestIf) : _actToSessionIdMap()
{
   ETG_TRACE_USR4(("GeniviAmWrapper(): Parameterized Constructor"));
   _geniviAMCallBackIf = geniviAmCallbackIf;
   _geniviAMRequestIf = geniviamRequestIf;

   if((nullptr != _geniviAMRequestIf) && (nullptr != _geniviAMCallBackIf))
   {
      // register callback handler
      _geniviAMRequestIf->setCallbackIf(_geniviAMCallBackIf);
   }
}

GeniviAmWrapper::~GeniviAmWrapper()
{
   ETG_TRACE_USR4(("GeniviAmWrapper(): Destructor"));

   clearDbusActToAmSessionIdMap();

   if(nullptr != _geniviAMRequestIf)
   {
      delete _geniviAMRequestIf;
      _geniviAMRequestIf = nullptr;
   }

   if(nullptr != _geniviAMCallBackIf)
   {
      delete _geniviAMCallBackIf;
      _geniviAMCallBackIf = nullptr;
   }
}

AmErrorCode GeniviAmWrapper::getMainSinkSoundPropertiesList(const AmSessionId amSessionId, const GeniviAMSinkId sinkId)
{
   ETG_TRACE_USR4(("GeniviAmWrapper::getMainSinkSoundPropertiesList: Sink Id : %d", sinkId));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   AmErrorCode errorCode = GeniviAmError;

   if(nullptr != _geniviAMRequestIf)
   {
      act = _geniviAMRequestIf->sendGetMainSinkSoundPropertiesList(sinkId);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("getMainSinkSoundPropertiesList: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = GeniviAmOk;
      }
      else
      {
         ETG_TRACE_ERR(("getMainSinkSoundPropertiesList: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("getMainSinkSoundPropertiesList: _geniviAMRequestIf is null"));
   }

   return errorCode;
}

void GeniviAmWrapper::getListMainSinkSoundPropertiesCB(const GeniviAMResult responseStatus, const act_t act)
{
   ETG_TRACE_USR4(("GeniviAmWrapper::getListMainSinkSoundPropertiesCB::ResponseStatus"));
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(GeniviAmReqSuccess != responseStatus)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_GetListMainSinkSoundPropertiesResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

void GeniviAmWrapper::getListMainSinkSoundPropertiesCB(const AmPropertyType2VolumeMap propertyValue, const act_t act)
{
   ETG_TRACE_USR4(("GeniviAmWrapper::getListMainSinkSoundPropertiesCB::PropertyList"));
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   const GeniviAMSinkId DefaultAmSinkId = 1;

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_UpdateAudioSinkVolumeList(amSessionId, amResult, DefaultAmSinkId, propertyValue);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

AmErrorCode GeniviAmWrapper::setMainSinkSoundProperty(const AmSessionId amSessionId, const GeniviAMSinkId sinkId, const GeniviAMPropertyType propertyType, const GeniviAMVolumeLevel volumeLevel)
{
   ETG_TRACE_USR4(("GeniviAmWrapper::setMainSinkSoundProperty: Sink Id : %d", sinkId));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   AmErrorCode errorCode = GeniviAmError;

   if(nullptr != _geniviAMRequestIf)
   {
      act = _geniviAMRequestIf->sendSetMainSinkSoundProperty(sinkId, propertyType, volumeLevel);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("setMainSinkSoundProperty: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = GeniviAmOk;
      }
      else
      {
         ETG_TRACE_ERR(("setMainSinkSoundProperty: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("setMainSinkSoundProperty: _geniviAMRequestIf is null"));
   }

   return errorCode;
}

void GeniviAmWrapper::setMainSinkSoundPropertyCB(const GeniviAMResult responseStatus, const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(GeniviAmReqSuccess != responseStatus)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_SetMainSinkSoundPropertyResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

void GeniviAmWrapper::onMainSinkSoundPropertyChanged(const GeniviAMSinkId sinkId, const GeniviAMPropertyType propertyType, const GeniviAMVolumeLevel volumeLevel)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   ETG_TRACE_USR4(("GeniviAmWrapper::onMainSinkSoundPropertyChangedSignal():: SinkId: %d, PropType: %d, PropVal: %d", sinkId, propertyType, volumeLevel));

   AmPropertyType2VolumeMap propertyMap;
   propertyMap.emplace(propertyType, volumeLevel);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_UpdateAudioSinkVolumeList(amSessionId, amResult, sinkId, propertyMap);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

void GeniviAmWrapper::mapDbusActToAmSessionId(const act_t dbusAct, const AmSessionId amSessionId)
{
   auto it = _actToSessionIdMap.find(dbusAct);
   if(it != _actToSessionIdMap.end())
   {
      //this should never happen
      ETG_TRACE_ERR(("Received DBus ACT is already available in the map"));
      //NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      _actToSessionIdMap.emplace(dbusAct, amSessionId);
   }
}

void GeniviAmWrapper::retrieveAmSessionIdFromDbusAct(const act_t dbusAct, AmSessionId& amSessionId)
{
   auto it = _actToSessionIdMap.find(dbusAct);
   if(it != _actToSessionIdMap.end())
   {
      amSessionId = it->second;

      //clear the dbus act_t once the required response is received
      _actToSessionIdMap.erase(it);
   }
   else
   {
      //this should never happen
      ETG_TRACE_ERR(("Received DBus ACT is not available in the map"));
      //NORMAL_M_ASSERT_ALWAYS();
   }
}

void GeniviAmWrapper::clearDbusActToAmSessionIdMap()
{
   _actToSessionIdMap.clear();
}

} //pmaudiomanager
