/**
 * @file GeniviAmCallBackIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for GeniviAM Command Interface DBUS callback
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "GeniviAmCallBackIf.h"
#include "GeniviAmWrapper.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/GeniviAmCallBackIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

using namespace ::org::genivi::audiomanager::CommandInterface;
using namespace ::asf::core;

namespace pmaudiomanager
{
    GeniviAmCallBackIf::GeniviAmCallBackIf()
    {
       ETG_TRACE_USR4(("GeniviAmCallBackIf() : Constructor"));

       _geniviAmWrapper = nullptr;
    }

    GeniviAmCallBackIf::GeniviAmCallBackIf(GeniviAmWrapper *geniviAmWrapper)
    {
       ETG_TRACE_USR4(("GeniviAmCallBackIf() : Parameterized Constructor"));

       _geniviAmWrapper = geniviAmWrapper;
    }

    GeniviAmCallBackIf::~GeniviAmCallBackIf()
    {
       ETG_TRACE_USR4(("GeniviAmCallBackIf() : Destructor"));

       _geniviAmWrapper = nullptr;
    }

    void GeniviAmCallBackIf::onServiceProxyAvailable(const ServiceState previousState, const ServiceState currentState,
        const ::std::string& objPath)
    {
       ETG_TRACE_USR4(("GeniviAmCallBackIf() : onServiceProxyAvailable"));
       (void)(previousState);
       (void)(currentState);
       (void)(objPath);
    }

    void GeniviAmCallBackIf::onServiceProxyUnavailable(const ServiceState previousState, const ServiceState currentState,
        const ::std::string& objPath)
    {
       ETG_TRACE_USR4(("GeniviAmCallBackIf() : onServiceProxyUnavailable"));
       (void)(previousState);
       (void)(currentState);
       (void)(objPath);
    }

    void GeniviAmCallBackIf::onGetListMainSinkSoundPropertiesError(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy,
        const ::boost::shared_ptr< GetListMainSinkSoundPropertiesError >& error)
    {
       ETG_TRACE_USR4(("GeniviAmCallBackIf() : onGetListMainSinkSoundPropertiesError"));

       (void)(proxy);

       if (nullptr != _geniviAmWrapper)
       {
          _geniviAmWrapper->getListMainSinkSoundPropertiesCB(GeniviAmReqFailed, error->getAct());
       }
    }

    void GeniviAmCallBackIf::onGetListMainSinkSoundPropertiesResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy,
        const ::boost::shared_ptr< GetListMainSinkSoundPropertiesResponse >& response)
    {
       ETG_TRACE_USR4(("GeniviAmCallBackIf() : onGetListMainSinkSoundPropertiesResponse"));

       (void)(proxy);

       if (nullptr != _geniviAmWrapper)
       {
          //Response is a vector of uint16 pairs: create a map from the vector
          auto propertyResponse = response->getListSoundProperties();

          AmPropertyType2VolumeMap mainSinkSoundPropertyResponseMap;

          for (auto propResponse : propertyResponse)
          {
             mainSinkSoundPropertyResponseMap.emplace(propResponse.getElem1(),propResponse.getElem2());
          }

          _geniviAmWrapper->getListMainSinkSoundPropertiesCB(mainSinkSoundPropertyResponseMap, response->getAct());
       }
    }

    void GeniviAmCallBackIf::onSetMainSinkSoundPropertyError(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy,
        const ::boost::shared_ptr< SetMainSinkSoundPropertyError >& error)
    {
       ETG_TRACE_USR4(("onSetMainSinkSoundPropertyError()"));

       (void)(proxy);

       if (nullptr != _geniviAmWrapper)
       {
          _geniviAmWrapper->setMainSinkSoundPropertyCB(GeniviAmReqFailed, error->getAct());
       }

    }

    void GeniviAmCallBackIf::onSetMainSinkSoundPropertyResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy,
        const ::boost::shared_ptr< SetMainSinkSoundPropertyResponse >& response)
    {
       ETG_TRACE_USR4(("onSetMainSinkSoundPropertyResponse()"));

       (void)(proxy);

       GeniviAMResult amResult = GeniviAmReqSuccess;

       if (response->getResult() != 0)
       {
          ETG_TRACE_ERR(("onSetMainSinkSoundPropertyResponse() : Error Code :: %d",response->getResult()));
          amResult = GeniviAmReqFailed;
       }

       if (nullptr != _geniviAmWrapper)
       {
          _geniviAmWrapper->setMainSinkSoundPropertyCB(amResult, response->getAct());
       }
    }

    void GeniviAmCallBackIf::onMainSinkSoundPropertyChangedError(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, 
        const ::boost::shared_ptr< MainSinkSoundPropertyChangedError >& error)
    {
       (void)(proxy);
       (void)(error);

       ETG_TRACE_ERR(("onMainSinkSoundPropertyChangedError()"));
    }

    void GeniviAmCallBackIf::onMainSinkSoundPropertyChangedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& proxy, 
        const ::boost::shared_ptr< MainSinkSoundPropertyChangedSignal >& signal)
    {
       (void)(proxy);

       ETG_TRACE_USR4(("onMainSinkSoundPropertyChangedSignal()"));

       if (nullptr != _geniviAmWrapper)
       {
          _geniviAmWrapper->onMainSinkSoundPropertyChanged(signal->getSinkID(), signal->getSoundProperty().getElem1(), signal->getSoundProperty().getElem2());
       }
    }

}   //pmaudiomanager
