/**
 * @file EcnrWrapper.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file used to request the EcNr methods and response from EcNr is forwarded to AmMainController
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "EcnrWrapper.h"
#include "EcnrCallBackIf.h"
#include "EcnrRequestIf.h"
#include "AmIfMessageCreator.h"
#include "AmMainController.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EcnrWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

#define FW_S_IMPORT_INTERFACE_ASSERT
#include "framework_if_if.h"

namespace pmaudiomanager {

EcnrWrapper::EcnrWrapper() : _actToSessionIdMap()
{
   ETG_TRACE_USR4(("EcnrWrapper(): Constructor"));

   _ecnrCallBackIf = new EcnrCallBackIf(this);
   _ecnrRequestIf = new EcnrRequestIf();

   if((nullptr != _ecnrRequestIf) && (nullptr != _ecnrCallBackIf))
   {
      // register callback handler
      _ecnrRequestIf->setCallbackIf(_ecnrCallBackIf);
   }
}

EcnrWrapper::~EcnrWrapper()
{
   ETG_TRACE_USR4(("EcnrWrapper(): Destructor"));

   clearDbusActToAmSessionIdMap();

   if(nullptr != _ecnrRequestIf)
   {
      delete _ecnrRequestIf;
      _ecnrRequestIf = nullptr;
   }

   if(nullptr != _ecnrCallBackIf)
   {
      delete _ecnrCallBackIf;
      _ecnrCallBackIf = nullptr;
   }
}

EcnrErrorCode EcnrWrapper::initialize(IN const AmSessionId amSessionId, IN const EcnrAppId appId,
      IN const EcnrConfigurationId configurationId)
{
   ETG_TRACE_USR4(("EcnrWrapper::EcnrInitialize: AppId : %d, ConfigurationId : %d", appId, configurationId));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   EcnrErrorCode errorCode = ECNR_ERROR;

   if(nullptr != _ecnrRequestIf)
   {
      act = _ecnrRequestIf->sendEcnrInitializeRequest(appId, configurationId);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("EcnrInitialize: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = ECNR_OK;
      }
      else
      {
         ETG_TRACE_ERR(("EcnrInitialize: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("EcnrInitialize: _ecnrRequestIf is null"));
   }

   return errorCode;
}

void EcnrWrapper::initializeResultCB(IN const EcnrRequestResult result, IN const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(ECNR_REQ_SUCCESS != result)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_EcnrInitializeResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

EcnrErrorCode EcnrWrapper::destroy(IN const AmSessionId amSessionId, IN const EcnrAppId appId)
{
   ETG_TRACE_USR4(("EcnrWrapper::EcnrDestroy: AppId : %d", appId));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   EcnrErrorCode errorCode = ECNR_ERROR;

   if(nullptr != _ecnrRequestIf)
   {
      act = _ecnrRequestIf->sendEcnrDestroyRequest(appId);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("EcnrDestroy: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = ECNR_OK;
      }
      else
      {
         ETG_TRACE_ERR(("EcnrDestroy: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("EcnrDestroy: _ecnrRequestIf is null"));
   }

   return errorCode;
}

void EcnrWrapper::destroyResultCB(IN const EcnrRequestResult result, IN const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(ECNR_REQ_SUCCESS != result)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_EcnrDestroyResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

EcnrErrorCode EcnrWrapper::setConfiguration(IN const AmSessionId amSessionId, IN const EcnrAppId appId,
      IN const EcnrConfigurationId configurationId)
{
   ETG_TRACE_USR4(("EcnrWrapper::EcnrSetConfiguration: AppId : %d, ConfigurationId : %d", appId, configurationId));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   EcnrErrorCode errorCode = ECNR_ERROR;

   if(nullptr != _ecnrRequestIf)
   {
      act = _ecnrRequestIf->sendEcnrSetConfigurationRequest(appId, configurationId);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("EcnrSetConfiguration: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = ECNR_OK;
      }
      else
      {
         ETG_TRACE_ERR(("EcnrSetConfiguration: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("EcnrSetConfiguration: _ecnrRequestIf is null"));
   }

   return errorCode;
}

void EcnrWrapper::setConfigurationResultCB(IN const EcnrRequestResult result, IN const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(ECNR_REQ_SUCCESS != result)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_EcnrSetConfigurationResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

EcnrErrorCode EcnrWrapper::startAudio(IN const AmSessionId amSessionId, IN const EcnrAppId appId)
{
   ETG_TRACE_USR4(("EcnrWrapper::EcnrStartAudio: AppId : %d", appId));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   EcnrErrorCode errorCode = ECNR_ERROR;

   if(nullptr != _ecnrRequestIf)
   {
      act = _ecnrRequestIf->sendEcnrStartAudioRequest(appId);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("EcnrStartAudio: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = ECNR_OK;
      }
      else
      {
         ETG_TRACE_ERR(("EcnrStartAudio: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("EcnrStartAudio: _ecnrRequestIf is null"));
   }

   return errorCode;
}

void EcnrWrapper::startAudioResultCB(IN const EcnrRequestResult result, IN const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(ECNR_REQ_SUCCESS != result)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_EcnrStartAudioResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

EcnrErrorCode EcnrWrapper::stopAudio(IN const AmSessionId amSessionId, IN const EcnrAppId appId)
{
   ETG_TRACE_USR4(("EcnrWrapper::EcnrStopAudio: AppId : %d", appId));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   EcnrErrorCode errorCode = ECNR_ERROR;

   if(nullptr != _ecnrRequestIf)
   {
      act = _ecnrRequestIf->sendEcnrStopAudioRequest(appId);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("EcnrStopAudio: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = ECNR_OK;
      }
      else
      {
         ETG_TRACE_ERR(("EcnrStopAudio: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("EcnrStopAudio: _ecnrRequestIf is null"));
   }

   return errorCode;
}

void EcnrWrapper::stopAudioResultCB(IN const EcnrRequestResult result, IN const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(ECNR_REQ_SUCCESS != result)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_EcnrStopAudioResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

EcnrErrorCode EcnrWrapper::setSendMuteSwitch(IN const AmSessionId amSessionId, IN const EcnrAppId appId,
      IN const EcnrSendMuteSwitch sendMuteSwitch)
{
   ETG_TRACE_USR4(("EcnrWrapper::EcnrSetSendMuteSwitch: AppId : %d, SendMuteSwitch : %d", appId, sendMuteSwitch));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   EcnrErrorCode errorCode = ECNR_ERROR;

   if(nullptr != _ecnrRequestIf)
   {
      act = _ecnrRequestIf->sendEcnrSetSendMuteSwitchRequest(appId, sendMuteSwitch);

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("EcnrSetSendMuteSwitch: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = ECNR_OK;
      }
      else
      {
         ETG_TRACE_ERR(("EcnrSetSendMuteSwitch: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("EcnrSetSendMuteSwitch: _ecnrRequestIf is null"));
   }

   return errorCode;
}

void EcnrWrapper::setSendMuteSwitchResultCB(IN const EcnrRequestResult result, IN const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(ECNR_REQ_SUCCESS != result)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_EcnrSetSendMuteSwitchResult(amSessionId, amResult);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

EcnrErrorCode EcnrWrapper::getVersion(IN const AmSessionId amSessionId)
{
   ETG_TRACE_USR4(("EcnrWrapper::EcnrGetVersion"));

   act_t act = ::ccdbusif::DEFAULT_ACT;
   EcnrErrorCode errorCode = ECNR_ERROR;

   if(nullptr != _ecnrRequestIf)
   {
      act = _ecnrRequestIf->sendEcnrGetVersionRequest();

      if(::ccdbusif::DEFAULT_ACT != act)
      {
         ETG_TRACE_USR4(("EcnrGetVersion: Success"));

         mapDbusActToAmSessionId(act, amSessionId);
         errorCode = ECNR_OK;
      }
      else
      {
         ETG_TRACE_ERR(("EcnrGetVersion: Failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("EcnrGetVersion: _ecnrRequestIf is null"));
   }

   return errorCode;
}

void EcnrWrapper::getVersionResultCB(IN const EcnrRequestResult result, IN const VersionNumber& versionNumber,
      IN const Version& versionString, IN const VersionComment& versionComment, IN const act_t act)
{
   AmResult amResult(AM_RESULT_OK, "");
   AmSessionId amSessionId(AM_SESSION_ID_DEFAULT);

   if(ECNR_REQ_SUCCESS != result)
   {
      amResult._amResultCode = AM_RESULT_ERR_GENERAL;
   }

   retrieveAmSessionIdFromDbusAct(act, amSessionId);

   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_EcnrGetVersionResult(amSessionId, amResult,
         versionString, versionComment, versionNumber);
   amIfMessage->traceMessage();
   (void)AmMainController::getInstance().handleAmIfMessage(amIfMessage);
}

void EcnrWrapper::mapDbusActToAmSessionId(const act_t dbusAct, const AmSessionId amSessionId)
{
   std::map<act_t, AmSessionId>::iterator it = _actToSessionIdMap.find(dbusAct);
   if(it != _actToSessionIdMap.end())
   {
      //this should never happen
      ETG_TRACE_ERR(("Received DBus ACT is already available in the map"));
      //NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      _actToSessionIdMap.insert(std::pair<act_t, AmSessionId>(dbusAct, amSessionId));
   }
}

void EcnrWrapper::retrieveAmSessionIdFromDbusAct(const act_t dbusAct, AmSessionId& amSessionId)
{
   std::map<act_t, AmSessionId>::iterator it = _actToSessionIdMap.find(dbusAct);
   if(it != _actToSessionIdMap.end())
   {
      amSessionId = it->second;

      //clear the dbus act_t once the required response is received
      _actToSessionIdMap.erase(it);
   }
   else
   {
      //this should never happen
      ETG_TRACE_ERR(("Received DBus ACT is not available in the map"));
      //NORMAL_M_ASSERT_ALWAYS();
   }
}

void EcnrWrapper::clearDbusActToAmSessionIdMap()
{
   _actToSessionIdMap.clear();
}

} //pmaudiomanager
