/**
 * @file EcnrRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for Ecnr DBUS Proxy
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "EcnrRequestIf.h"
#include "IEcnrCallBackIf.h"
#include "CcDbusIfTypes.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EcnrRequestIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

using namespace ::org::bosch::ecnr::service;
using namespace ::asf::core;

namespace pmaudiomanager {

EcnrRequestIf::EcnrRequestIf() : IEcnrRequestIf(false)
{
   ETG_TRACE_USR4(("EcnrRequestIf(): Constructor"));

   _callBackIf = nullptr;
}

EcnrRequestIf::~EcnrRequestIf()
{
   ETG_TRACE_USR4(("EcnrRequestIf(): Destructor"));

   _callBackIf = nullptr;
}

void EcnrRequestIf::createProxy(void)
{
   ETG_TRACE_USR4(("createProxy(): Ecnr"));

   if(nullptr != _callBackIf)
   {
      _serviceProxy = ServiceProxy::createProxy("ecnrServicePort", *this);
   }
}

void EcnrRequestIf::destroyProxy(void)
{
   ETG_TRACE_USR4(("destroyProxy(): Ecnr"));

   if(nullptr != _serviceProxy.get())
   {
      _serviceProxy.reset();
   }
}

void EcnrRequestIf::setCallbackIf(IEcnrCallBackIf* callBackIf)
{
   _callBackIf = callBackIf;

   createProxy();
}

void EcnrRequestIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("onAvailable(): Ecnr"));

   if((nullptr != _serviceProxy.get()) && (proxy == _serviceProxy))
   {
      if(nullptr != _callBackIf)
      {
         _callBackIf->onServiceProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
               _serviceProxy->getDBusObjectPath());

         // nothing to register
      }

      _serviceAvailability = true;
   }
}

void EcnrRequestIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("onUnavailable(): Ecnr"));

   if((nullptr != _serviceProxy.get()) && (proxy == _serviceProxy))
   {
      if(nullptr != _callBackIf)
      {
         _callBackIf->onServiceProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(),
               _serviceProxy->getDBusObjectPath());
      }

      // nothing to deregister
      _serviceAvailability = false;
   }
}

act_t EcnrRequestIf::sendEcnrInitializeRequest(uint8 ecnrAppId, uint32 ecnrConfigurationId)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_serviceAvailability && (nullptr != _callBackIf) && (nullptr != _serviceProxy.get()))
   {
      act = _serviceProxy->sendEcnrInitializeRequest(*_callBackIf, ecnrAppId, ecnrConfigurationId);
   }

   return act;
}

act_t EcnrRequestIf::sendEcnrDestroyRequest(uint8 ecnrAppId)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_serviceAvailability && (nullptr != _callBackIf) && (nullptr != _serviceProxy.get()))
   {
      act = _serviceProxy->sendEcnrDestroyRequest(*_callBackIf, ecnrAppId);
   }

   return act;
}

act_t EcnrRequestIf::sendEcnrSetConfigurationRequest(uint8 ecnrAppId, uint32 ecnrConfigurationId)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_serviceAvailability && (nullptr != _callBackIf) && (nullptr != _serviceProxy.get()))
   {
      act = _serviceProxy->sendEcnrSetConfigurationRequest(*_callBackIf, ecnrAppId, ecnrConfigurationId);
   }

   return act;
}

act_t EcnrRequestIf::sendEcnrStartAudioRequest(uint8 ecnrAppId)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_serviceAvailability && (nullptr != _callBackIf) && (nullptr != _serviceProxy.get()))
   {
      act = _serviceProxy->sendEcnrStartAudioRequest(*_callBackIf, ecnrAppId);
   }

   return act;
}

act_t EcnrRequestIf::sendEcnrStopAudioRequest(uint8 ecnrAppId)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_serviceAvailability && (nullptr != _callBackIf) && (nullptr != _serviceProxy.get()))
   {
      act = _serviceProxy->sendEcnrStopAudioRequest(*_callBackIf, ecnrAppId);
   }

   return act;
}

act_t EcnrRequestIf::sendEcnrSetSendMuteSwitchRequest(uint8 ecnrAppId, uint8 ecnrSendMuteSwitch)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_serviceAvailability && (nullptr != _callBackIf) && (nullptr != _serviceProxy.get()))
   {
      act = _serviceProxy->sendEcnrSetSendMuteSwitchRequest(*_callBackIf, ecnrAppId, ecnrSendMuteSwitch);
   }

   return act;
}

act_t EcnrRequestIf::sendEcnrGetVersionRequest(void)
{
   act_t act = ::ccdbusif::DEFAULT_ACT;

   if(_serviceAvailability && (nullptr != _callBackIf) && (nullptr != _serviceProxy.get()))
   {
      act = _serviceProxy->sendEcnrGetVersionRequest(*_callBackIf);
   }

   return act;
}

} //pmaudiomanager
