/**
 * @file IEcnrRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Interface definition for Ecnr DBUS Proxy
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#ifndef IEcnrRequestIf_h
#define IEcnrRequestIf_h

#include "asf/core/Types.h"

namespace pmaudiomanager {

// class forward declarations
class IEcnrCallBackIf;

/**
 * Ecnr Request interface class
 */
class IEcnrRequestIf
{
public:
   /**
    * Destructor of IEcnrRequestIf class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~IEcnrRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callBackIf: callback handler
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setCallbackIf(IEcnrCallBackIf* callBackIf) = 0;

   /**
    * Send EcnrInitialize Request
    *
    * @param[in] ecnrAppId: application id
    * @param[in] ecnrConfigurationId: configuration id
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEcnrInitializeRequest(uint8 ecnrAppId, uint32 ecnrConfigurationId) = 0;

   /**
    * Send EcnrDestroy Request
    *
    * @param[in] ecnrAppId: application id
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEcnrDestroyRequest(uint8 ecnrAppId) = 0;

   /**
    * Send EcnrSetConfiguration Request
    *
    * @param[in] ecnrAppId: application id
    * @param[in] ecnrConfigurationId: configuration id
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEcnrSetConfigurationRequest(uint8 ecnrAppId, uint32 ecnrConfigurationId) = 0;

   /**
    * Send EcnrStartAudio Request
    *
    * @param[in] ecnrAppId: application id
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEcnrStartAudioRequest(uint8 ecnrAppId) = 0;

   /**
    * Send EcnrStopAudio Request
    *
    * @param[in] ecnrAppId: application id
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEcnrStopAudioRequest(uint8 ecnrAppId) = 0;

   /**
    * Send EcnrSetSendMuteSwitch Request
    *
    * @param[in] ecnrAppId: application id
    * @param[in] ecnrSendMuteSwitch: mute switch value
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEcnrSetSendMuteSwitchRequest(uint8 ecnrAppId, uint8 ecnrSendMuteSwitch) = 0;

   /**
    * Send EcnrGetVersion Request
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return asynchronous completion token
    */
   virtual act_t sendEcnrGetVersionRequest(void) = 0;

protected:
   /**
    * Parameterized constructor of IEcnrRequestIf
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IEcnrRequestIf(bool serviceAvailability){_serviceAvailability = serviceAvailability;}

   bool _serviceAvailability;
};

} //pmaudiomanager

#endif // IEcnrRequestIf_h
