/**
 * @file EcnrWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file used to request the EcNr methods and response from EcNr is forwarded to AmMainController
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#ifndef EcnrWrapper_h
#define EcnrWrapper_h

#include "PmAudioManagerIfTypes.h"
#include "EcnrTypes.h"
#include "CcDbusIfTypes.h"

namespace pmaudiomanager {

// class forward declarations
class IEcnrCallBackIf;
class IEcnrRequestIf;

/**
 * Ecnr wrapper class.
 */
class EcnrWrapper final
{
public:
   /**
    * Constructor of EcnrWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   EcnrWrapper();

   /**
    * Destructor of EcnrWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~EcnrWrapper();

   /**
    * Copy Constructor of EcnrWrapper class
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   EcnrWrapper(IN const EcnrWrapper& ref);

   /**
    * Assignment Operator of EcnrWrapper class
    *
    * @param[in] ref: reference to copy
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   EcnrWrapper& operator=(IN const EcnrWrapper& ref);

   /**
    * Initialize
    *
    * @param[in] amSessionId - am session id
    * @param[in] appId: application id
    * @param[in] configurationId: configuration id
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrErrorCode: ECNR_OK/ECNR_ERROR.
    */
   EcnrErrorCode initialize(IN const AmSessionId amSessionId, IN const EcnrAppId appId,
         IN const EcnrConfigurationId configurationId);

   /**
    * Initialize ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from Ecnr service
    *                    ECNR_REQ_FAILED - error received from Ecnr service due to any reason
    * @param[in] act : asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void initializeResultCB(IN const EcnrRequestResult result, IN const act_t act);

   /**
    * Destroy
    *
    * @param[in] amSessionId - am session id
    * @param[in] appId: application id
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrErrorCode: ECNR_OK/ECNR_ERROR.
    */
   EcnrErrorCode destroy(IN const AmSessionId amSessionId, IN const EcnrAppId appId);

   /**
    * Destroy ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from Ecnr service
    *                    ECNR_REQ_FAILED - error received from Ecnr service due to any reason
    * @param[in] act : asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void destroyResultCB(IN const EcnrRequestResult result, IN const act_t act);

   /**
    * SetConfiguration
    *
    * @param[in] amSessionId - am session id
    * @param[in] appId: application id
    * @param[in] configurationId: configuration id
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrErrorCode: ECNR_OK/ECNR_ERROR.
    */
   EcnrErrorCode setConfiguration(IN const AmSessionId amSessionId, IN const EcnrAppId appId,
         IN const EcnrConfigurationId configurationId);

   /**
    * SetConfiguration ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from Ecnr service
    *                    ECNR_REQ_FAILED - error received from Ecnr service due to any reason
    * @param[in] act : asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setConfigurationResultCB(IN const EcnrRequestResult result, IN const act_t act);

   /**
    * StartAudio
    *
    * @param[in] amSessionId - am session id
    * @param[in] appId: application id
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrErrorCode: ECNR_OK/ECNR_ERROR.
    */
   EcnrErrorCode startAudio(IN const AmSessionId amSessionId, IN const EcnrAppId appId);

   /**
    * StartAudio ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from Ecnr service
    *                    ECNR_REQ_FAILED - error received from Ecnr service due to any reason
    * @param[in] act : asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void startAudioResultCB(IN const EcnrRequestResult result, IN const act_t act);

   /**
    * StopAudio
    *
    * @param[in] amSessionId - am session id
    * @param[in] appId: application id
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrErrorCode: ECNR_OK/ECNR_ERROR.
    */
   EcnrErrorCode stopAudio(IN const AmSessionId amSessionId, IN const EcnrAppId appId);

   /**
    * StopAudio ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from Ecnr service
    *                    ECNR_REQ_FAILED - error received from Ecnr service due to any reason
    * @param[in] act : asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void stopAudioResultCB(IN const EcnrRequestResult result, IN const act_t act);

   /**
    * SetSendMuteSwitch
    *
    * @param[in] amSessionId - am session id
    * @param[in] appId: application id
    * @param[in] sendMuteSwitch: mute switch value
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrErrorCode: ECNR_OK/ECNR_ERROR.
    */
   EcnrErrorCode setSendMuteSwitch(IN const AmSessionId amSessionId, IN const EcnrAppId appId,
         IN const EcnrSendMuteSwitch sendMuteSwitch);

   /**
    * SetSendMuteSwitch ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from Ecnr service
    *                    ECNR_REQ_FAILED  - error received from Ecnr service due to any reason
    * @param[in] act : asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setSendMuteSwitchResultCB(IN const EcnrRequestResult result, IN const act_t act);

   /**
    * GetVersion
    *
    * @param[in] amSessionId - am session id
    * @param[out]
    * @param[in,out]
    *
    * @return EcnrErrorCode: ECNR_OK/ECNR_ERROR.
    */
   EcnrErrorCode getVersion(IN const AmSessionId amSessionId);

   /**
    * GetVersion ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from Ecnr service
    *                    ECNR_REQ_FAILED  - error received from Ecnr service due to any reason
    * @param[in] versionNumber:  version number
    * @param[in] versionString:  version string
    * @param[in] versionComment: version comment
    * @param[in] act : asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void getVersionResultCB(IN const EcnrRequestResult result, IN const VersionNumber& versionNumber,
         IN const Version& versionString, IN const VersionComment& versionComment, IN const act_t act);

private:
   /**
    * This method is used to store the mapping between dbus act and AmSessionId
    *
    * @param[in] dbusAct: dbus asynchronous completion token
    * @param[in] amSessionId: am session id
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void mapDbusActToAmSessionId(IN const act_t dbusAct, IN const AmSessionId amSessionId);

   /**
    * This method is used to retrieve the AmSessionId from dbus act
    *
    * @param[in] dbusAct: dbus asynchronous completion token
    * @param[out] amSessionId: am session id
    * @param[in,out]
    *
    * @return void
    */
   void retrieveAmSessionIdFromDbusAct(IN const act_t dbusAct, OUT AmSessionId& amSessionId);

   /**
    * This method is used to clear the mapping between dbus act and AmSessionId
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void clearDbusActToAmSessionIdMap();

   IEcnrCallBackIf* _ecnrCallBackIf; /**< pointer to Ecnr Callback */
   IEcnrRequestIf*  _ecnrRequestIf;  /**< pointer to Ecnr Request */
   std::map<act_t, AmSessionId> _actToSessionIdMap; /**< map between act and AmSessionId */
};

} //pmaudiomanager

#endif // EcnrWrapper_h
