/**
 * @file GStreamerPlayer.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Class that enhances the playing of audio files using g-streamer player
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "GStreamerPlayer.h"
#include "AudioPlayerWrapper.h"
#include "FileUtils.h"
#include <time.h>
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/GStreamerPlayer.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

#define ALSA_ALIGNMENT_THRESHOLD_NSEC  (120000000)
#define PTHREAD_RT_PRIO                30
#define PLAY_FILE_SCHEME               "file://"
#define PLAY_ERROR_MESSAGE             "Error In Playing The Audio File"
#define STOP_PLAYBACK_ERROR_MESSAGE    "Error In Stopping The Audio File"

namespace pmaudiomanager
{

AudioPlayerWrapper* GStreamerPlayer::_audioPlayerWrapper = nullptr;
std::stringstream GStreamerPlayer::_sinkDeviceName;
std::stringstream GStreamerPlayer::_sourceDeviceName;
GMainLoop* GStreamerPlayer::_gstreamerMainLoop = nullptr;
PlayCount GStreamerPlayer::_filePlayCount = PLAY_COUNT_DEFAULT;
bool GStreamerPlayer::_playStarted = false;
bool GStreamerPlayer::_waitforCondSignal = false;
bool GStreamerPlayer::_setThreadAttributes = false;
AmSessionId GStreamerPlayer::_sessionId = AM_SESSION_ID_DEFAULT;
ToneType GStreamerPlayer::_toneType = RING_TONE;

pthread_mutex_t playerThreadMutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t threadWaitSignal = PTHREAD_COND_INITIALIZER;
pthread_attr_t playerThreadAttr;

GStreamerPlayer::GStreamerPlayer(AudioPlayerWrapper* audioPlayerWrapper,
      const AmSessionId sessionId, const ToneType toneType)
{
   _audioPlayerWrapper = audioPlayerWrapper;
   _setThreadAttributes = false;
   pthread_cond_init(&threadWaitSignal, nullptr);
   pthread_mutex_init(&playerThreadMutex, nullptr);
   setAttributesToGStreamerThread();
   _sessionId = sessionId;
   _toneType = toneType;
}

GStreamerPlayer::~GStreamerPlayer()
{
   pthread_cond_destroy(&threadWaitSignal);
   pthread_mutex_destroy(&playerThreadMutex);
   // clean up memory
   pthread_attr_destroy(&playerThreadAttr);
   _audioPlayerWrapper = nullptr;
}

void GStreamerPlayer::setAttributesToGStreamerThread()
{
   if(_setThreadAttributes == false)
   {
      sched_param param;
      int ret = 0;
      bool status = true;

      // initialized with default attributes
      ret = pthread_attr_init(&playerThreadAttr);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_init(&playerThreadAttr) returned with ret= %d ", ret ));
         status = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_init(&playerThreadAttr) returned ret= %d ", ret ));
      }

      ret = pthread_attr_setschedpolicy(&playerThreadAttr, SCHED_FIFO);  // <-- RT prio policy
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setschedpolicy(&playerThreadAttr, SCHED_FIFO) returned with ret= %d ", ret ));
         status = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setschedpolicy(&playerThreadAttr, SCHED_FIFO) returned ret= %d ", ret ));
      }

      // safe to get existing scheduling param
      ret = 0;
      ret = pthread_attr_getschedparam(&playerThreadAttr, &param);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_getschedparam(&playerThreadAttr, &param) returned with ret= %d ", ret ));
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_getschedparam(&playerThreadAttr, &param) returned ret= %d ", ret ));
      }

      // set the priority; others are unchanged
      param.sched_priority = PTHREAD_RT_PRIO;

      // setting the new scheduling param
      ret = pthread_attr_setschedparam(&playerThreadAttr, &param);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setschedparam(&playerThreadAttr, &param) returned with ret= %d ", ret ));
         status = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setschedparam(&playerThreadAttr, &param) returned ret= %d ", ret ));
      }

      ret = pthread_attr_setinheritsched(&playerThreadAttr, PTHREAD_EXPLICIT_SCHED);  // <-- set RT prio to be inherited
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setinheritsched(&playerThreadAttr, PTHREAD_EXPLICIT_SCHED) returned with ret= %d ", ret ));
         status = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setinheritsched(&playerThreadAttr, PTHREAD_EXPLICIT_SCHED) returned ret= %d ", ret ));
      }

      _setThreadAttributes = status;
   }
   else
   {
      ETG_TRACE_USR1((" SUCCESS:Attributes are already set "));
   }
   // set thread prio: END
}

AmResultCode GStreamerPlayer::setAlsaDeviceRequest(const AudioSink& sink, const AudioSource& /* source */,
   const ToneType toneType)
{
   ETG_TRACE_USR2(("setAlsaDevice: sink device received from ARL: %s", sink.c_str()));

   AmResultCode amResultCode = AM_RESULT_OK;

   if(true == sink.empty())
   {
      ETG_TRACE_ERR(("ALSA deviceName received as Empty"));
      amResultCode = AM_RESULT_ERR_GENERAL;
   }
   else
   {
      _sinkDeviceName.str("");
      _sinkDeviceName << sink;
      _toneType = toneType;
   }

   ETG_TRACE_USR2(("setAlsaDevice: _sinkDeviceName: %s", _sinkDeviceName.str().c_str()));

   return amResultCode;
}

AmResultCode GStreamerPlayer::playAudioFileRequest(const FilePath& filePath, const PlayCount playCount)
{
   AmResultCode amResultCode = AM_RESULT_OK;

   bool playStarted = false;

   const char *filename = nullptr;

   pthread_t threadHandle = 0;

   if (true == com::bosch::pmcommon::checkFileExist(filePath))
   {
      setAttributesToGStreamerThread();

      if(true == _setThreadAttributes)
      {
         pthread_mutex_lock(&playerThreadMutex);

         filename = strdup(filePath.c_str());

         ETG_TRACE_USR2(("playAudioFileRequest: File To Play: %s", filename));

         int result = pthread_create(&threadHandle, &playerThreadAttr, GStreamerPlayer::startPlayback,
               (void*)const_cast<char*>(filename));

         if(0 == result)
         {
            ETG_TRACE_USR2(("playAudioFileRequest: Wait for the conditional signal : %d", _waitforCondSignal));

            struct timespec waitforGstreamerUpdate = {0, 0};
            int condnWaitTimeoutInSec = com::bosch::pmcommon::PmConfiguration::getInstance().getWaitTimeForAudioPlayerResponse();
            waitforGstreamerUpdate.tv_sec = time(nullptr) + condnWaitTimeoutInSec;

            // This condition check is used to overcome the Spurious wakeup of pthread_cond_wait
            while(!_waitforCondSignal)
            {
               // This cond_wait is to wait till the creation of the gstreamer pipeline and playing the ringtone

               // If no response is received within 5 sec from Gstreamer thread, timeout is occurred.
               int retVal = pthread_cond_timedwait(&threadWaitSignal, &playerThreadMutex, &waitforGstreamerUpdate);

               if(ETIMEDOUT == retVal)
               {
                  ETG_TRACE_ERR(("playAudioFileRequest: Timeout occured for pthread_cond_timedwait"));
                  break;
               }
            }

            _waitforCondSignal = false;

            if(true == _playStarted)
            {
               playStarted = true;
               _filePlayCount = (playCount > 0)?(PlayCount)(playCount-1):0;
            }
            else
            {
               ETG_TRACE_ERR((" ERROR: Something went wrong with the Player thread "));
            }
         }
         else
         {
            ETG_TRACE_ERR((" ERROR: Player thread creation failed "));
            free((void*)filename);
         }

         pthread_mutex_unlock(&playerThreadMutex);
      }
      else
      {
         ETG_TRACE_ERR((" ERROR: Attributes are not set properly "));
      }
   }
   else
   {
      ETG_TRACE_ERR((" ERROR: Invalid filePath"));
   }

   if(_audioPlayerWrapper)
   {
      AmResult amResult;
      PlayStatus playStatus;

      if(playStarted == false)
      {
         //TODO: Revisit the error message
         playStatus = PlayStatus::PLAYBACK_ERROR;
         amResultCode = AM_RESULT_ERR_GENERAL;
         amResult._amResultCode = AM_RESULT_ERR_GENERAL;
         amResult._amResultMessage = PLAY_ERROR_MESSAGE;
         stopPlayer();
      }
      else
      {
         playStatus = PlayStatus::PLAYING;
         amResultCode = AM_RESULT_OK;
      }

      _audioPlayerWrapper->playbackStatus(_sessionId, playStatus, amResult, _toneType);
   }
   else
   {
      ETG_TRACE_ERR((" ERROR: _audioPlayerWrapper is null"));
   }

   return amResultCode;
}

void* GStreamerPlayer::startPlayback(void* fileInfo)
{
   ETG_TRACE_USR4(("startPlayback: entered"));

   if (nullptr == fileInfo)
   {
      ETG_TRACE_USR4(("startPlayback:fileInfo is null"));
      sendSignalToWaitingThread(false);
      return nullptr;
   }

   if (_sinkDeviceName.str().empty())
   {
      ETG_TRACE_USR4(("startPlayback:_sinkDeviceName is empty"));
      sendSignalToWaitingThread(false);
      return nullptr;
   }
   else
   {
      ETG_TRACE_USR4(("startPlayback: device :%s", _sinkDeviceName.str().c_str()));
   }

   char* filePath = nullptr;
   guint64 l_iAlsaSinkAlignementThreshold = ALSA_ALIGNMENT_THRESHOLD_NSEC;

   filePath = (char*)fileInfo;

   ETG_TRACE_USR4(("startPlayback: filename:%s", filePath));

   std::stringstream fileUri;

   // Construct File path
   fileUri.str("");
   fileUri << PLAY_FILE_SCHEME;
   fileUri << filePath;

   ETG_TRACE_USR4(("startPlayback: filename to play:%s", fileUri.str().c_str()));

   free(fileInfo);

   // Initialise GStreamer
   GError *gstInitError = nullptr;
   bool initCheck = gst_init_check(nullptr, nullptr, &gstInitError);
   if(false == initCheck)
   {
      ETG_TRACE_ERR(("G-Streamer Initialization failed"));

      if(gstInitError)
      {
         if(gstInitError->message)
            ETG_TRACE_ERR(("gstInitError: Error message: %s", gstInitError->message));

         g_error_free(gstInitError);
      }

      sendSignalToWaitingThread(false);
      return nullptr;
   }

   // set up playbin -- Start
   GstElement *playBinElement = gst_element_factory_make("playbin", "play");
   if (nullptr == playBinElement)
   {
      ETG_TRACE_USR4(("Not able to create playBinElement"));
      sendSignalToWaitingThread(false);
      return nullptr;
   }

   GstElement *sinkElement = gst_element_factory_make("alsasink", "sink");
   if (nullptr == sinkElement)
   {
      ETG_TRACE_USR4(("Not able to create sinkElement"));
      gst_object_unref((GstObject*) playBinElement);
      sendSignalToWaitingThread(false);
      return nullptr;
   }

   g_object_set((GObject*) playBinElement, "uri", fileUri.str().c_str(), nullptr);
   g_object_set((GObject*) sinkElement, "device", _sinkDeviceName.str().c_str(), nullptr);
   g_object_set((GObject*) sinkElement, "alignment-threshold", l_iAlsaSinkAlignementThreshold, nullptr);
   g_object_set((GObject*) playBinElement, "audio-sink", sinkElement, nullptr);

   // set up playbin -- End

   // Adding a message handler to the playbin
   GstBus* gstBus = gst_pipeline_get_bus(GST_PIPELINE_CAST((void*) playBinElement));

   if (!gstBus)
   {
      ETG_TRACE_USR4(("startPlayback: gstBus is null: Unable to create pipeline"));
      gst_object_unref((GstObject*) playBinElement);
      sendSignalToWaitingThread(false);
      return nullptr;
   }

   _gstreamerMainLoop = g_main_loop_new(nullptr, FALSE);

   if(nullptr == _gstreamerMainLoop)
   {
      ETG_TRACE_USR4(("startPlayback: _gstreamerMainLoop is null"));
      gst_object_unref(gstBus);
      gst_object_unref((GstObject*) playBinElement);
      sendSignalToWaitingThread(false);
      return nullptr;
   }

   gst_bus_add_watch(gstBus, gstBusCallBack, playBinElement);

   gst_object_unref(gstBus);

   gst_element_set_state(playBinElement, GST_STATE_PLAYING);

   ETG_TRACE_USR4(("startPlayback: Play audio file"));

   // Main Eventloop is run
   g_main_loop_run(_gstreamerMainLoop);

   ETG_TRACE_USR4(("startPlayBack: Cleaning the player thread"));

   gst_element_set_state(playBinElement, GST_STATE_NULL);

   gst_object_unref((GstObject*) playBinElement);

   if(_gstreamerMainLoop != nullptr)
   {
      g_main_loop_unref (_gstreamerMainLoop);
      _gstreamerMainLoop = nullptr;
   }

   sendSignalToWaitingThread(false);

   ETG_TRACE_USR4(("startPlayback: Player thread cleaned"));

   return nullptr;
}

gboolean GStreamerPlayer::gstBusCallBack(GstBus *bus_local, GstMessage *msg, gpointer data)
{
   (void) bus_local;

   GstElement* playElement = (GstElement*)data;

   if (msg == nullptr)
   {
      ETG_TRACE_USR4(("GstBusCallBack: msg is null"));
      return true;
   }

   switch (GST_MESSAGE_TYPE(msg))
   {
      case GST_MESSAGE_BUFFERING:
      {
         ETG_TRACE_USR4(("gstBusCallBack: GST_MESSAGE_BUFFERING"));
         gint percent = 0;
         gst_message_parse_buffering(msg, &percent);
      }
      break;

      case GST_MESSAGE_EOS:
      {
         // PlayBack finished
         ETG_TRACE_USR4(("gstBusCallBack: GST_MESSAGE_EOS"));

         pthread_mutex_lock(&playerThreadMutex);

         if (0x00 == _filePlayCount)
         {
            // ASF provides the localMessage concept - TBC
            // @Reference: 2.7. Local Message Example in ASF user guide
            PlayStatus playStatus = PlayStatus::STOPPED;
            AmResult amResult;

            if(_audioPlayerWrapper)
            {
               _audioPlayerWrapper->playbackStatus(_sessionId, playStatus, amResult, _toneType);
            }
            else
            {
               ETG_TRACE_ERR((" gstBusCallBack: _audioPlayerWrapper is null"));
            }
         }
         else
         {
            gint64 time_nanoseconds = 0;
            gint64 stop_nanoseconds = (gint64)(GST_CLOCK_TIME_NONE);

            if (!gst_element_seek(playElement, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH, GST_SEEK_TYPE_SET,
                  time_nanoseconds, GST_SEEK_TYPE_NONE, stop_nanoseconds))
            {
               ETG_TRACE_ERR(("gstBusCallBack: Seeking failed"));

               // TODO: send the corresponding Error notification audio event to AM
               // A generic function to be written to update the playback error

               PlayStatus playStatus = PlayStatus::PLAYBACK_ERROR;
               AmResult amResult;
               amResult._amResultCode = AM_RESULT_ERR_GENERAL;
               amResult._amResultMessage = PLAY_ERROR_MESSAGE;

               if(_audioPlayerWrapper)
               {
                  _audioPlayerWrapper->playbackStatus(_sessionId, playStatus, amResult, _toneType);
               }
               else
               {
                  ETG_TRACE_ERR((" gstBusCallBack: _audioPlayerWrapper is null"));
               }

               _filePlayCount = 0;
            }
            else
            {
               if (0xFF != _filePlayCount)
                  _filePlayCount--; // FF indicates continuous play so no need to decrement the play count
            }
         }

         pthread_mutex_unlock(&playerThreadMutex);
      }
      break;

      case GST_MESSAGE_STATE_CHANGED:
      {
         // gst_play_pipeline changed state
         ETG_TRACE_USR4(("gstBusCallBack: GST_MESSAGE_STATE_CHANGED"));
         GstState oldState;
         GstState newState;
         GstState pending;
         gst_message_parse_state_changed(msg, &oldState, &newState, &pending);
         gstPlayerStateChanged(newState, oldState);
      }
      break;

      case GST_MESSAGE_WARNING:
      {
         // Warning received from GST
         ETG_TRACE_USR4(("gstBusCallBack: GST_MESSAGE_WARNING"));

         gchar *debug = nullptr;
         GError *error = nullptr;

         gst_message_parse_warning(msg, &error, &debug);

         if(error)
         {
            if(error->message)
               ETG_TRACE_ERR(("gstBusCallBack: Warning message =%s", error->message));

            g_error_free(error);
         }

         g_free(debug);
      }
      break;

      case GST_MESSAGE_ERROR:
      {
         // Error received from GST
         ETG_TRACE_USR4(("gstBusCallBack: GST_MESSAGE_ERROR"));
         gchar *debug = nullptr;
         GError *error = nullptr;
         gst_message_parse_error(msg, &error, &debug);

         if(error)
         {
            if(error->message)
               ETG_TRACE_ERR(("gstBusCallBack: GST_MESSAGE_ERROR: %s", error->message));

            g_error_free(error);
         }

         g_free(debug);

         // Stop the playback and kill the thread
         sendSignalToWaitingThread(false);
      }
      break;

      default:
      {
         ETG_TRACE_USR2 (("GstBusCallBack :got %s message", GST_MESSAGE_TYPE_NAME (msg)));
      }
      break;
   }

   return true;
}

void GStreamerPlayer::gstPlayerStateChanged(GstState new_state, GstState old_state)
{
   ETG_TRACE_USR2(("gstPlayerStateChanged: %u => %u", ETG_ENUM(TR_GSTSTATE, old_state), ETG_ENUM(TR_GSTSTATE, new_state)));

   if (GST_STATE_PLAYING == new_state)
   {
      if(false == _playStarted)
      {
         // Set playing state
         // Currently the play status update alone is sufficient
         sendSignalToWaitingThread(true);
      }
   }
}

AmResultCode GStreamerPlayer::stopPlaybackRequest()
{
   ETG_TRACE_USR4(("stopPlaybackRequest:entered"));

   AmResultCode amResultCode = stopPlayer();

   if(_audioPlayerWrapper)
   {
      AmResult amResult;
      amResult._amResultCode = amResultCode;
      PlayStatus playStatus;

      if(AM_RESULT_OK != amResultCode)
      {
         amResult._amResultMessage = STOP_PLAYBACK_ERROR_MESSAGE;
         playStatus = PlayStatus::STOP_PLAYBACK_ERROR;
      }
      else
      {
         playStatus = PlayStatus::STOPPED;
      }

      _audioPlayerWrapper->playbackStatus(_sessionId, playStatus, amResult, _toneType);
   }
   else
   {
      ETG_TRACE_ERR((" stopPlaybackRequest: _audioPlayerWrapper is null"));
   }

   ETG_TRACE_USR4(("stopPlaybackRequest: Exit"));

   return amResultCode;
}

AmResultCode GStreamerPlayer::stopPlayer()
{
   ETG_TRACE_USR2(("stopPlayer: Entry"));

   AmResultCode amResultCode = AM_RESULT_ERR_GENERAL;

   pthread_mutex_lock(&playerThreadMutex);

   ETG_TRACE_USR2(("stopPlayer: _waitforCondSignal : %d", _waitforCondSignal));

   if (nullptr != _gstreamerMainLoop)
   {
      if (g_main_loop_is_running(_gstreamerMainLoop))
      {
         g_main_loop_quit(_gstreamerMainLoop);

         struct timespec waitforUpdate = {0, 0};
         int condnWaitTimeoutInSec = com::bosch::pmcommon::PmConfiguration::getInstance().getWaitTimeForAudioPlayerResponse();
         waitforUpdate.tv_sec = time(nullptr) + condnWaitTimeoutInSec;

         // This condition check is used to overcome the Spurious wakeup of pthread_cond_wait
         while(!_waitforCondSignal)
         {
            // This cond_wait for clear all the ref(playbin element, etc,.) created in the startPlayback thread
            int retVal = pthread_cond_timedwait(&threadWaitSignal, &playerThreadMutex, &waitforUpdate);

            if(ETIMEDOUT == retVal)
            {
               ETG_TRACE_ERR(("stopPlaybackRequest: Timeout occured"));
               break;
            }
            else
            {
               amResultCode = AM_RESULT_OK;
            }
         }

         _waitforCondSignal = false;

         ETG_TRACE_USR2(("stopPlayer: Condition signal received"));
      }
      else
      {
         // The execution comes here, when the play itself has not started or already stopped
         ETG_TRACE_ERR(("stopPlayer: g_main loop is not running"));
         _playStarted = false;
         _waitforCondSignal = false;
         _filePlayCount = 0;
      }
   }
   else
   {
      ETG_TRACE_ERR(("stopPlayer:_gstreamerMainLoop is null"));
   }

   pthread_mutex_unlock(&playerThreadMutex);

   return amResultCode;
}

void GStreamerPlayer::sendSignalToWaitingThread(bool responseStatus)
{
   ETG_TRACE_USR4(("sendSignalToWaitingThread:: responseStatus = %d", responseStatus));

   pthread_mutex_lock(&playerThreadMutex);

   _playStarted = responseStatus;

   _waitforCondSignal = true;

   ETG_TRACE_USR4(("_playStarted :%d", _playStarted));

   pthread_cond_signal(&threadWaitSignal);

   pthread_mutex_unlock(&playerThreadMutex);

   return;
}

} // namespace pmaudiomanager
