/**
 * @file AudioPlayerWrapper.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the wrapper functions of AudioPlayer requests and response methods.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "AudioPlayerWrapper.h"
#include "LoopbackController.h"
#include "GStreamerPlayer.h"
#include "AmIfMessageCreator.h"
#include "AmMainController.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AudioPlayerWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager {

   AudioPlayerWrapper::AudioPlayerWrapper()
   {
      ETG_TRACE_USR4(("AudioPlayerWrapper: Constructor"));
      _audioPlayerController = new AudioPlayerController(this);
   }

   AudioPlayerWrapper::~AudioPlayerWrapper()
   {
      ETG_TRACE_USR4(("AudioPlayerWrapper: Destructor"));

      if(nullptr != _audioPlayerController)
      {
         delete _audioPlayerController;
         _audioPlayerController = nullptr;
      }
   }

   AmResultCode AudioPlayerWrapper::playAudioFileRequest(const AmSessionId sessionId,
         const FilePath& filePath, const PlayCount playCount)
   {
      ETG_TRACE_USR4(("AudioPlayerWrapper::playAudioFileRequest: FilePath = \"%80s\", PlayCount = %d",
            filePath.c_str(), playCount));

      AmResultCode result = AM_RESULT_ERR_GENERAL;

      if(nullptr != _audioPlayerController)
      {
         result = _audioPlayerController->playAudioFileRequest(sessionId, filePath, playCount);
      }

      return result;
   }

   AmResultCode AudioPlayerWrapper::stopPlaybackRequest(const AmSessionId sessionId)
   {
      ETG_TRACE_USR4(("AudioPlayerWrapper::stopPlaybackRequest() entered"));

      AmResultCode result = AM_RESULT_ERR_GENERAL;

      if(nullptr != _audioPlayerController)
      {
         result = _audioPlayerController->stopPlaybackRequest(sessionId);
      }

      return result;
   }

   AmResultCode AudioPlayerWrapper::setAlsaDeviceRequest(const AmSessionId sessionId,
         const AudioSink& sink, const AudioSource& source, const ToneType toneType)
   {
      ETG_TRACE_USR4(("AudioPlayerWrapper::setAlsaDeviceRequest: AudioSink = \"%20s\", AudioSource = \"%20s\"",
            sink.c_str(), source.c_str()));

      AmResultCode result = AM_RESULT_ERR_GENERAL;

      if(nullptr != _audioPlayerController)
      {
         result = _audioPlayerController->setAlsaDeviceRequest(sessionId, sink, source, toneType);
      }

      return result;
   }

   void AudioPlayerWrapper::playbackStatus(const AmSessionId sessionId, const PlayStatus playStatus,
         const AmResult& amResult, const ToneType toneType)
   {
      ETG_TRACE_USR4(("AudioPlayerWrapper::playbackStatus"));

      com::bosch::pmcommon::LoopbackData* playbackStatusLBMsg = getNewAmLoopbackMessage_PlaybackStatus(
            com::bosch::pmcommon::PmSubComponentId::PmAudioManager, sessionId, amResult, playStatus, toneType);
      com::bosch::pmcommon::LoopbackController::getInstance().postLoopbackMessage(playbackStatusLBMsg);
   }

   AmResultCode AudioPlayerWrapper::destroyAudioPlayer(const AmSessionId sessionId)
   {
      ETG_TRACE_USR4(("AudioPlayerWrapper::destroyAudioPlayer for session ID: %d", sessionId));

      AmResultCode result = AM_RESULT_ERR_GENERAL;

      if(nullptr != _audioPlayerController)
      {
         result = _audioPlayerController->destroyAudioPlayer(sessionId);
      }

      return result;
   }
} //pmaudiomanager
