/**
 * @file AudioPlayerController.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the wrapper functions of AudioPlayer requests and response methods.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "AudioPlayerController.h"
#include "LoopbackController.h"
#include "GStreamerPlayer.h"
#include "AmIfMessageCreator.h"
#include "AmMainController.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AudioPlayerController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager {

   AudioPlayerController::AudioPlayerController(AudioPlayerWrapper* audioPlayerWrapper)
   {
      ETG_TRACE_USR4(("AudioPlayerController: Constructor"));
      _audioPlayerWrapper = audioPlayerWrapper;
   }

   AudioPlayerController::~AudioPlayerController()
   {
      ETG_TRACE_USR4(("AudioPlayerController: Destructor"));

      _audioPlayerWrapper = nullptr;

      for (auto& iter : _sessionIdAudioPlayerMap)
      {
         delete iter.second;
      }

      _sessionIdAudioPlayerMap.clear();
   }

   AmResultCode AudioPlayerController::playAudioFileRequest(const AmSessionId sessionId,
         const FilePath& filePath, const PlayCount playCount)
   {
      ETG_TRACE_USR4(("AudioPlayerController::playAudioFileRequest: FilePath = \"%80s\", PlayCount = %d",
            filePath.c_str(), playCount));

      AmResultCode result = AM_RESULT_ERR_GENERAL;

      auto iter = _sessionIdAudioPlayerMap.find(sessionId);
      if(iter != _sessionIdAudioPlayerMap.end())
      {
         result = iter->second->playAudioFileRequest(filePath, playCount);
      }

      return result;
   }

   AmResultCode AudioPlayerController::stopPlaybackRequest(const AmSessionId sessionId)
   {
      ETG_TRACE_USR4(("AudioPlayerController::stopPlaybackRequest() entered"));

      AmResultCode result = AM_RESULT_ERR_GENERAL;

      auto iter = _sessionIdAudioPlayerMap.find(sessionId);
      if(iter != _sessionIdAudioPlayerMap.end())
      {
         if(iter->second)
         {
            result = iter->second->stopPlaybackRequest();

            if (result == AM_RESULT_OK)
            {
               delete iter->second;
               _sessionIdAudioPlayerMap.erase(iter);
            }
            else
            {
               ETG_TRACE_ERR(("stopPlaybackRequest() Error in stopPlayBack for sessionId: %d", sessionId));
            }
         }
         else
         {
            ETG_TRACE_ERR(("stopPlaybackRequest() No valid AudioPlayer for sessionId: %d", sessionId));
         }
      }
      else
      {
         ETG_TRACE_ERR(("stopPlaybackRequest() for Invalid sessionId: %d", sessionId));
      }

      return result;
   }

   AmResultCode AudioPlayerController::setAlsaDeviceRequest(const AmSessionId sessionId,
         const AudioSink& sink, const AudioSource& source, const ToneType toneType)
   {
      ETG_TRACE_USR4(("AudioPlayerController::setAlsaDeviceRequest: AudioSink = \"%20s\", AudioSource = \"%20s\"",
            sink.c_str(), source.c_str()));

      AmResultCode result = AM_RESULT_ERR_GENERAL;

      auto iter = _sessionIdAudioPlayerMap.find(sessionId);

      if(iter == _sessionIdAudioPlayerMap.end())
      {
         iter = _sessionIdAudioPlayerMap.emplace_hint(_sessionIdAudioPlayerMap.end(), sessionId,
               new GStreamerPlayer(_audioPlayerWrapper, sessionId, toneType));
      }

      result = iter->second->setAlsaDeviceRequest(sink, source, toneType);
      return result;
   }

   AmResultCode AudioPlayerController::destroyAudioPlayer(IN const AmSessionId sessionId)
   {
      AmResultCode result = AM_RESULT_ERR_GENERAL;

      auto iter = _sessionIdAudioPlayerMap.find(sessionId);

      if(iter != _sessionIdAudioPlayerMap.end())
      {
         if(iter->second)
         {
            delete iter->second;
            _sessionIdAudioPlayerMap.erase(iter);
            result = AM_RESULT_OK;
         }
      }

      return result;
   }
} //pmaudiomanager
