/**
 * @file GStreamerPlayer.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief Class that enhances the playing of audio files using g-streamer player
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This class is used for playing audio files using g-streamer library
 *          The g-streamer version used is 1.0. In the context of PM, vehicle ringtone and waiting mode
 *          tones shall be played using the functions exposed by this class.
 *          Prerequisites for the playing the audio file are:
 *          a. Clients shall set the File path of the audio file.
 *          b. Clients shall inform the number of times the audio file has to be repeated.
 *          c. Clients shall inform the sink device element.
 *             Hint: The ALSA Device name provided by ARL.
 *
 *          The G-streamer has to run in a separate thread context and hence a thread is spawned based on
 *          POSIX library. The priority of this thread also to be set as it was previously done in the legacy
 *          component. PTHREAD-MUTEX is used for synchronizing the ASF component thread and G-streamer thread.
 *
 * @ingroup PmAudioManager
 */

#ifndef GStreamerPlayer_h
#define GStreamerPlayer_h

#include <sstream>
#include "IAudioPlayer.h"

// Required for ALSA: check and remove if not required
#include <alsa/asoundlib.h>
#include <alsa/version.h>

// Required for G-Streamer
#include <gst/gst.h>
#include <glib.h>

// Required for POSIX library
#include <pthread.h>
#include <sched.h>

namespace pmaudiomanager
{

//class forward declaration
class AudioPlayerWrapper;

class GStreamerPlayer final : public IAudioPlayer
{
public:

   /**
    * This method is used to play the given audio file. A thread is spawned to start the play back of
    * the file in the received audio file path. POSIX thread library is used.
    * NOTE: Unfortunately ASF does not support(provide API) setting of the thread priority. Hence POSIX is used.
    * Reference link:- https://rb-tracker.bosch.com/tracker01/browse/ASF-4659
    *
    * @param[in] filePath  : Audio file path
    * @param[in] playCount : Repeat count to play the audio file
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   virtual AmResultCode playAudioFileRequest( IN const FilePath& filePath,
         IN const PlayCount playCount = PLAY_COUNT_DEFAULT) override;

   /**
    * This method is used to stop the playback of the audio file
    *
    * This method would be invoked when:
    *    a. The EOS event was indicated by the G-Streamer and the maximum play count is reached
    *    b. Client has requested to stop the play back
    *
    * param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   virtual AmResultCode stopPlaybackRequest() override;

   /**
    * This method is used to set the ALSA device
    *
    * @param[in] sink   : Audio sink
    * @param[in] source : Audio source
    * @param[in] toneType : Type of tone
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   virtual AmResultCode setAlsaDeviceRequest(IN const AudioSink& sink, IN const AudioSource& source,
   IN const ToneType toneType) override;

   /**
    * Constructor of GStreamerPlayer class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   GStreamerPlayer(AudioPlayerWrapper*, const AmSessionId sessionId, const ToneType toneType);

   /**
    * Destructor of GStreamerPlayer class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~GStreamerPlayer();

   /**
    * This method updates the player state changes to the clients like StateMachine who has triggered
    * the play request.
    *
    * @param[in] newState : New state of the player
    * @param[in] oldState : Previous state of the player
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   static void gstPlayerStateChanged(IN GstState newState, IN GstState oldState);

   /**
    * The state of the GstBus will be informed asynchronously. It will work
    * only if the the GMainLoop is run
    *
    * @param[in] GstBus*    : Bus for which the updates are received
    *           GstMessage : Message from the library like End of stream, Warning message etc.,
    *           gpointer   : The user data which was sent to the player as reference. Typically the pointer to
    *                        the pipeline element is passed to the player so that when the EOS event is triggered
    *                        the element is used to start the playback from the beginning by seeking
    * @param[out]
    * @param[in,out]
    *
    * @return boolean
    */
   static gboolean gstBusCallBack(IN GstBus* bus, IN GstMessage* msg, IN gpointer data);

private:
   /**
    * Thread handler function for the thread spawned in playFile method.
    * Once the thread is instantiated successfully, the thread handler function is executed.
    *    1. A G-streamer pipeline is created through playbin element
    *    2. A bus is attached to watch the pipeline
    *    3. The G-Mainloop is run to receive the events from G-Streamer library
    *
    * @param[in] fileInfo  : File path which was provided in pthread_create()
    * @param[out]
    * @param[in,out]
    *
    * @return void*
    */
   static void* startPlayback(IN void* fileInfo);

   /**
    * Function to set the G-Streamer thread priority if not already set
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   static void setAttributesToGStreamerThread();

   /**
    * Method to signal that an operation has been performed from the GStreamer thread to the waiting thread(ASF
    * component). The entry thread(ASF component) waits for the play / stop ringtone to get processed using
    * the pthread_cond_wait() method. Once the job is performed / started,the G-streamer thread intimates
    * the entry thread using this function. The waiting thread now proceeds after receiving the signal by
    * updating the status to Am.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   static void sendSignalToWaitingThread(IN bool status);

   /**
    * Method to clear the gstreamer pipeline and quit the playback thread.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   static AmResultCode stopPlayer();

   static AudioPlayerWrapper* _audioPlayerWrapper;  /**< Instance of the AudioPlayerWrapper class */
   static std::stringstream   _sinkDeviceName;      /**< Device name of the sink */
   static std::stringstream   _sourceDeviceName;    /**< Device name of the source */
   static GMainLoop*          _gstreamerMainLoop;   /**< Instance of the gmainloop */
   static PlayCount           _filePlayCount;       /**< Number of times the file has to be played */
   static bool                _playStarted;         /**< variable to indicate whether the playing has started */
   static bool                _waitforCondSignal;   /**< Pthread condition variable */
   static bool                _setThreadAttributes; /**< Variable set to avoid setting the thread attributes repeatedly */
   static AmSessionId         _sessionId;           /**< Session Id */
   static ToneType            _toneType;            /**< Type of tone */
}; // class GStreamerPlayer

} // namespace pmaudiomanager

#endif // GStreamerPlayer_h
