/**
 * @file ArlWrapper.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the wrapper functions of ARL requests and response methods.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#include "ArlWrapper.h"
#include "AudioRouteHandler.h"
#include "AmMainController.h"
#include "ahl_fw.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ArlWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager {

ArlWrapper::ArlWrapper()
{
   ETG_TRACE_USR4(("ArlWrapper: Constructor"));

   _audioRouteHandler = nullptr;
}

ArlWrapper::ArlWrapper(ahl_tclBaseOneThreadApp* mainApplication)
{
   ETG_TRACE_USR4(("ArlWrapper: Parameterized Constructor"));

   _audioRouteHandler = new AudioRouteHandler(this, mainApplication);

   if(nullptr != _audioRouteHandler)
   {
      _audioRouteHandler->setSourceAvailability(SRC_PLAYABLE, AVAILABILITY_REASON_NEWMEDIA,
            AM_PHONEAUDIO, SUB_SOURCE_1);

      _audioRouteHandler->setSourceAvailability(SRC_PLAYABLE, AVAILABILITY_REASON_NEWMEDIA,
            AM_WAITINGMODE, SUB_SOURCE_1);

      _audioRouteHandler->setSourceAvailability(SRC_PLAYABLE, AVAILABILITY_REASON_NEWMEDIA,
            AM_VOICERECOGNITION, SUB_SOURCE_1);
   }
}

ArlWrapper::~ArlWrapper()
{
   ETG_TRACE_USR4(("ArlWrapper: Destructor"));

   if(nullptr != _audioRouteHandler)
   {
      delete _audioRouteHandler;
      _audioRouteHandler = nullptr;
   }
}

AmResultCode ArlWrapper::requestAudioRoute(const AudioChannel channel)
{
   ETG_TRACE_USR4(("ArlWrapper::requestAudioRoute: Channel = %u", ETG_CENUM(AudioChannel, channel)));

   AmResultCode amResultCode = AM_RESULT_ERR_GENERAL;

   if(nullptr != _audioRouteHandler)
   {
      amResultCode = _audioRouteHandler->sendAudioRouteRequest(channel);
   }
   else
   {
      ETG_TRACE_ERR(("requestAudioRoute: _audioRouteHandler is null"));
   }

   return amResultCode;
}

bool ArlWrapper::requestAudioRouteCB(const AmResult& amResult, const AudioChannel channel,
      const AudioDeviceList& audioSink, const AudioDeviceList& audioSource)
{
   //TODO logic to retrieve the AmSessionId yet to be implemented
   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_RequestAudioRouteResult(0x01, amResult, channel,
         audioSink, audioSource);

   bool retVal = sendCbMsg2AmMainController(amIfMessage);

   return retVal;
}

AmResultCode ArlWrapper::releaseAudioRoute(AudioChannel channel)
{
   ETG_TRACE_USR4(("ArlWrapper::releaseAudioRoute: Channel = %u", ETG_CENUM(AudioChannel, channel)));

   AmResultCode amResultCode = AM_RESULT_ERR_GENERAL;

   if(nullptr != _audioRouteHandler)
   {
      amResultCode = _audioRouteHandler->sendAudioReleaseRequest(channel);
   }
   else
   {
      ETG_TRACE_ERR(("releaseAudioRoute: _audioRouteHandler is null"));
   }

   return amResultCode;
}

bool ArlWrapper::releaseAudioRouteCB(const AmResult& amResult, const AudioChannel channel)
{
   //TODO logic to retrieve the AmSessionId yet to be implemented
   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_ReleaseAudioRouteResult(0x01, amResult, channel);

   bool retVal = sendCbMsg2AmMainController(amIfMessage);

   return retVal;
}

AmResultCode ArlWrapper::setSourceAvailability(const SrcAvailability srcAvailability,
      const AvailabilityReason availabilityReason, const AudioChannel channel, const SubSource subSource)
{
   AmResultCode amResultCode = AM_RESULT_ERR_GENERAL;

   if(nullptr != _audioRouteHandler)
   {
      amResultCode = _audioRouteHandler->setSourceAvailability(srcAvailability, availabilityReason, channel,
            subSource);
   }
   else
   {
      ETG_TRACE_ERR(("setSourceAvailability: _audioRouteHandler is null"));
   }

   return amResultCode;
}

bool ArlWrapper::onSourceActivityCB(const AudioChannel channel, const SubSource subSource,
      const SrcActivity srcActivity)
{
   //TODO logic to retrieve the AmSessionId yet to be implemented
   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_SourceActivityResult(0x01, channel, subSource,
         srcActivity);

   bool retVal = sendCbMsg2AmMainController(amIfMessage);

   return retVal;
}

AmResultCode ArlWrapper::requestMuteAction(const AudioChannel channel, const MuteState muteAction,
      const MuteReason muteReason)
{
   AmResultCode amResultCode = AM_RESULT_ERR_GENERAL;

   if(nullptr != _audioRouteHandler)
   {
      amResultCode = _audioRouteHandler->setMuteStateAction(channel, muteAction, muteReason);
   }
   else
   {
      ETG_TRACE_ERR(("requestMuteAction: _audioRouteHandler is null"));
   }

   return amResultCode;
}

bool ArlWrapper::requestMuteActionCB(const AmResult& amResult, const AudioChannel channel, const MuteState muteState)
{
   std::shared_ptr<AmIfMessage> amIfMessage = getNewAmIfMessage_RequestMuteActionResult(amResult, channel, muteState);

   bool retVal = sendCbMsg2AmMainController(amIfMessage);

   return retVal;
}

AmResultCode ArlWrapper::sendSourceActivityResult(const AudioChannel channel, const SrcActivity srcActivity)
{
   AmResultCode amResultCode = AM_RESULT_ERR_GENERAL;

   if(nullptr != _audioRouteHandler)
   {
      amResultCode = _audioRouteHandler->sendSourceActivityResult(channel, srcActivity);
   }
   else
   {
      ETG_TRACE_ERR(("sendSourceActivityResult: _audioRouteHandler is null"));
   }

   return amResultCode;
}

bool ArlWrapper::sendCbMsg2AmMainController(IN std::shared_ptr<AmIfMessage>& amIfMessage)
{
   if(nullptr == amIfMessage)
   {
      ETG_TRACE_ERR(("sendCbMsg2AmMainController: amIfMessage is null"));
      return false;
   }

   amIfMessage->traceMessage();

   AmResult amResult = AmMainController::getInstance().handleAmIfMessage(amIfMessage);

   if(AM_RESULT_OK == amResult._amResultCode)
   {
      return true;
   }
   else
   {
      return false;
   }
}

} //pmaudiomanager
