/**
 * @file AudioRouteHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file used to request the ARL methods and response from ARL is forwarded to ArlWrapper
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmAudioManager
 */

#ifndef AudioRouteHandler_h
#define AudioRouteHandler_h

#define ARL_S_IMPORT_INTERFACE_ASF_STR
#include "audio_routing_asf_lib.h"

#include "PmAudioManagerIfTypes.h"
#include "ArlTypes.h"
#include "AsfTimer.h"
#include "AsfTimerCallbackData.h"

namespace pmaudiomanager {

//class forward declarations
class ArlWrapper;

/**
 *
 */
class AudioRouteHandler final : public arl_tclISource_ASF
{
public:
   /**
    * Parameterized Constructor of AudioRouteHandler class
    *
    * @param[in] arlWrapper - pointer of ArlWrapper class
    * @param[in] mainApplication - pointer of PM main application thread
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   AudioRouteHandler(ArlWrapper* arlWrapper, ahl_tclBaseOneThreadApp* mainApplication);

   /**
    * Destructor of AudioRouteHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual ~AudioRouteHandler();

   /**
    * Copy Constructor of AudioRouteHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   AudioRouteHandler(const AudioRouteHandler& other);

   /**
    * Overloading '=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   //lint -e{1720} suppress "assignment operator has non-const parameter", see copy-and-swap idiom
   const AudioRouteHandler& operator=(const AudioRouteHandler&);

   /**
    * This method is used to set the source availability
    *
    * @param[in] srcAvailability - source availability
    * @param[in] availabilityReason - availability reason
    * @param[in] channel - audio channel
    * @param[in] subSource - sub source
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode setSourceAvailability(const SrcAvailability srcAvailability,
         const AvailabilityReason availabilityReason, const AudioChannel channel,
         const SubSource subSource = SUB_SOURCE_DEFAULT);

   /**
    * This method is used to send the source activity result
    *
    * @param[in] channel - audio channel
    * @param[in] srcActivity - source activity
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode sendSourceActivityResult(const AudioChannel channel, const SrcActivity srcActivity);

   /**
    * This method is used to send the audio source allocate request
    *
    * @param[in] channel - audio channel
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode sendAudioRouteRequest(const AudioChannel channel);

   /**
    * This method is used to send the audio source deallocate request
    *
    * @param[in] channel - audio channel
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode sendAudioReleaseRequest(const AudioChannel channel);

   /**
    * This method is used to set the mute state request
    *
    * @param[in] channel - audio channel
    * @param[in] muteAction - mute action
    * @param[in] muteReason - mute reason
    * @param[out]
    * @param[in,out]
    *
    * @return AmResultCode
    */
   AmResultCode setMuteStateAction(const AudioChannel channel, const MuteState muteAction,
            const MuteReason muteReason = MUTE_REASON_OTHER);

   /**
    * Member function overridden from arl_tclISource class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return tBool
    */
   virtual tBool bOnSrcActivity(arl_tenSource enSrcNum, tU16 SubSource, const arl_tSrcActivity& rfcoSrcActivity) override;

   /**
    * Member function overridden from arl_tclISource class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return tBool
    */
   virtual tBool bOnAllocate(arl_tenSource enSrcNum, const arl_tAllocRouteResult& rfcoAllocRoute) override;

   /**
    * Member function overridden from arl_tclISource class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return tBool
    */
   virtual tBool bOnDeAllocate(arl_tenSource enSrcNum) override;

   /**
    * Member function overridden from arl_tclISource class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return tBool
    */
   virtual tBool bOnMuteState(arl_tenSource enSrcNum, arl_tenMuteState enMuteState) override;

   /**
    * This method is used to notify timer elapsed callback
    *
    * @param[in] data              - timer data
    * @param[in] timerCallbackData - timer callback data
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void timerElapsed(com::bosch::pmcommon::TimerData<std::string,ms_t,RepeatCount> data,
         com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData);

protected:
   /**
    * Constructor of AudioRouteHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   AudioRouteHandler();

private:
   /**
    * This method is used to start the source allocation wait timer
    *
    * @param[in] channel - audio channel
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void startTimer(const AudioChannel channel);

   /**
    * This method is used to stop the source allocation wait timer
    *
    * @param[in] channel - audio channel
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void stopTimer(const AudioChannel channel);

   /**
    * This method is used to convert the source to arl source enum type
    *
    * @param[in] channel - audio channel
    * @param[out] enSrcNum - arl source enum value
    * @param[in,out]
    *
    * @return void
    */
   void convertAudioChannelToArlSourceNum(const AudioChannel channel, arl_tenSource& enSrcNum);

   /**
    * This method is used to convert the source availability to arl source availability enum type
    *
    * @param[in] srcAvailability - source availability
    * @param[out] arlSrcAvailability - arl source availability
    * @param[in,out]
    *
    * @return void
    */
   void convertSrcAvailabilityToArlSrcAvailability(const SrcAvailability srcAvailability,
         arl_tenSrcAvailability& arlSrcAvailability);

   /**
    * This method is used to convert the availability reason to arl availability reason enum type
    *
    * @param[in] availabilityReason - availability reason
    * @param[out] arlAvailabilityReason - arl availability reason
    * @param[in,out]
    *
    * @return void
    */
   void convertAvailabilityReasonToArlAvailabilityReason(const AvailabilityReason availabilityReason,
         arl_tenAvailabilityReason& arlAvailabilityReason);

   /**
    * This method is used to convert the source activity to arl source activity enum type
    *
    * @param[in] srcActivity - source activity
    * @param[out] arlSrcActivity - arl source activity
    * @param[in,out]
    *
    * @return void
    */
   void convertSrcActivityToArlSrcActivity(const SrcActivity srcActivity, arl_tSrcActivity& arlSrcActivity);

   /**
    * This method is used to convert the mute action to arl mute action enum type
    *
    * @param[in] muteAction - mute action
    * @param[out] arlMuteAction - arl mute action
    * @param[in,out]
    *
    * @return void
    */
   void convertMuteActionToArlMuteAction(const MuteState muteAction, arl_tenMuteAction& arlMuteAction);

   /**
    * This method is used to convert the mute reason to arl mute reason enum type
    *
    * @param[in] muteReason - mute reason
    * @param[out] arlMuteReason - arl mute reason
    * @param[in,out]
    *
    * @return void
    */
   void convertMuteReasonToArlMuteReason(const MuteReason muteReason,
         audioroutinglib::arl_tclBitmask<arl_enMuteReason>& arlMuteReason);

   /**
    * This method is used to convert the arl source enum to internal channel enum
    *
    * @param[in] enSrcNum - arl source enum
    * @param[out] channel - audio channel
    * @param[in,out]
    *
    * @return void
    */
   void convertArlSourceNumToAudioChannel(const arl_tenSource enSrcNum, AudioChannel& channel);

   /**
    * This method is used to convert the arl source activity enum to internal source activity enum
    *
    * @param[in] arlSrcActivity - arl source activity
    * @param[out] srcActivity - source activity
    * @param[in,out]
    *
    * @return void
    */
   void convertArlSrcActivityToSrcActivity(const arl_tSrcActivity& arlSrcActivity, SrcActivity& srcActivity);

   /**
    * This method is used to convert the arl mute state enum to internal mute state enum
    *
    * @param[in] arlMuteState - arl mute state
    * @param[out] muteState - mute state
    * @param[in,out]
    *
    * @return void
    */
   void convertArlMuteStateToMuteState(const arl_tenMuteState arlMuteState, MuteState& muteState);

   ArlWrapper* _arlWrapper; /**< pointer to ArlWrapper */
   com::bosch::pmcommon::ITimer<AudioRouteHandler, std::string, ms_t, RepeatCount>* _timerPhoneAudio;
   com::bosch::pmcommon::ITimer<AudioRouteHandler, std::string, ms_t, RepeatCount>* _timerVehicleRingtone;
   com::bosch::pmcommon::ITimer<AudioRouteHandler, std::string, ms_t, RepeatCount>* _timerWaitingModeTone;
   com::bosch::pmcommon::ITimer<AudioRouteHandler, std::string, ms_t, RepeatCount>* _timerVoiceRecognition;
};

} //pmaudiomanager

#endif // AudioRouteHandler_h
