/**
 * @file AmIfMessageResult.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @briefThis file contains the definition of all AmIfMessageResult
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the implementation for all PmAudioManager result Messages
 *
 * @ingroup PmAudioManager
 */

#include "AmIfMessageResult.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/AmIfMessageResult.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_AUDIO_MANAGER
#endif
#endif

namespace pmaudiomanager
{
   AmIfMessageResult::AmIfMessageResult(IN const AmIfMsgId messageId, IN const AmSessionId amSessionId,
         IN const AmResult& amResult) : AmIfMessage(messageId, amSessionId), _amResult(amResult)
   {
      ETG_TRACE_USR1(("AmIfMessageResult: [0x%p] creating PM Audio Manager IF message object", (void*) this));
   }

   AmIfMessageResult::AmIfMessageResult(IN const AmIfMessageResult& other) :
         AmIfMessage(other), _amResult(other._amResult)
   {
      ETG_TRACE_USR1(("AmIfMessageResult: [0x%p] copy-creating PM Audio Manager IF message object", (void*) this));
   }

   AmIfMessageResult::~AmIfMessageResult()
   {
      ETG_TRACE_USR1(("AmIfMessageResult: [0x%p] destroying PM Audio Manager IF message object", (void*) this));
   }

   void swap(AmIfMessageResult& first, AmIfMessageResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessage&>(first), static_cast<AmIfMessage&>(second));
      swap(first._amResult, second._amResult);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessageResult& AmIfMessageResult::operator=(AmIfMessageResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessageResult::operator==(const AmIfMessageResult& rhs) const
   {
      return ((AmIfMessage::operator ==(rhs))
            && (this->_amResult == rhs._amResult));
   }

   void AmIfMessageResult::traceMessage(void) const
   {
      AmIfMessage::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] AmResultCode = %u, AmResultMessage = \"%50s\" ",
            (const void*) this, ETG_CENUM(AmResultCode, _amResult._amResultCode), _amResult._amResultMessage.c_str()));
   }

   // *****************************************************************************************************************

   AmIfMessage_EcnrInitializeResult::AmIfMessage_EcnrInitializeResult() :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_INITIALIZE_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrInitializeResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrInitializeResult::AmIfMessage_EcnrInitializeResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) : AmIfMessageResult(AM_IF_MSG_ID_ECNR_INITIALIZE_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrInitializeResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrInitializeResult::AmIfMessage_EcnrInitializeResult(
         IN const AmIfMessage_EcnrInitializeResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrInitializeResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrInitializeResult::~AmIfMessage_EcnrInitializeResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrInitializeResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_EcnrInitializeResult& first, AmIfMessage_EcnrInitializeResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_EcnrInitializeResult& AmIfMessage_EcnrInitializeResult::operator=(
         AmIfMessage_EcnrInitializeResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_EcnrInitializeResult::operator==(const AmIfMessage_EcnrInitializeResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_EcnrInitializeResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_EcnrDestroyResult::AmIfMessage_EcnrDestroyResult() :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_DESTROY_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrDestroyResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrDestroyResult::AmIfMessage_EcnrDestroyResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) : AmIfMessageResult(AM_IF_MSG_ID_ECNR_DESTROY_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrDestroyResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrDestroyResult::AmIfMessage_EcnrDestroyResult(
         IN const AmIfMessage_EcnrDestroyResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrDestroyResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrDestroyResult::~AmIfMessage_EcnrDestroyResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrDestroyResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_EcnrDestroyResult& first, AmIfMessage_EcnrDestroyResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_EcnrDestroyResult& AmIfMessage_EcnrDestroyResult::operator=(
         AmIfMessage_EcnrDestroyResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_EcnrDestroyResult::operator==(const AmIfMessage_EcnrDestroyResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_EcnrDestroyResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_EcnrSetConfigurationResult::AmIfMessage_EcnrSetConfigurationResult() :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_SET_CONFIGURATION_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetConfigurationResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrSetConfigurationResult::AmIfMessage_EcnrSetConfigurationResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_SET_CONFIGURATION_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetConfigurationResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrSetConfigurationResult::AmIfMessage_EcnrSetConfigurationResult(
         IN const AmIfMessage_EcnrSetConfigurationResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetConfigurationResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrSetConfigurationResult::~AmIfMessage_EcnrSetConfigurationResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetConfigurationResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_EcnrSetConfigurationResult& first, AmIfMessage_EcnrSetConfigurationResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_EcnrSetConfigurationResult& AmIfMessage_EcnrSetConfigurationResult::operator=(
         AmIfMessage_EcnrSetConfigurationResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_EcnrSetConfigurationResult::operator==(const AmIfMessage_EcnrSetConfigurationResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_EcnrSetConfigurationResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_EcnrSetSendMuteSwitchResult::AmIfMessage_EcnrSetSendMuteSwitchResult() :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_SET_SEND_MUTE_SWITCH_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetSendMuteSwitchResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrSetSendMuteSwitchResult::AmIfMessage_EcnrSetSendMuteSwitchResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_SET_SEND_MUTE_SWITCH_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetSendMuteSwitchResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrSetSendMuteSwitchResult::AmIfMessage_EcnrSetSendMuteSwitchResult(
         IN const AmIfMessage_EcnrSetSendMuteSwitchResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetSendMuteSwitchResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrSetSendMuteSwitchResult::~AmIfMessage_EcnrSetSendMuteSwitchResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrSetSendMuteSwitchResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_EcnrSetSendMuteSwitchResult& first, AmIfMessage_EcnrSetSendMuteSwitchResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_EcnrSetSendMuteSwitchResult& AmIfMessage_EcnrSetSendMuteSwitchResult::operator=(
         AmIfMessage_EcnrSetSendMuteSwitchResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_EcnrSetSendMuteSwitchResult::operator==(const AmIfMessage_EcnrSetSendMuteSwitchResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_EcnrSetSendMuteSwitchResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_EcnrStartAudioResult::AmIfMessage_EcnrStartAudioResult() :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_START_AUDIO_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStartAudioResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrStartAudioResult::AmIfMessage_EcnrStartAudioResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) : AmIfMessageResult(AM_IF_MSG_ID_ECNR_START_AUDIO_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStartAudioResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrStartAudioResult::AmIfMessage_EcnrStartAudioResult(
         IN const AmIfMessage_EcnrStartAudioResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStartAudioResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrStartAudioResult::~AmIfMessage_EcnrStartAudioResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStartAudioResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_EcnrStartAudioResult& first, AmIfMessage_EcnrStartAudioResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_EcnrStartAudioResult& AmIfMessage_EcnrStartAudioResult::operator=(
         AmIfMessage_EcnrStartAudioResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_EcnrStartAudioResult::operator==(const AmIfMessage_EcnrStartAudioResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_EcnrStartAudioResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_EcnrStopAudioResult::AmIfMessage_EcnrStopAudioResult() :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_STOP_AUDIO_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStopAudioResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrStopAudioResult::AmIfMessage_EcnrStopAudioResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) : AmIfMessageResult(AM_IF_MSG_ID_ECNR_STOP_AUDIO_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStopAudioResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrStopAudioResult::AmIfMessage_EcnrStopAudioResult(
         IN const AmIfMessage_EcnrStopAudioResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStopAudioResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrStopAudioResult::~AmIfMessage_EcnrStopAudioResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrStopAudioResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_EcnrStopAudioResult& first, AmIfMessage_EcnrStopAudioResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_EcnrStopAudioResult& AmIfMessage_EcnrStopAudioResult::operator=(
         AmIfMessage_EcnrStopAudioResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_EcnrStopAudioResult::operator==(const AmIfMessage_EcnrStopAudioResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_EcnrStopAudioResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_EcnrGetVersionResult::AmIfMessage_EcnrGetVersionResult() :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_GET_VERSION_RESULT, AM_SESSION_ID_DEFAULT, AmResult()),
         _version(""), _versionComment(""), _versionNumber()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrGetVersionResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrGetVersionResult::AmIfMessage_EcnrGetVersionResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult, IN const Version& version, IN const VersionComment& versionComment,
         IN const VersionNumber& versionNumber) :
         AmIfMessageResult(AM_IF_MSG_ID_ECNR_GET_VERSION_RESULT, amSessionId, amResult),
         _version(version), _versionComment(versionComment), _versionNumber(versionNumber)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrGetVersionResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrGetVersionResult::AmIfMessage_EcnrGetVersionResult( IN const AmIfMessage_EcnrGetVersionResult& other) :
         AmIfMessageResult(other),
         _version(other._version), _versionComment(other._versionComment), _versionNumber(other._versionNumber)
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrGetVersionResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_EcnrGetVersionResult::~AmIfMessage_EcnrGetVersionResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_EcnrGetVersionResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_EcnrGetVersionResult& first, AmIfMessage_EcnrGetVersionResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
      swap(first._version, second._version);
      swap(first._versionComment, second._versionComment);
      swap(first._versionNumber, second._versionNumber);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_EcnrGetVersionResult& AmIfMessage_EcnrGetVersionResult::operator=(
         AmIfMessage_EcnrGetVersionResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_EcnrGetVersionResult::operator==(const AmIfMessage_EcnrGetVersionResult& rhs) const
   {
      return ((AmIfMessageResult::operator ==(rhs))
            && (this->_version == rhs._version)
            && (this->_versionComment == rhs._versionComment)
            && (this->_versionNumber == rhs._versionNumber));
   }

   void AmIfMessage_EcnrGetVersionResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] Version = \"%50s\", VersionComment = \"%50s\" ",
            (const void*) this, _version.c_str(), _versionComment.c_str()));

      for(size_t index = 0; index < _versionNumber.size(); index++)
      {
         ETG_TRACE_USR4(("traceMessage: VersionNumber = %d", _versionNumber.at(index)));
      }
   }

   // *****************************************************************************************************************

   AmIfMessage_RequestAudioRouteResult::AmIfMessage_RequestAudioRouteResult() :
         AmIfMessageResult(AM_IF_MSG_ID_REQUEST_AUDIO_ROUTE_RESULT, AM_SESSION_ID_DEFAULT, AmResult()),
         _audioChannel(AM_UNKNOWN), _audioSink(), _audioSource()
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestAudioRouteResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_RequestAudioRouteResult::AmIfMessage_RequestAudioRouteResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult, IN const AudioChannel audioChannel, IN const AudioDeviceList& audioSink,
         IN const AudioDeviceList& audioSource) :
         AmIfMessageResult(AM_IF_MSG_ID_REQUEST_AUDIO_ROUTE_RESULT, amSessionId, amResult),
         _audioChannel(audioChannel), _audioSink(audioSink), _audioSource(audioSource)
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestAudioRouteResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_RequestAudioRouteResult::AmIfMessage_RequestAudioRouteResult(
         IN const AmIfMessage_RequestAudioRouteResult& other) : AmIfMessageResult(other),
         _audioChannel(other._audioChannel), _audioSink(other._audioSink), _audioSource(other._audioSource)
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestAudioRouteResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_RequestAudioRouteResult::~AmIfMessage_RequestAudioRouteResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestAudioRouteResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_RequestAudioRouteResult& first, AmIfMessage_RequestAudioRouteResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
      swap(first._audioChannel, second._audioChannel);
      swap(first._audioSink, second._audioSink);
      swap(first._audioSource, second._audioSource);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_RequestAudioRouteResult& AmIfMessage_RequestAudioRouteResult::operator=(
         AmIfMessage_RequestAudioRouteResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_RequestAudioRouteResult::operator==(const AmIfMessage_RequestAudioRouteResult& rhs) const
   {
      return ((AmIfMessageResult::operator ==(rhs))
            && (this->_audioChannel == rhs._audioChannel)
            && (this->_audioSink == rhs._audioSink)
            && (this->_audioSource == rhs._audioSource));
   }

   void AmIfMessage_RequestAudioRouteResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] AudioChannel = %u", (const void*) this,
            ETG_CENUM(AudioChannel, _audioChannel)));

      for(size_t index = 0; index < _audioSink.size(); index++)
      {
         ETG_TRACE_USR4(("traceMessage: AudioSink = \"%50s\" ", _audioSink.at(index).c_str()));
      }

      for(size_t index = 0; index < _audioSource.size(); index++)
      {
         ETG_TRACE_USR4(("traceMessage: AudioSource = \"%50s\" ", _audioSource.at(index).c_str()));
      }
   }

   // *****************************************************************************************************************

   AmIfMessage_ReleaseAudioRouteResult::AmIfMessage_ReleaseAudioRouteResult() :
         AmIfMessageResult(AM_IF_MSG_ID_RELEASE_AUDIO_ROUTE_RESULT, AM_SESSION_ID_DEFAULT, AmResult()),
         _audioChannel(AM_UNKNOWN)
   {
      ETG_TRACE_USR1(("AmIfMessage_ReleaseAudioRouteResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_ReleaseAudioRouteResult::AmIfMessage_ReleaseAudioRouteResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult, IN const AudioChannel audioChannel) :
         AmIfMessageResult(AM_IF_MSG_ID_RELEASE_AUDIO_ROUTE_RESULT, amSessionId, amResult),
         _audioChannel(audioChannel)
   {
      ETG_TRACE_USR1(("AmIfMessage_ReleaseAudioRouteResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_ReleaseAudioRouteResult::AmIfMessage_ReleaseAudioRouteResult(
         IN const AmIfMessage_ReleaseAudioRouteResult& other) : AmIfMessageResult(other),
         _audioChannel(other._audioChannel)
   {
      ETG_TRACE_USR1(("AmIfMessage_ReleaseAudioRouteResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_ReleaseAudioRouteResult::~AmIfMessage_ReleaseAudioRouteResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_ReleaseAudioRouteResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_ReleaseAudioRouteResult& first, AmIfMessage_ReleaseAudioRouteResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
      swap(first._audioChannel, second._audioChannel);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_ReleaseAudioRouteResult& AmIfMessage_ReleaseAudioRouteResult::operator=(
         AmIfMessage_ReleaseAudioRouteResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_ReleaseAudioRouteResult::operator==(const AmIfMessage_ReleaseAudioRouteResult& rhs) const
   {
      return ((AmIfMessageResult::operator ==(rhs))
            && (this->_audioChannel == rhs._audioChannel));
   }

   void AmIfMessage_ReleaseAudioRouteResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] AudioChannel = %u", (const void*) this,
            ETG_CENUM(AudioChannel, _audioChannel)));
   }

   // *****************************************************************************************************************

   AmIfMessage_SourceActivityResult::AmIfMessage_SourceActivityResult() :
         AmIfMessageResult(AM_IF_MSG_ID_SOURCE_ACTIVITY_RESULT, AM_SESSION_ID_DEFAULT, AmResult()),
         _audioChannel(AM_UNKNOWN), _subSource(SUB_SOURCE_DEFAULT), _srcActivity(SRC_ACTIVITY_OFF)
   {
      ETG_TRACE_USR1(("AmIfMessage_SourceActivityResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SourceActivityResult::AmIfMessage_SourceActivityResult(IN const AmSessionId amSessionId,
         IN const AudioChannel audioChannel, IN const SubSource subSource, IN const SrcActivity srcActivity) :
         AmIfMessageResult(AM_IF_MSG_ID_SOURCE_ACTIVITY_RESULT, amSessionId, AmResult()),
         _audioChannel(audioChannel), _subSource(subSource), _srcActivity(srcActivity)
   {
      ETG_TRACE_USR1(("AmIfMessage_SourceActivityResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SourceActivityResult::AmIfMessage_SourceActivityResult(
         IN const AmIfMessage_SourceActivityResult& other) : AmIfMessageResult(other),
         _audioChannel(other._audioChannel), _subSource(other._subSource), _srcActivity(other._srcActivity)
   {
      ETG_TRACE_USR1(("AmIfMessage_SourceActivityResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SourceActivityResult::~AmIfMessage_SourceActivityResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_SourceActivityResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_SourceActivityResult& first, AmIfMessage_SourceActivityResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
      swap(first._audioChannel, second._audioChannel);
      swap(first._subSource, second._subSource);
      swap(first._srcActivity, second._srcActivity);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_SourceActivityResult& AmIfMessage_SourceActivityResult::operator=(
         AmIfMessage_SourceActivityResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_SourceActivityResult::operator==(const AmIfMessage_SourceActivityResult& rhs) const
   {
      return ((AmIfMessageResult::operator ==(rhs))
            && (this->_audioChannel == rhs._audioChannel)
            && (this->_subSource == rhs._subSource)
            && (this->_srcActivity == rhs._srcActivity));
   }

   void AmIfMessage_SourceActivityResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] AudioChannel = %u, SubSource = %d, SrcActivity = %u", (const void*) this,
            ETG_CENUM(AudioChannel, _audioChannel), _subSource, ETG_CENUM(SrcActivity, _srcActivity)));
   }

   // *****************************************************************************************************************

   AmIfMessage_RequestMuteActionResult::AmIfMessage_RequestMuteActionResult() :
         AmIfMessageResult(AM_IF_MSG_ID_REQUEST_MUTE_ACTION_RESULT, AM_SESSION_ID_DEFAULT, AmResult()),
         _audioChannel(AM_UNKNOWN), _muteState(false)
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestMuteActionResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_RequestMuteActionResult::AmIfMessage_RequestMuteActionResult(IN const AmResult& amResult,
         IN const AudioChannel audioChannel, IN const MuteState muteState) :
         AmIfMessageResult(AM_IF_MSG_ID_REQUEST_MUTE_ACTION_RESULT, AM_SESSION_ID_DEFAULT, amResult),
         _audioChannel(audioChannel), _muteState(muteState)
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestMuteActionResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_RequestMuteActionResult::AmIfMessage_RequestMuteActionResult(
         IN const AmIfMessage_RequestMuteActionResult& other) : AmIfMessageResult(other),
         _audioChannel(other._audioChannel), _muteState(other._muteState)
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestMuteActionResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_RequestMuteActionResult::~AmIfMessage_RequestMuteActionResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_RequestMuteActionResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_RequestMuteActionResult& first, AmIfMessage_RequestMuteActionResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
      swap(first._audioChannel, second._audioChannel);
      swap(first._muteState, second._muteState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_RequestMuteActionResult& AmIfMessage_RequestMuteActionResult::operator=(
         AmIfMessage_RequestMuteActionResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_RequestMuteActionResult::operator==(const AmIfMessage_RequestMuteActionResult& rhs) const
   {
      return ((AmIfMessageResult::operator ==(rhs))
            && (this->_audioChannel == rhs._audioChannel)
            && (this->_muteState == rhs._muteState));
   }

   void AmIfMessage_RequestMuteActionResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] AudioChannel = %u, MuteState = %d", (const void*) this,
            ETG_CENUM(AudioChannel, _audioChannel), _muteState));
   }

   // *****************************************************************************************************************

   AmIfMessage_PlaybackStatus::AmIfMessage_PlaybackStatus() :
         AmIfMessageResult(AM_IF_MSG_ID_PLAYBACK_STATUS, AM_SESSION_ID_DEFAULT, AmResult()),
         _playStatus(PlayStatus::NOT_PLAYING), _toneType(RING_TONE)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlaybackStatus: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlaybackStatus::AmIfMessage_PlaybackStatus(IN const AmSessionId amSessionId, IN const AmResult& amResult,
         IN const PlayStatus playStatus, const ToneType toneType) :
         AmIfMessageResult(AM_IF_MSG_ID_PLAYBACK_STATUS, amSessionId, amResult),
         _playStatus(playStatus), _toneType(toneType)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlaybackStatus: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlaybackStatus::AmIfMessage_PlaybackStatus(IN const AmIfMessage_PlaybackStatus& other) :
         AmIfMessageResult(other), _playStatus(other._playStatus), _toneType(other._toneType)
   {
      ETG_TRACE_USR1(("AmIfMessage_PlaybackStatus: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_PlaybackStatus::~AmIfMessage_PlaybackStatus()
   {
      ETG_TRACE_USR1(("AmIfMessage_PlaybackStatus: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_PlaybackStatus& first, AmIfMessage_PlaybackStatus& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
      swap(first._playStatus, second._playStatus);
      swap(first._toneType, second._toneType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_PlaybackStatus& AmIfMessage_PlaybackStatus::operator=(AmIfMessage_PlaybackStatus other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_PlaybackStatus::operator==(const AmIfMessage_PlaybackStatus& rhs) const
   {
      return ((AmIfMessageResult::operator ==(rhs))
            && (this->_playStatus == rhs._playStatus)
            && (this->_toneType == rhs._toneType));
   }

   void AmIfMessage_PlaybackStatus::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] PlayStatus = %u", (const void*) this, ETG_CENUM(PlayStatus, _playStatus)));
   }

   // *****************************************************************************************************************

   AmIfMessage_GetListMainSinkSoundPropertiesResult::AmIfMessage_GetListMainSinkSoundPropertiesResult() :
         AmIfMessageResult(AM_IF_MSG_ID_GAM_GET_LIST_MAIN_SINK_SND_PROP_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_GetListMainSinkSoundPropertiesResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_GetListMainSinkSoundPropertiesResult::AmIfMessage_GetListMainSinkSoundPropertiesResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) : AmIfMessageResult(AM_IF_MSG_ID_GAM_GET_LIST_MAIN_SINK_SND_PROP_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_GetListMainSinkSoundPropertiesResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_GetListMainSinkSoundPropertiesResult::AmIfMessage_GetListMainSinkSoundPropertiesResult(
         IN const AmIfMessage_GetListMainSinkSoundPropertiesResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_GetListMainSinkSoundPropertiesResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_GetListMainSinkSoundPropertiesResult::~AmIfMessage_GetListMainSinkSoundPropertiesResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_GetListMainSinkSoundPropertiesResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_GetListMainSinkSoundPropertiesResult& first, AmIfMessage_GetListMainSinkSoundPropertiesResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_GetListMainSinkSoundPropertiesResult& AmIfMessage_GetListMainSinkSoundPropertiesResult::operator=(
         AmIfMessage_GetListMainSinkSoundPropertiesResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_GetListMainSinkSoundPropertiesResult::operator==(const AmIfMessage_GetListMainSinkSoundPropertiesResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_GetListMainSinkSoundPropertiesResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_SetMainSinkSoundPropertyResult::AmIfMessage_SetMainSinkSoundPropertyResult() :
         AmIfMessageResult(AM_IF_MSG_ID_GAM_SET_MAIN_SINK_SND_PROP_RESULT, AM_SESSION_ID_DEFAULT, AmResult())
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMainSinkSoundPropertyResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetMainSinkSoundPropertyResult::AmIfMessage_SetMainSinkSoundPropertyResult(IN const AmSessionId amSessionId,
         IN const AmResult& amResult) : AmIfMessageResult(AM_IF_MSG_ID_GAM_SET_MAIN_SINK_SND_PROP_RESULT, amSessionId, amResult)
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMainSinkSoundPropertyResult: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetMainSinkSoundPropertyResult::AmIfMessage_SetMainSinkSoundPropertyResult(
         IN const AmIfMessage_SetMainSinkSoundPropertyResult& other) : AmIfMessageResult(other)
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMainSinkSoundPropertyResult: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_SetMainSinkSoundPropertyResult::~AmIfMessage_SetMainSinkSoundPropertyResult()
   {
      ETG_TRACE_USR1(("AmIfMessage_SetMainSinkSoundPropertyResult: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_SetMainSinkSoundPropertyResult& first, AmIfMessage_SetMainSinkSoundPropertyResult& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_SetMainSinkSoundPropertyResult& AmIfMessage_SetMainSinkSoundPropertyResult::operator=(
         AmIfMessage_SetMainSinkSoundPropertyResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_SetMainSinkSoundPropertyResult::operator==(const AmIfMessage_SetMainSinkSoundPropertyResult& rhs) const
   {
      return (AmIfMessageResult::operator ==(rhs));
   }

   void AmIfMessage_SetMainSinkSoundPropertyResult::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();
   }

   // *****************************************************************************************************************

   AmIfMessage_UpdateAudioSinkVolumeList::AmIfMessage_UpdateAudioSinkVolumeList() :
         AmIfMessageResult(AM_IF_MSG_ID_GAM_UPDATE_AUDIO_SINK_VOL_LIST, AM_SESSION_ID_DEFAULT, AmResult()),
         _sinkId(0), _propertyMap({})
   {
      ETG_TRACE_USR1(("AmIfMessage_UpdateAudioSinkVolumeList: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_UpdateAudioSinkVolumeList::AmIfMessage_UpdateAudioSinkVolumeList(IN const AmSessionId amSessionId,
         IN const AmResult& amResult, IN const GeniviAMSinkId& sinkId, IN const AmPropertyType2VolumeMap& propertymap) :
         AmIfMessageResult(AM_IF_MSG_ID_GAM_UPDATE_AUDIO_SINK_VOL_LIST, amSessionId, amResult),
         _sinkId(sinkId), _propertyMap(propertymap)
   {
      ETG_TRACE_USR1(("AmIfMessage_UpdateAudioSinkVolumeList: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_UpdateAudioSinkVolumeList::AmIfMessage_UpdateAudioSinkVolumeList( IN const AmIfMessage_UpdateAudioSinkVolumeList& other) :
         AmIfMessageResult(other),
         _sinkId(other._sinkId), _propertyMap(other._propertyMap)
   {
      ETG_TRACE_USR1(("AmIfMessage_UpdateAudioSinkVolumeList: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmIfMessage_UpdateAudioSinkVolumeList::~AmIfMessage_UpdateAudioSinkVolumeList()
   {
      ETG_TRACE_USR1(("AmIfMessage_UpdateAudioSinkVolumeList: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmIfMessage_UpdateAudioSinkVolumeList& first, AmIfMessage_UpdateAudioSinkVolumeList& second)
   {
      using std::swap;

      swap(static_cast<AmIfMessageResult&>(first), static_cast<AmIfMessageResult&>(second));
      swap(first._sinkId, second._sinkId);
      swap(first._propertyMap, second._propertyMap);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmIfMessage_UpdateAudioSinkVolumeList& AmIfMessage_UpdateAudioSinkVolumeList::operator=(
         AmIfMessage_UpdateAudioSinkVolumeList other)
   {
      swap(*this, other);

      return *this;
   }

   bool AmIfMessage_UpdateAudioSinkVolumeList::operator==(const AmIfMessage_UpdateAudioSinkVolumeList& rhs) const
   {
      return ((AmIfMessageResult::operator ==(rhs))
            && (this->_sinkId == rhs._sinkId)
            && (this->_propertyMap == rhs._propertyMap));
   }

   void AmIfMessage_UpdateAudioSinkVolumeList::traceMessage(void) const
   {
      AmIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] Sink Id = \"%50d\"",
            (const void*) this, _sinkId));
   }

   // *****************************************************************************************************************

   AmLoopbackMessage_PlaybackStatus::AmLoopbackMessage_PlaybackStatus() :
      com::bosch::pmcommon::LoopbackData(com::bosch::pmcommon::PmSubComponentId::PmAudioManager, 0x00),
      _playbackStatusMsg(AmIfMessage_PlaybackStatus())
   {
      ETG_TRACE_USR1(("AmLoopbackMessage_PlaybackStatus: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmLoopbackMessage_PlaybackStatus::AmLoopbackMessage_PlaybackStatus(
         IN const com::bosch::pmcommon::PmSubComponentId subComponentId, IN const AmSessionId amSessionId,
         IN const AmResult& amResult, IN const PlayStatus playStatus, IN const ToneType toneType) :
         com::bosch::pmcommon::LoopbackData(subComponentId, AM_IF_MSG_ID_PLAYBACK_STATUS),
         _playbackStatusMsg(AmIfMessage_PlaybackStatus(amSessionId, amResult, playStatus, toneType))
   {
      ETG_TRACE_USR1(("AmLoopbackMessage_PlaybackStatus: [0x%p] creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmLoopbackMessage_PlaybackStatus::AmLoopbackMessage_PlaybackStatus(IN const AmLoopbackMessage_PlaybackStatus& other) :
         com::bosch::pmcommon::LoopbackData(other), _playbackStatusMsg(other._playbackStatusMsg)
   {
      ETG_TRACE_USR1(("AmLoopbackMessage_PlaybackStatus: [0x%p] copy-creating PM Audio Manager IF message object",
            (void*) this));
   }

   AmLoopbackMessage_PlaybackStatus::~AmLoopbackMessage_PlaybackStatus()
   {
      ETG_TRACE_USR1(("AmLoopbackMessage_PlaybackStatus: [0x%p] destroying PM Audio Manager IF message object",
            (void*) this));
   }

   void swap(AmLoopbackMessage_PlaybackStatus& first, AmLoopbackMessage_PlaybackStatus& second)
   {
      using std::swap;

      swap(static_cast<com::bosch::pmcommon::LoopbackData&>(first),
            static_cast<com::bosch::pmcommon::LoopbackData&>(second));
      swap(first._playbackStatusMsg, second._playbackStatusMsg);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AmLoopbackMessage_PlaybackStatus& AmLoopbackMessage_PlaybackStatus::operator=(AmLoopbackMessage_PlaybackStatus other)
   {
      swap(*this, other);
      return *this;
   }

   bool AmLoopbackMessage_PlaybackStatus::operator==(const AmLoopbackMessage_PlaybackStatus& rhs) const
   {
      return ((LoopbackData::operator==(rhs))
            && (this->_playbackStatusMsg == rhs._playbackStatusMsg));
   }

   void AmLoopbackMessage_PlaybackStatus::traceMessage(void) const
   {
      LoopbackData::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] PmSubComponentId = %d, LoopbackMsgId = %d", (const void*) this,
            _pmSubComponentId, _loopbackMsgId));
   }

   // *****************************************************************************************************************

} // namespace pmaudiomanager
