/* ETG definitions */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Timer.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#endif
#endif
#include "FunctionTracer.h"
#include "Timer.h"

#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/syscall.h>
#include <sys/prctl.h>
#include <unistd.h>
#include "Dispatcher.h"
#include "TraceDefinitions.h"
#include "FwAssert.h"
#include "FwISmTimerCallback.h"

#if SIGACTION_THREAD

/* global to all instances of timer class */
static pid_t mSigactionThreadID;
static pthread_attr_t mThreadAttr;
static pthread_t mThreadHandle;
#define DEFAULT_STACKSIZE 10000000
static int mThreadReferenceCount = 0;
static int mEnabledTimerClass = TIMER_CLASS_ALL; /* all timer classes enabled */

void *Timer::SigactionThread(void *context)
{
    Timer *timer = (Timer *)context;

    pthread_setcancelstate(PTHREAD_CANCEL_ENABLE ,NULL);
    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS ,NULL);

    sigemptyset(&timer->mSigAction.sa_mask);

    timer->mSigAction.sa_sigaction = Timer::DefaultCallback;
    timer->mSigAction.sa_flags = SA_SIGINFO;

    sigaction(SIGUSR1, &timer->mSigAction, NULL);

    mSigactionThreadID = (pid_t)syscall(SYS_gettid);

    ETG_TRACE_USR4(("SigactionThread, tid=%x", mSigactionThreadID));

    /* endless: processing of incoming signals */
    while(1) sleep(1000);

    return NULL;
}

#endif

Timer::Timer()
{
   memset(&mSigAction, 0, sizeof(mSigAction));

#if SIGACTION_THREAD

    /* at first startup... */
    if (mThreadReferenceCount == 0) {
        tResult res;

        /* start the signal receiver */
        mSigactionThreadID = 0;
        FW_FATAL_ASSERT(pthread_attr_init(&mThreadAttr) == 0);

        /* set the size of the stack. Default stack size is too small. */
        pthread_attr_setstacksize(&mThreadAttr, DEFAULT_STACKSIZE);

        /* create the thread */
        res = pthread_create(&mThreadHandle, &mThreadAttr, &Timer::SigactionThread, (void *)this);
        if (res) {
            ETG_TRACE_FATAL(("pthread_create: err=%d", res));
            FW_FATAL_ASSERT(res == 0);
        }

        /* wait for tid set */
        while(!mSigactionThreadID) usleep(2000);

        /* timer handling thread created, reference is one */
        mThreadReferenceCount = 1;

    /* further instance of Timer class: just increase reference, timer handling thread already running */
    } else {
        mThreadReferenceCount++;
    }

#else
    sigemptyset(&mSigAction.sa_mask);

    mSigAction.sa_sigaction = Timer::DefaultCallback;
    mSigAction.sa_flags = SA_SIGINFO;

    sigaction(SIGUSR1, &mSigAction, NULL);
#endif

    /* this timer is in general class until it is changed by: SetTimerClass() */
    mTimerClass = TIMER_CLASS_OTHERS;
}

Timer::~Timer()
{
#if SIGACTION_THREAD

    /* if the last reference to the timer handling thread: kill him */
    if (mThreadReferenceCount == 1) {
        int res=0;

        /* cancel the timer thread */
        res=pthread_cancel(mThreadHandle);

        /* waiting for thread to join */
        if(res==0)
        {
            pthread_join(mThreadHandle,NULL);
        }

        /* no further thread is running. At next Constructor it will be recreated again */
        mThreadReferenceCount = 0;

    /* not the last reference, decrement reference counter */
    } else {
        mThreadReferenceCount--;
    }
#endif

    /* delete all timers */
    CancelAllTimers();
}

void Timer::DefaultCallback(int signo, siginfo_t *info, void *context)
{
    (void)context;

    prctl(PR_SET_NAME, "Timer", 0, 0, 0, 0);

    if((info->si_code == SI_TIMER) && (signo == SIGUSR1))
    {
        TimerContext *timerContext = (TimerContext*)(info->si_value.sival_ptr);
        if (!timerContext) {
            return;
        }

        if (timerContext->magic != TIMER_MAGIC) {
            return;
        }

        Timer *_this = (Timer *)(timerContext->self);
        if (!_this) {
            return;
        }

        /* check the enabled timer class against my timer class */
        if (!(mEnabledTimerClass & _this->mTimerClass)) {

            /* my timer class is not enabled! */
            return;
        }

        /* copy the specific context */
        TimerContext copyOftimerContext = *timerContext;

        /* timer wants to send a state machine message? */
        if(copyOftimerContext.isSMtimer)
        {
           if(0 != copyOftimerContext.smTimerCallback)
           {
              copyOftimerContext.smTimerCallback->handleSmTimerCallback((const char*)copyOftimerContext.userData);
           }
           else
           {
              Dispatcher::GetInstance().SendMessage((const char*) copyOftimerContext.userData);
           }
        }
        else
        {
           /* or timer wants to call a callback function */
           (copyOftimerContext.pCallBackFn)(copyOftimerContext.timerId,copyOftimerContext.userObject,copyOftimerContext.userData);
        }
    }
}

bool Timer::StartTimer(timer_t &timerIndex ,long timeMilliseconds,long intervalMilliseconds, void *pObject,tCallBackFn pCallBackFn, const void *userData)
{
    return StartTimerInternal(timerIndex , timeMilliseconds, intervalMilliseconds ,false,pObject,pCallBackFn, userData);
}

bool Timer::StartSMTimer(timer_t& timerIndex, long timeMilliseconds, long intervalMilliseconds, const char* message, ::fw::ISmTimerCallback* callback /*= 0*/)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   FW_IF_NULL_PTR_RETURN_FALSE(message[0]);
   // callback can be 0

   return StartTimerInternal(timerIndex, timeMilliseconds, intervalMilliseconds, true, NULL, NULL, message, callback);
}

bool Timer::StartTimerInternal(timer_t &timerIndex ,long timeMilliseconds, long intervalMilliseconds, bool isSMtimer,void *pObject,tCallBackFn pCallBackFn, const void *userData, ::fw::ISmTimerCallback* smCallback /*= 0*/)
{
    /*
     * create a new timer
     */
    m_Lock.lock();

    struct sigevent sev;
    struct itimerspec its;
    Timer::TimerContext* context = NULL;

    /* find empty entry */
    unsigned int i;
    for(i=0; i<m_contexts.size(); i++) {
        if (m_contexts[i]->used == 0) {
            context = m_contexts[i];
            break;
        }
    }

    /* no free slot found, create a new one */
    int isNewContext = 0;
    if (i == m_contexts.size()) {

        /* create a new context */
        isNewContext = 1;
        context = new TimerContext;
    }

    if(0 == context)
    {
       ETG_TRACE_ERR(("failed to create timer context"));
       ETG_TRACE_ERRMEM((" #CONN: Timer: 0 == context"));
       m_Lock.unlock();
       return false;
    }

    context->self        = this;
    context->userData    = userData;
    context->userObject  = pObject;
    context->pCallBackFn = pCallBackFn;
    context->isSMtimer   = isSMtimer;
    context->smTimerCallback = smCallback;
    context->timerId     = 0; // not timer id given right now
    context->magic       = TIMER_MAGIC;

    /* frame sigevent */
    memset(&sev, 0, sizeof(sev));
#if SIGACTION_THREAD
    sev.sigev_notify = SIGEV_THREAD_ID;
    sev._sigev_un._tid = mSigactionThreadID;
#else
    sev.sigev_notify = SIGEV_SIGNAL;
#endif
    sev.sigev_signo = SIGUSR1;
    sev.sigev_value.sival_ptr = (void*) context;

    if (timer_create(CLOCK_MONOTONIC, &sev, &context->timerId) == -1)
    {
        ETG_TRACE_ERR(("failed to create a timer: error=%s", strerror(errno)));
        ETG_TRACE_ERRMEM((" #CONN: failed to create timer: error=%s", strerror(errno)));
        delete context;
        if(0 == isNewContext) // if reused context is deleted, then remove it from context list.
        {
            m_contexts.erase(m_contexts.begin() + i);
        }
        m_Lock.unlock();
        return false;
    }

    if (context->timerId == 0)
    {
        ETG_TRACE_ERR(("timer not created, timer id is 0 "));
        delete context;
        if(0 == isNewContext)
        {
            m_contexts.erase(m_contexts.begin() + i);
        }
        m_Lock.unlock();
        return false;
    }

    // set the systems timer index as return value
    timerIndex = context->timerId;

    /* start the timer */
    memset(&its, 0, sizeof(its)); // set struct to zero
    its.it_value.tv_sec = timeMilliseconds/1000;
    its.it_value.tv_nsec = (timeMilliseconds%1000)*1000000L;

    its.it_interval.tv_sec = intervalMilliseconds/1000;
    its.it_interval.tv_nsec = (intervalMilliseconds%1000)*1000000L;

    if (timer_settime(context->timerId, 0, &its, NULL) == -1)
    {
        ETG_TRACE_ERR(("failed to start the timer "));
        ETG_TRACE_ERRMEM((" #CONN: failed to start the timer"));
        delete context;
        if(0 == isNewContext)
        {
            m_contexts.erase(m_contexts.begin() + i);
        }
        m_Lock.unlock();
        return false;
    }

    /* the context is now used and valid */
    context->used        = 1;

    /* if it was a new context: add it to the context vector */
    if (isNewContext) {
        m_contexts.push_back(context);
    }

    m_Lock.unlock();

    /* debug: check the enabled timer class against my timer class */
    if (!(mEnabledTimerClass & mTimerClass)) {

        /* my timer class is not enabled! */
        ETG_TRACE_USR1(("WARNING: Started timer ID %p is currently filtered out (class=0x%x, whiteListFilter=0x%x)", timerIndex, mTimerClass, mEnabledTimerClass));
    }

    return true;
}

void Timer::CancelTimer (timer_t timerIndex)
{
    int ret;

    /* only valid id's */
    if (timerIndex == 0) return;

    m_Lock.lock();

    for(unsigned int iter = 0; iter < m_contexts.size(); iter++)
    {

        if(timerIndex == m_contexts[iter]->timerId && m_contexts[iter]->used == 1)
        {
            struct itimerspec its;

            /* stop the timer */
            memset(&its, 0, sizeof(its)); // set struct to zero
            if (timer_settime(m_contexts[iter]->timerId, 0, &its, NULL) == -1) {
                ETG_TRACE_ERR(("failed to stop the timer %p: %s", m_contexts[iter]->timerId, strerror(errno)));
            }

            // delete the timer
            ret = timer_delete(m_contexts[iter]->timerId);
            if (ret == -1)
            {
               ETG_TRACE_ERR(("failed to delete the timer %p: %s , may be timer is already stopped", m_contexts[iter]->timerId, strerror(errno)));
            }

            /* mark this slot as not used anymore */
            m_contexts[iter]->used = 0;

            break;
        }
    }

    m_Lock.unlock();
}

void Timer::CancelAllTimers()
{
    int ret;

    m_Lock.lock();

    /* cancel all running timers */
    for(unsigned int iter = 0; iter < m_contexts.size(); iter++)
    {

        if(m_contexts[iter]->used == 1)
        {
            struct itimerspec its;

            /* stop the timer */
            memset(&its, 0, sizeof(its)); // set struct to zero
            if (timer_settime(m_contexts[iter]->timerId, 0, &its, NULL) == -1) {
                ETG_TRACE_ERR(("failed to stop the timer %p: %s", m_contexts[iter]->timerId, strerror(errno)));
            }

            // delete the timer
            ret = timer_delete(m_contexts[iter]->timerId);
            if (ret == -1)
            {
                ETG_TRACE_ERR(("failed to delete the timer %p: %s , may be timer is already stopped", m_contexts[iter]->timerId, strerror(errno)));
            }

            /* mark this slot as not used anymore */
            m_contexts[iter]->used = 0;
        }
    }

    /* delete the memory used for timer contexts */
    for(unsigned int iter = 0; iter < m_contexts.size(); iter++)
    {
        delete m_contexts[iter];
    }
    m_contexts.clear();

    m_Lock.unlock();
}

void Timer::EnableTimerClass(const int timerClass)
{
    mEnabledTimerClass = timerClass;
}

void Timer::SetTimerClass(const int timerClass)
{
    mTimerClass = timerClass;
}
