#include <iostream>
using namespace std;

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pthread.h>
#include <errno.h>
#include <sys/prctl.h>
#include <setjmp.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ThreadFactory.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#endif
#endif

#include "ThreadFactory.h"
#include "FunctionTracer.h"
#include "TraceDefinitions.h"
#include "Framework_Errorcodes.h"
#include "Utils.h"
#include "Timer.h"


#define DEFAULT_STACKSIZE 10000000

/* needed to init the exception handler for signal in GM Gen2 target */
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
// it is not allowed to call OSAL internal function directly
// extern "C" void init_exception_handler(const char *name);
#endif

ThreadFactory::ThreadFactory() : mLock(), mFreeWorkerThreads(), mInitialized(false),
      mNoOfThreads(mDefaultNoOfThreads)
{
	memset(mThreadPrefix, 0, sizeof(mThreadPrefix));
	memset(mThreadContext, 0, sizeof(mThreadContext));

	// set default thread prefix (media player)
	mThreadPrefix[0] = 'M';
	mThreadPrefix[1] = 'P';
}

ThreadFactory::ThreadFactory(const int numberOfThreads, const char* threadPrefix) : mLock(), mFreeWorkerThreads(),
      mInitialized(false), mNoOfThreads(mDefaultNoOfThreads)
{
	memset(mThreadPrefix, 0, sizeof(mThreadPrefix));
   memset(mThreadContext, 0, sizeof(mThreadContext));

	// set default thread prefix (media player)
	mThreadPrefix[0] = 'M';
	mThreadPrefix[1] = 'P';

	// check given data
	if((numberOfThreads > 0) && (numberOfThreads <= mMaxNoOfThreads)) {
		mNoOfThreads = numberOfThreads;
	}
	if((threadPrefix != NULL) && (threadPrefix[0] != '\0')) {
		memset(mThreadPrefix, 0, sizeof(mThreadPrefix));
		strncpy(mThreadPrefix, threadPrefix, mSizeOfThreadPrefix);
	}
}

int ThreadFactory::Do(TFThread *thread, int param , void *userContext)
{
    ENTRY
	if(!mInitialized)
	{
		ETG_TRACE_ERR(("Thread registration rejected as ThreadFactory is not initialized yet ! "));
		return FW_ERR_TF_NOT_INIT;
	}	
	
	int threadIndex;

	/* wait for a free thread */
	int value;
	sem_getvalue(&mFreeWorkerThreads, &value);
    ETG_TRACE_USR3(("mFreeWorkerThreads=%d", value));

	int res;
    while(1) {
        res = sem_wait(&mFreeWorkerThreads);
        if (!res) break;
        res = errno;
        if (res != EINTR) break;
    }
    FW_FATAL_ASSERT(res == 0);

	// lock this section
	FW_FATAL_ASSERT(pthread_mutex_lock(&mLock) == 0);

	// search loop
	while(1) {

		// find an available worker thread
		for(threadIndex=0; threadIndex<mNoOfThreads; threadIndex++) {

			// is this thread free?
			if (!mThreadContext[threadIndex].busy) {
				mThreadContext[threadIndex].busy = 1; // now this thread is booked

				// unlock this section
				FW_FATAL_ASSERT(pthread_mutex_unlock(&mLock) == 0);

				// set the user values
				mThreadContext[threadIndex].userContext = userContext;
				mThreadContext[threadIndex].threadFunction = thread;
				mThreadContext[threadIndex].functionID = param;

			    // unlock the mutex before trigger the thread
				pthread_mutex_unlock(&mThreadContext[threadIndex].mtx);

				return threadIndex;
			}
		}

        // unlock this section
        FW_FATAL_ASSERT(pthread_mutex_unlock(&mLock) == 0);

		// no thread found: this cannot be happen
		FW_FATAL_ASSERT(threadIndex != mNoOfThreads);
	}

	return -1;
}

void *ThreadFactory::startFunction(void *context)
{
    ENTRY_INTERNAL
	tThreadContext *ctx = (tThreadContext *)context;
	int ret;

	FW_FATAL_ASSERT(ctx != NULL);
	
	// init the threads name
	ResetName(ctx);

	// set the thread property to enable the cancellation later on
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE ,NULL);
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS ,NULL);

	// endless control loop
	while(!ctx->stopThread) {

		// lock the mutex to wait for next task to do
		ret = pthread_mutex_lock(&ctx->mtx);
		if (ret || ctx->stopThread) break; // error in locking or stop requested

		// test
		// printf("thread %d: running", ctx->count);

		// do the user function
		ctx->threadFunction->Do(ctx->functionID, ctx->userContext);

		// reset the thread name
		ResetName(ctx);

		// mark thread is not busy anymore, can be reused
		ctx->busy = 0;

		// one worker got free, count this
		sem_post(ctx->freeWorkerThreads);
	}

	// end of thread
	ETG_TRACE_USR1(("thread %d: stopping", ctx->count));
	return NULL;
}

void ThreadFactory::Create(void)
{
   ENTRY_INTERNAL

   // check if already created
   if (true == mInitialized)
   {
      return;
   }

   // setup the lock mutex
   FW_FATAL_ASSERT(pthread_mutex_init(&mLock, NULL) == 0);

   /* set the number of threads that are free initial */
   memset(&mFreeWorkerThreads, 0, sizeof(mFreeWorkerThreads));
   sem_init(&mFreeWorkerThreads, 0, mNoOfThreads);

   // create all threads but dont start them
   for (int i=0; i<mNoOfThreads; i++)
   {
      CreateThread(i);
   }

   mInitialized = true;
}

int ThreadFactory::CreateThread(const int threadIndex)
{
    ENTRY_INTERNAL
    int res;

    // init the factory thread context
    memset(&mThreadContext[threadIndex], 0, sizeof(ThreadFactory::tThreadContext));
    mThreadContext[threadIndex].count = threadIndex+1;  // set the count of this thread for debugging only
    mThreadContext[threadIndex].freeWorkerThreads = &mFreeWorkerThreads;

    // set thread prefix
    mThreadContext[threadIndex].prefix = mThreadPrefix;

    // setup the wait mutex
    FW_FATAL_ASSERT(pthread_mutex_init(&mThreadContext[threadIndex].mtx, NULL) == 0);

    // lock the mutex to stop the thread after starting
    FW_FATAL_ASSERT(pthread_mutex_lock(&mThreadContext[threadIndex].mtx) == 0);

    // set the default thread attributes
    FW_FATAL_ASSERT(pthread_attr_init(&mThreadContext[threadIndex].mThreadAttr) == 0);

    // set the size of the stack. Default stack size is too small.
    pthread_attr_setstacksize(&mThreadContext[threadIndex].mThreadAttr, DEFAULT_STACKSIZE);

    // create the thread
    res = pthread_create(&mThreadContext[threadIndex].mThreadHandle, &mThreadContext[threadIndex].mThreadAttr, &ThreadFactory::startFunction, &mThreadContext[threadIndex]);
    if (res) {
        ETG_TRACE_FATAL(("pthread_create: err=%d", res));
        FW_FATAL_ASSERT(res == 0);
    }

    return 0;
}

bool ThreadFactory::JoinTimeoutCallBack(timer_t timerID , void* instance ,const void *userData)
{
    (void)(timerID);
    (void)(instance);

    ENTRY_INTERNAL

    jmp_buf *jumpBuffer = (jmp_buf *)userData;
    //ThreadFactory *self = (ThreadFactory *)instance;

    /* jump after the pthread_join position */
    longjmp (*jumpBuffer, 1);

    return 0;
}

int ThreadFactory::CancelThread(const int threadIndex)
{
    ENTRY_INTERNAL

    // send the interrupting signal to thread
    pthread_cancel(mThreadContext[threadIndex].mThreadHandle);

    // create a supervision timer to jump over the join after a certain time (3000 ms)
    Timer joinTimeout;
    timer_t joinTimeoutID;
    jmp_buf jumpBuffer;
    joinTimeout.StartTimer(OUT joinTimeoutID, IN 3000L, 0L, IN this, IN &JoinTimeoutCallBack, IN (void *)&jumpBuffer);

    /* set the jump back point for long jump out of join timeout handler */
    switch(setjmp(jumpBuffer))
    {
        case 0: // normal processing
            // is thread ended?
            pthread_join(mThreadContext[threadIndex].mThreadHandle, NULL);
            break;
        case 1: // abort by timer callback
            ETG_TRACE_ERR(("CancelThread: pthread_join() skipped"));
            break;
        default:
            break;
    }

    // end the join timeout timer
    joinTimeout.CancelTimer(IN joinTimeoutID);

    // unlock the mutex to enable the destroy
    pthread_mutex_unlock(&mThreadContext[threadIndex].mtx);

    // destroy the mutex
    pthread_mutex_destroy(&mThreadContext[threadIndex].mtx);

    return 0;
}

int ThreadFactory::SendSignal(const int threadIndex, const int sig)
{
    ENTRY_INTERNAL

    //tgkill(-1, mThreadContext[threadIndex].mThreadHandle, sig);
    pthread_kill(mThreadContext[threadIndex].mThreadHandle, sig);

    return 0;
}

int ThreadFactory::KillThread(const int threadIndex)
{
    ENTRY_INTERNAL

    if (mThreadContext[threadIndex].busy) {

        CancelThread(threadIndex);

        CreateThread(threadIndex);

        // signal that one worker got free
        sem_post(mThreadContext[threadIndex].freeWorkerThreads);
    }

    return 0;
}

int ThreadFactory::KillAllThreads(void)
{
    ENTRY_INTERNAL
	int i;
	
	mInitialized = false;

	// wakeup all threads to stop them
	for (i=0; i<mNoOfThreads; i++) {

		// flag to stop the thread
		mThreadContext[i].stopThread = 1;

		// unlock the mutex to let the thread run
		FW_FATAL_ASSERT(pthread_mutex_unlock(&mThreadContext[i].mtx) == 0);
	}

	// now join all threads
	for (i=0; i<mNoOfThreads; i++) {

	    CancelThread(i);
	}
	return 0;
}

void ThreadFactory::SetName(const char *name)
{
	prctl(PR_SET_NAME, name, 0, 0, 0, 0);
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
// it is not allowed to call OSAL internal function directly
//  init_exception_handler(name);
#endif
}

void ThreadFactory::ResetName(tThreadContext *ctx)
{
	// reset the thread name
	char name[64];
	snprintf(name, sizeof(name)-1, "%s_Thread_%02d", ctx->prefix, ctx->count);
	SetName(name);
}

ThreadFactory::tThreadID ThreadFactory::GetThreadId()
{
    return static_cast<ThreadFactory::tThreadID>(pthread_self());
}

int ThreadFactory::GetThreadsRunning()
{
    int value;
    sem_getvalue(&mFreeWorkerThreads, &value);
    return mNoOfThreads - value;
}

int ThreadFactory::SetPriority(int prio)
{
	int policy;
	struct sched_param param;

	pthread_getschedparam(pthread_self(), &policy, &param);

	param.__sched_priority = prio;
	policy = SCHED_FIFO;

	pthread_setschedparam(pthread_self(), policy, &param);

	return 0;
}
