/*
 * MessageQueue.cpp
 *
 *  Created on: Jul 1, 2013
 *      Author: Dinesh
 */

/* system header include*/
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/time.h>

/*Trace includes*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_mp.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/MessageQueue.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#endif
#endif
#include "TraceDefinitions.h"
#include "FunctionTracer.h"
#include "FwAssert.h"

/* Class header include*/
#include "MessageQueue.h"

#ifdef VARIANT_S_FTR_ENABLE_FW_MEMORY_CHECKER
#include "FwMemoryChecker.h"
#endif

/*lint -save -e571 */

#if OLD_IMPL

#define MAGIC 0x8976543 /**< some random but fixed number as magic token. Be careful to just use a hex value which fits to the size of int! */

/**
 * header for the message to be transferred
 */
struct tMessageHeader
{
    int size;   /**< size of the actual user data */
    int token;  /**< a token to compare if sent data is valid */
};

MessageQueue::MessageQueue(const char *name)
{
    mName = strdup(name); // will be freed by destructor
    memset(&mCountingSemaphore, 0, sizeof(mCountingSemaphore));
    sem_init(&mCountingSemaphore, 0, 0);
}

MessageQueue::~MessageQueue(void)
{
    sem_destroy(&mCountingSemaphore);
    if (mName) {
        free((void *)mName);
    }
}

tResult MessageQueue::Push(const void * message, size_t msgSize, int priority)
{
    tMessageHeader *msgHeader;
    mAccessLock.lock();

    /* allocate the buffer ( space for the header + data) */
    unsigned char* buffer = (unsigned char *)malloc(msgSize + sizeof(tMessageHeader));
    if (!buffer) {
        mAccessLock.unlock();
        return -1;
    }

    /* set the message header */
    msgHeader        = (tMessageHeader *)buffer;
    msgHeader->token = MAGIC;
    msgHeader->size  = msgSize;

    /*set the message data*/
    memcpy(buffer+sizeof(tMessageHeader),message,msgSize);

    /*insert the message into the queue according to its priority*/
    mMessages[priority].push_back(buffer);
    ReleaseMessageWait();

    // debug printf("MessageQueue::Push message:%s,size:%d,strlen:%d\n",(char*)message,msgSize,strlen((char*)message));

    mAccessLock.unlock();
    return 0;
}

tResult MessageQueue::Pop(void** message,size_t *msgSize)
{
    tResult res =-1;
    map<int, vector<void*> >::iterator iter;
    mAccessLock.lock();

    /*look for message lists from high to low prio -- loop over message queue prios */
    for(iter = mMessages.begin(); iter != mMessages.end(); ++iter)
    {
        /* are there messages for the current prio */
        if(iter->second.size() > 0)
        {
            /*get the message at the front of the queue*/
            void* buffer = iter->second.front();

            /*set the return message with the popped element*/
            tMessageHeader *msgHeader = (tMessageHeader *)buffer;
            if(msgHeader->token == MAGIC)
            {
                *message = (unsigned char *)malloc(msgHeader->size);
                if(*message)
                {
                    memcpy(*message,(unsigned char*)buffer + sizeof(tMessageHeader),msgHeader->size);
                    *msgSize = msgHeader->size;
                }
            }
            else
            {
                ETG_TRACE_ERR (("Unknown message with wrong magic found in the queue : %s",mName));
            }

            //debug printf("MessageQueue::Pop message:%s,size:%d,strlen:%d\n",(char*)*message,*msgSize,strlen((char*)*message));

            /*free the element and erase it from the queue*/
            free(buffer);
            iter->second.erase(iter->second.begin());

            res = 0;    /*a message found and poped out*/
            break;
        }
    }

    mAccessLock.unlock();
    return res;
}

void* MessageQueue::WaitForMessage (size_t *msgSize,unsigned int time_out_in_ms)
{
    tResult res;

    void *retMessage = NULL;

    /* wait for a new message */
    res = StartMessageWait(time_out_in_ms);

    /* wait ends with timeout */
    if(ETIMEDOUT == res) {

        ETG_TRACE_ERR (("MessageQueue::WaitForMessage -> timeout(%d msec) occured while reading queue %s",time_out_in_ms,mName));

    /*wait ends with message arrived in queue*/
    } else if (!res) {

        res = Pop(&retMessage,msgSize);
        if(res) {
            ETG_TRACE_ERR (("MessageQueue::WaitForMessage ->error:%d no message found in queue even after wait in queue %s , Should NOT OCCUR!!",res,mName));
        }

    /*wait ends with an error*/
    } else {
        ETG_TRACE_ERR(("MessageQueue::WaitForMessage error while waiting for message from queue : %d (%s) !",res,mName));
    }

    return retMessage;
}

uint64_t MessageQueue::GetCurMessagesCount()
{
    int semCount = 0;
    sem_getvalue(&mCountingSemaphore, &semCount);
    return (uint64_t)semCount;
}

void* MessageQueue::CreateMessageBuffer (size_t size)
{
    return malloc(size);
}

tResult MessageQueue::DropMessage(void* message )
{
    if(message)
        free(message);
    return 0;
}

tResult MessageQueue::Flush()
{
    map<int, vector<void*> >::iterator iter;
    mAccessLock.lock();

    /*iterate through all the priorities list*/
    for(iter = mMessages.begin(); iter != mMessages.end(); ++iter)
    {
        while(iter->second.size() > 0)
        {
            //free the last element
            free(iter->second.back());
            iter->second.pop_back();

            /* care about the semaphore */
            sem_trywait(&mCountingSemaphore);
        }
    }

    mAccessLock.unlock();
    return 0;
}

tResult MessageQueue::ReleaseMessageWait(void)
{
    return sem_post(&mCountingSemaphore);
}

tResult MessageQueue::StartMessageWait(unsigned int milliSeconds)
{
    tResult res;

    /* wait without timeout? */
    if (!milliSeconds) {

        while(1) {
            res = sem_wait(&mCountingSemaphore);
            if (!res) break;
            res = errno;
            if (res != EINTR) break;
        }
        if (res != ETIMEDOUT && res) {
            ETG_TRACE_ERR (("%20s: error on sem_wait: %d/%s", mName, errno, strerror(errno)));
        }

    } else { // wait with timeout

        /* wait on counting semaphore */
        struct timespec timeToWait;

        /*get current time*/
        clock_gettime(CLOCK_REALTIME , &timeToWait);

        /*update timeToWait with wait duration */
        timeToWait.tv_sec += milliSeconds/1000;
        unsigned long long nsec = (unsigned long long)timeToWait.tv_nsec + (unsigned long long)((milliSeconds%1000) * 1000000L);
        if(nsec >= 1000000000L)
        {
            timeToWait.tv_sec += nsec/1000000000L;
            nsec = nsec % 1000000000L;
        }
        timeToWait.tv_nsec = nsec;

        /* wait for semaphore */
        while(1) {
            res = sem_timedwait(&mCountingSemaphore, &timeToWait);
            if (!res) break; // no error
            res = errno; // get the last error
            if (res != EINTR) break;
        }
        if (res != ETIMEDOUT && res) {
            ETG_TRACE_ERR (("%20s: error on sem_timedwait: %d/%s", mName, errno, strerror(errno)));
        }
    }

    return res;
}

tResult MessageQueue::DoLock()
{
    mAccessLock.lock();
    return 0;
}

tResult MessageQueue::UnLock()
{
    mAccessLock.unlock();
    return 0;
}

void * MessageQueue::Peek(const int iMessageNumber, size_t *msgSize)
{
    map<int, vector<void*> >::iterator iter;
    void *message = NULL;
    int iMsg = (int)iMessageNumber;

    *msgSize = 0;

    /*look for message lists from high to low prio*/
    for(iter = mMessages.begin(); iter != mMessages.end(); ++iter)
    {
        // Debug printf("Peek: iMsg=%d ?= (int)iter->second.size()=%d\n", iMsg, (int)iter->second.size()); fflush(stdout);

        /* is the iMessageNumber inside the current prio */
        if (iMsg < (int)iter->second.size()) {

            /* get the message */
            void* buffer = iter->second.at(iMsg);

            // Debug printf("Peek: iMsg=%d, buffer=%s\n", iMsg, (char *)(buffer + sizeof(tMessageHeader))); fflush(stdout);

            /* set the return message with the popped element */
            tMessageHeader *msgHeader = (tMessageHeader *)buffer;
            if(msgHeader->token == MAGIC)
            {
                message = (void *)malloc(msgHeader->size);
                if(message)
                {
                    memcpy(message,(unsigned char*)buffer + sizeof(tMessageHeader),msgHeader->size);
                    *msgSize = msgHeader->size;
                }
            }
            else
            {
                ETG_TRACE_ERR (("Unknown message with wrong magic found in the queue : %s",mName));
            }

            break;

        /* message must be in another prio */
        } else {

            iMsg -= iter->second.size();
        }
    }

    return message;
}

tResult MessageQueue::Remove(const int iMessageNumber)
{
    map<int, vector<void*> >::iterator iter;
    int iMsg = (int)iMessageNumber;

    /*look for message lists from high to low prio*/
    for(iter = mMessages.begin(); iter != mMessages.end(); ++iter)
    {
        /* is the iMessageNumber inside the current prio */
        if (iMsg < (int)iter->second.size()) {

            /* get the message */
            void* buffer = iter->second.at(iMsg);

            /* free the buffer */
            free(buffer);

            /* erase the message */
            iter->second.erase(iter->second.begin() + iMsg);

            /* care about the semaphore */
            sem_trywait(&mCountingSemaphore);

            /* end of search */
            break;

        /* message must be in another prio */
        } else {

            iMsg -= iter->second.size();
        }
    }

    return 0;
}

#else // OLD_IMPL

static FILE *fpLog = NULL;

MessageQueue::MessageQueue(const char *name)
{
    mName = strdup(name); // will be freed by destructor
    memset(&mCountingSemaphore, 0, sizeof(mCountingSemaphore));
    sem_init(&mCountingSemaphore, 0, 0);

    /* init the queue array */
    memset(&mQueueArray, 0, sizeof(mQueueArray));

#if 0
    if (!fpLog) {
        fpLog = fopen("/tmp/DropMessage.log", "w");
    }
#endif
}

MessageQueue::~MessageQueue(void)
{
    sem_destroy(&mCountingSemaphore);
    if (mName) {
        free((void *)mName);
    }

    /* free the queue array */
    for(int i=0; i<mMaxPriority; i++) {
        if (mQueueArray[i].base) {
            free(mQueueArray[i].base);
        }
    }
}

MessageQueue::MessageQueue(const MessageQueue& ref) :
mQueueArray(/*ref.mQueueArray*/),
mName(0 /*ref.mName*/),
mAccessLock(/*ref.mAccessLock*/),
mCountingSemaphore(/*ref.mCountingSemaphore*/)
{
   (void)(ref);

   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();
}

MessageQueue& MessageQueue::operator=(const MessageQueue& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   // mQueueArray = ref.mQueueArray;
   // mName = ref.mName;
   // mAccessLock = ref.mAccessLock;
   // mCountingSemaphore = ref.mCountingSemaphore;

   return *this;
}

tResult MessageQueue::Push(const void * message, size_t msgSize, int priority)
{

    /* check the max-prio */
    if (priority >= mMaxPriority || priority < 0) {
        return -1;
    }

    mAccessLock.lock();

    /* is it the very first message for this priority? */
    if (mQueueArray[priority].base == NULL) {

        /* create the queue */
        mQueueArray[priority].base = (tQueueElement *)malloc(mIntegralSize * sizeof(tQueueElement));
        if (!mQueueArray[priority].base) {
            mAccessLock.unlock();
            return -2;
        }
        mQueueArray[priority].head = mQueueArray[priority].base;
        mQueueArray[priority].tail = mQueueArray[priority].head;
        mQueueArray[priority].size = mIntegralSize;
        mQueueArray[priority].end = mQueueArray[priority].base + mQueueArray[priority].size;
    }

    /* push the message */
    mQueueArray[priority].tail->message = message;
    mQueueArray[priority].tail->size = msgSize;

    /* move tail one step further (wrap around at end of queue) */
    mQueueArray[priority].tail++;
    if (mQueueArray[priority].tail == mQueueArray[priority].end) {
        mQueueArray[priority].tail = mQueueArray[priority].base;
    }

    /* has tail reached the head: the queue is full, a realloc is needed */
    if (mQueueArray[priority].tail == mQueueArray[priority].head) {

        /* save the head offset to the base for resseting it after reallocation */
        intptr_t headOffset = mQueueArray[priority].head - mQueueArray[priority].base;

        mQueueArray[priority].base = (tQueueElement *)realloc((void *)mQueueArray[priority].base, (mQueueArray[priority].size + mIntegralSize) * sizeof(tQueueElement));
        if (!mQueueArray[priority].base) {

            /* malloc failed, invalidate this queue */
            mQueueArray[priority].end = NULL;
            mQueueArray[priority].size = 0;
            mQueueArray[priority].head = NULL;
            mQueueArray[priority].tail = NULL;

            mAccessLock.unlock();
            return -3;
        }

        /* set tail to new memory allocated */
        mQueueArray[priority].tail = mQueueArray[priority].base + mQueueArray[priority].size;

        /* re-set new head */
        mQueueArray[priority].head = mQueueArray[priority].base + headOffset;

        /* correct size and end info for this queue */
        mQueueArray[priority].size += mIntegralSize;
        mQueueArray[priority].end = mQueueArray[priority].base + mQueueArray[priority].size;
    }

    /* message was processed, signal this to the receiver */
    ReleaseMessageWait();

    mAccessLock.unlock();
    return 0;
}

tResult MessageQueue::Pop(void** message,size_t *msgSize)
{
    mAccessLock.lock();

    /* look for the first priority which has a unread message */
    for(int priority=0; priority < mMaxPriority; priority++) {

        /* has priority an unread message? */
        if (mQueueArray[priority].head != mQueueArray[priority].tail) {

            /* read it */
            *message = (void *)mQueueArray[priority].head->message;
            *msgSize = mQueueArray[priority].head->size;

            /* move head one step further (and do the wrap around) */
            mQueueArray[priority].head++;
            if (mQueueArray[priority].head == mQueueArray[priority].end) {
                mQueueArray[priority].head = mQueueArray[priority].base;
            }

            /* message read: end */
            mAccessLock.unlock();

            if (*message) {
                return 0;
            } else {
                return -2; // out of memory
            }
        }
    }

    /* no message found */
    mAccessLock.unlock();
    return -1;
}

void* MessageQueue::WaitForMessage (size_t *msgSize,unsigned int time_out_in_ms)
{
    tResult res;

    void *retMessage = NULL;

    /* wait for a new message */
    res = StartMessageWait(time_out_in_ms);

    /* wait ends with timeout */
    if(ETIMEDOUT == res) {

        ETG_TRACE_ERR (("MessageQueue::WaitForMessage -> timeout(%d msec) occured while reading queue %s",time_out_in_ms,mName));

    /*wait ends with message arrived in queue*/
    } else if (!res) {

        res = Pop(&retMessage,msgSize);
        if(res) {
            ETG_TRACE_ERR (("MessageQueue::WaitForMessage ->error:%d no message found in queue even after wait in queue %s , Should NOT OCCUR!!",res,mName));
        }

    /*wait ends with an error*/
    } else {
        ETG_TRACE_ERR(("MessageQueue::WaitForMessage error while waiting for message from queue : %d (%s) !",res,mName));
    }

    return retMessage;
}

uint64_t MessageQueue::GetCurMessagesCount()
{
    int semCount = 0;
    sem_getvalue(&mCountingSemaphore, &semCount);
    return (uint64_t)semCount;
}

void* MessageQueue::CreateMessageBuffer(size_t size)
{
    void *message;
    message = malloc(size);
    if (fpLog) {
        fprintf(fpLog, "CreateMessageBuffer: message=%p\n", message);
    }
#ifdef VARIANT_S_FTR_ENABLE_FW_MEMORY_CHECKER
   FW_MEMORY_CHECK_NEW_OBJ(message);
#endif
    return message;
}

void hexDump (FILE *fp, const char *desc, void *addr, int len);

tResult MessageQueue::DropMessage(void* message)
{
    if (fpLog) {
        fprintf(fpLog, "DropMessage: message=%p\n", message);
        hexDump(fpLog, "DropMessage", message, 32);
    }

#ifdef VARIANT_S_FTR_ENABLE_FW_MEMORY_CHECKER
   FW_MEMORY_CHECK_DEL_OBJ(message);
#endif
    if(message)
        free(message);
    return 0;
}

tResult MessageQueue::Flush()
{
    mAccessLock.lock();

    /* loop over the priorities */
    for(int priority=0; priority < mMaxPriority; priority++) {

        /* loop over all unread messages */
        while(1) {

            /* end of loop? */
            if (mQueueArray[priority].head == mQueueArray[priority].tail) {
                break;
            }

            /* drop the memory for this message */
            DropMessage((void *)mQueueArray[priority].head->message);

            /* de-count this message */
            sem_trywait(&mCountingSemaphore);

            /* move head one step further (and do the wrap around) */
            mQueueArray[priority].head++;
            if (mQueueArray[priority].head == mQueueArray[priority].end) {
                mQueueArray[priority].head = mQueueArray[priority].base;
            }
        }
    }

    mAccessLock.unlock();
    return 0;
}

tResult MessageQueue::ReleaseMessageWait(void)
{
    return sem_post(&mCountingSemaphore);
}

tResult MessageQueue::StartMessageWait(unsigned int milliSeconds)
{
    tResult res;

    /* wait without timeout? */
    if (!milliSeconds) {

        while(1) {
            res = sem_wait(&mCountingSemaphore);
            if (!res) break;
            res = errno;
            if (res != EINTR) break;
        }
        if (res != ETIMEDOUT && res) {
            ETG_TRACE_ERR (("%20s: error on sem_wait: %d/%s", mName, errno, strerror(errno)));
        }

    } else { // wait with timeout

        /* wait on counting semaphore */
        struct timespec timeToWait;

        /*get current time*/
        clock_gettime(CLOCK_REALTIME , &timeToWait);

        /*update timeToWait with wait duration */
        timeToWait.tv_sec += milliSeconds/1000;
        unsigned long long nsec = (unsigned long long)timeToWait.tv_nsec + (unsigned long long)((milliSeconds%1000) * 1000000L);
        if(nsec >= 1000000000L)
        {
            timeToWait.tv_sec += nsec/1000000000L;
            nsec = nsec % 1000000000L;
        }
        timeToWait.tv_nsec = nsec;

        /* wait for semaphore */
        while(1) {
            res = sem_timedwait(&mCountingSemaphore, &timeToWait);
            if (!res) break; // no error
            res = errno; // get the last error
            if (res != EINTR) break;
        }
        if (res != ETIMEDOUT && res) {
            ETG_TRACE_ERR (("%20s: error on sem_timedwait: %d/%s", mName, errno, strerror(errno)));
        }
    }

    return res;
}

tResult MessageQueue::DoLock()
{
    mAccessLock.lock();
    return 0;
}

tResult MessageQueue::UnLock()
{
    mAccessLock.unlock();
    return 0;
}

void * MessageQueue::Peek(const int iMessageNumber, size_t *msgSize)
{
    mAccessLock.lock();
    int messageCounter = 0;

    /* loop over the priorities */
    for(int priority=0; priority < mMaxPriority; priority++) {

        /* use a temp head and do not modify the real head of queue */
        tQueueElement *head = mQueueArray[priority].head;

        /* loop over all unread messages */
        while(1) {

            /* end of loop? */
            if (head == mQueueArray[priority].tail) {
                break;
            }

            /* is desired message found? */
            if (iMessageNumber == messageCounter) {

                /* return the message and exit */
                *msgSize = head->size;
                mAccessLock.unlock();
                return (void *)head->message;
            }

            /* next message to be checked */
            messageCounter++;

            /* move head one step further (and do the wrap around) */
            head++;
            if (head == mQueueArray[priority].end) {
                head = mQueueArray[priority].base;
            }
        }
    }

    mAccessLock.unlock();

    return NULL;
}

tResult MessageQueue::Remove(const int iMessageNumber)
{
    mAccessLock.lock();

    int messageCounter = 0;

    /* loop over the priorities */
    for(int priority=0; priority < mMaxPriority; priority++) {

        /* use a temp head and do not modify the real head of queue */
        tQueueElement *element = mQueueArray[priority].head;

        /* loop over all unread messages */
        while(1) {

            /* end of loop? */
            if (element == mQueueArray[priority].tail) {
                break;
            }

            /* is desired message found? */
            if (iMessageNumber == messageCounter) {

                /* drop the memory for this message */
                DropMessage((void *)element->message);

                /* remove this element from queue by shifting all elements in front of it one step to the right */
                while(1) {

                    /* compute the next element after the current one */
                    tQueueElement *next = element;
                    next++;
                    if (next == mQueueArray[priority].end) {
                        next = mQueueArray[priority].base;
                    }

                    /* end of loop? */
                    if (next == mQueueArray[priority].tail) {

                        /* correct the tail */
                        mQueueArray[priority].tail = element;

                        /* end */
                        break;
                    }

                    /* copy the data from the next to the current one */
                    element->message = next->message;
                    element->size = next->size;

                    /* further elements to shift */
                    element = next;
                }

                /* de-count the removed message */
                sem_trywait(&mCountingSemaphore);

                mAccessLock.unlock();
                return 0;
            }

            /* next message to be checked */
            messageCounter++;

            /* move head one step further (and do the wrap around) */
            element++;
            if (element == mQueueArray[priority].end) {
                element = mQueueArray[priority].base;
            }
        }
    }

    mAccessLock.unlock();

    return -1;
}

#endif // OLD_IMPL
