/**
 * @file FwUtils.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Utilities.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Utilities.
 */

#include "FwUtils.h"
#include "FwStringUtils.h"
#include "FwAssert.h"
#include "TraceDefinitions.h"
#include "FwTrace.h"

#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <cerrno>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <ifaddrs.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netpacket/packet.h>
#include <sys/types.h>
#include <sys/syscall.h>
#include <sys/prctl.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FwUtils.cpp.trc.h"
#endif
#endif

namespace fw {

static const size_t _threadNameBufferSize = 17; /**< minimum buffer size for thread name (including NULL termination) */
static const size_t _localTimerBufferSize = 16; /**< minimum buffer size for current time (including NULL termination) */

bool isProcessRunning(const ::std::string& processName)
{
   // copied from http://stackoverflow.com/questions/6898337/determine-programmatically-if-a-program-is-running
   DIR* dir;
   struct dirent* ent;
   char buf[512];
   long pid;
   char pname[100] = { 0 };
   char state;
   FILE *fp = 0;

   if(0 == (dir = opendir("/proc")))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   while(0 != (ent = readdir(dir)))
   {
      long lpid = atol(ent->d_name);
      if(lpid < 0)
      {
         continue;
      }

      snprintf(buf, sizeof(buf), "/proc/%ld/stat", lpid);
      fp = fopen(buf, "r");
      if(0 != fp)
      {
         if(3 != fscanf(fp, "%ld (%[^)]) %c", &pid, pname, &state))
         {
            fclose(fp);
            closedir(dir);
            FW_NORMAL_ASSERT_ALWAYS();
            return false;
         }

         if(0 == strcmp(pname, processName.c_str()))
         {
            fclose(fp);
            closedir(dir);
            // return (pid_t)lpid;
            return true;
         }

         fclose(fp);
      }
   }

   closedir(dir);

   return false;
}

bool getNetworkMacAddress(::std::vector<unsigned char>& macAddress, ::std::string& macAddressString, const ::std::string& networkInterface)
{
   if(true == networkInterface.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   // see also https://linux.die.net/man/3/getifaddrs
   struct ifaddrs* ifList = 0;
   bool result = false;

   if(0 != getifaddrs(&ifList))
   {
      // failed
      int errNumber = errno;
      ETG_TRACE_ERR((" getNetworkMacAddress(): getifaddrs failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      // success
      struct ifaddrs* ifEntry = 0;
      const char* name;

      for(ifEntry = ifList; ifEntry != 0; ifEntry = ifEntry->ifa_next)
      {
         // check Address of interface
         if(0 == ifEntry->ifa_addr)
         {
            continue;
         }

         // check Name of interface
         if(0 != ifEntry->ifa_name)
         {
            name = ifEntry->ifa_name;
         }
         else
         {
            name = "";
         }

         ETG_TRACE_USR3((" getNetworkMacAddress(): name=%s", name));

         if(0 == strcmp(name, networkInterface.c_str()))
         {
            // match => check family
            // AF_PACKET family provides the MAC address
            // AF_INET/AF_INET6 family provides the IP address
            if(AF_PACKET == ifEntry->ifa_addr->sa_family)
            {
               // match => found
               result = true;

               // get the MAC address
               struct sockaddr_ll* addr = (struct sockaddr_ll*)ifEntry->ifa_addr;

               macAddress.reserve(addr->sll_halen);
               for(unsigned char i = 0; i < addr->sll_halen; i++)
               {
                  macAddress.push_back(addr->sll_addr[i]);
               }

               // create printable string; example: 08:00:27:f0:95:6f => 6 bytes => 17 char + NULL
               macAddressString.reserve(3 * macAddress.size());
               char hexString[3];
               for(size_t i = 0; i < macAddress.size(); i++)
               {
                  convertByte2HexString(hexString, macAddress[i]);
                  macAddressString.push_back(hexString[0]);
                  macAddressString.push_back(hexString[1]);
               }

               // end the loop
               break;
            }
         }
      }
   }

   return result;
}

long int calcAbsoluteDifference(const long int value1, const long int value2)
{
   if(value1 > value2)
   {
      return (value1 - value2);
   }
   else
   {
      return (value2 - value1);
   }
}

pid_t getProcessId(void)
{
   return getpid();
}

pid_t getThreadId(void)
{
   return (pid_t)(syscall(SYS_gettid));
}

const char* getThreadName(char* buffer, const size_t bufferSize)
{
   if((0 != buffer) && (_threadNameBufferSize <= bufferSize))
   {
      memset(buffer, '\0', bufferSize);

      /*
      http://www.kernel.org/doc/man-pages/online/pages/man2/prctl.2.html
      PR_GET_NAME (since Linux 2.6.11)
            Return the process name for the calling process, in the buffer pointed
            to by (char *) arg2.  The buffer should allow space for up to 16 bytes;
            the returned string will be null-terminated if it is shorter than that.
      */
      (void)prctl(PR_GET_NAME, buffer, 0, 0, 0);

      return buffer;
   }
   else
   {
      return "";
   }
}

const char* getTime(char* buffer, const size_t bufferSize)
{
   if((0 != buffer) && (_localTimerBufferSize <= bufferSize))
   {
      struct timeval sys_time;
      struct tm* tm_info;

      if(-1 == gettimeofday(&sys_time, NULL))
      {
         return "";
      }
      else
      {
         tm_info = localtime(&sys_time.tv_sec);

         sprintf(buffer, "%02d:%02d:%02d.%06d", tm_info->tm_hour, tm_info->tm_min, tm_info->tm_sec, (int)(sys_time.tv_usec));
         buffer[_localTimerBufferSize - 1]  = '\0';

         return buffer;
      }
   }
   else
   {
      return "";
   }
}

bool doesFileExist(const ::std::string& fileName)
{
   if(true == fileName.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   struct stat statBuffer;
   int result = stat(fileName.c_str(), &statBuffer);
   if(0 != result)
   {
      // failed => check error code
      int errNumber = errno;
      if(ENOENT == errNumber)
      {
         ETG_TRACE_USR1((" doesFileExist(): file %s does not exist", fileName.c_str()));
      }
      else
      {
         // should never happen else we have to check, maybe wrong folder permissions
         ETG_TRACE_ERR((" doesFileExist(): stat of %100s failed: ERROR=%d (%s)", fileName.c_str(), errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      return false;
   }
   else
   {
      // success
      ETG_TRACE_USR1((" doesFileExist(): file %s exists", fileName.c_str()));

      return true;
   }
}

bool getGroupId4GroupName(__gid_t& groupId, const ::std::string& groupName)
{
   if(true == groupName.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   struct group* groupStructure;

   for(int i = 0; i < 3; i++)
   {
      groupStructure = getgrnam(groupName.c_str());
      if(0 == groupStructure)
      {
         // either the requested entry was not found, or an error occurred
         const int errNumber(errno);

         if(0 == errNumber)
         {
            // requested entry was not found
            ETG_TRACE_USR1((" getGroupId4GroupName(): group %s not found", groupName.c_str()));
            return false;
         }
         else
         {
            // error occurred
            if(EINTR == errNumber)
            {
               // try again
            }
            else
            {
               ETG_TRACE_ERR((" getGroupId4GroupName(): getgrnam failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
               FW_NORMAL_ASSERT_ALWAYS();
               return false;
            }
         }
      }
      else
      {
         // success
         groupId = groupStructure->gr_gid;
         ETG_TRACE_USR1((" getGroupId4GroupName(): group %s found", groupName.c_str()));
         return true;
      }
   }

   FW_NORMAL_ASSERT_ALWAYS();
   return false;
}

bool getUserId4UserName(__uid_t& userId, const ::std::string& userName)
{
   if(true == userName.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   struct passwd* passwdStructure;

   for(int i = 0; i < 3; i++)
   {
      passwdStructure = getpwnam(userName.c_str());
      if(0 == passwdStructure)
      {
         // either the requested entry was not found, or an error occurred
         const int errNumber(errno);

         if(0 == errNumber)
         {
            // requested entry was not found
            ETG_TRACE_USR1((" getUserId4UserName(): user %s not found", userName.c_str()));
            return false;
         }
         else
         {
            // error occurred
            if(EINTR == errNumber)
            {
               // try again
            }
            else
            {
               ETG_TRACE_ERR((" getUserId4UserName(): getpwnam failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
               FW_NORMAL_ASSERT_ALWAYS();
               return false;
            }
         }
      }
      else
      {
         // success
         userId = passwdStructure->pw_uid;
         ETG_TRACE_USR1((" getUserId4UserName(): user %s found", userName.c_str()));
         return true;
      }
   }

   FW_NORMAL_ASSERT_ALWAYS();
   return false;
}

} //fw
