/**
 * @file FwTestCommandServer.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Test command server.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of test command server.
 */

#include "FwTestCommandServer.h"
#include "FwITestCommandClient.h"

#include "Lock.h"
#include "TraceDefinitions.h"
#define ETRACE_S_IMPORT_INTERFACE_ETG
#include "etrace_fw.h"

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#define TRACE_S_IMPORT_INTERFACE_TYPES
#include "trace_if.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FwTestCommandServer.cpp.trc.h"
#endif
#endif

namespace fw {

// static local members to avoid includes in header file (e.g. OSAL related includes)
static LockForeverAndNonReentrant _lock;
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
static OSAL_tIODescriptor _device = OSAL_ERROR;
#endif
::std::set< ITestCommandClient* > TestCommandServer::_clients;

TestCommandServer::TestCommandServer()
{
   // _clients
   _created = false;
   _channel = 0;
}

TestCommandServer::~TestCommandServer()
{
   _clients.clear();
   destroy();
}

TestCommandServer& TestCommandServer::getInstance(void)
{
   static TestCommandServer server;
   return server;
}

void TestCommandServer::create(const unsigned int channel)
{
   _lock.lock();

   if(true == _created)
   {
      _lock.unlock();
      return;
   }

   _channel = channel;

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   if(OSAL_ERROR == _device)
   {
      _device = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);
   }

   if(OSAL_ERROR != _device)
   {
      // register with the trace driver
      OSAL_trIOCtrlLaunchChannel traceChannel;

      traceChannel.enTraceChannel = (TR_tenTraceChan)_channel;
      traceChannel.pCallback      = (OSAL_tpfCallback)traceCallback; // need to cast here because prototype does not match

      if(OSAL_ERROR != OSAL_s32IOControl(_device, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t)(&traceChannel)))
      {
         ETG_TRACE_ERR((" trace channel registered: id=%u", _channel));
         _created = true;
      }
      else
      {
         ETG_TRACE_FATAL((" trace channel registration failed (OSAL_s32IOControl): id=%u", _channel));

         if(OSAL_OK == OSAL_s32IOClose(_device))
         {
            _device = OSAL_ERROR;
         }
         else
         {
            ETG_TRACE_FATAL((" trace channel registration failed (OSAL_s32IOClose): id=%u", _channel));
         }
      }
   }
   else
   {
      ETG_TRACE_FATAL((" trace channel registration failed (OSAL_IOOpen): id=%u", _channel));
   }

#else

   _created = true;

#endif

   _lock.unlock();
}

void TestCommandServer::destroy(void)
{
   _lock.lock();

   if(false == _created)
   {
      _lock.unlock();
      return;
   }

   if(0 < _clients.size())
   {
      _lock.unlock();
      return;
   }

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   if(OSAL_ERROR != _device)
   {
      // deregister with the trace driver
      OSAL_trIOCtrlLaunchChannel traceChannel;

      traceChannel.enTraceChannel = (TR_tenTraceChan)_channel;
      traceChannel.pCallback      = (OSAL_tpfCallback)traceCallback; // need to cast here because prototype does not match

      if(OSAL_ERROR != OSAL_s32IOControl(_device, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t)(&traceChannel)))
      {
         ETG_TRACE_ERR((" trace channel deregistered: id=%u", _channel));
      }
      else
      {
         ETG_TRACE_FATAL((" trace channel deregistration failed (OSAL_s32IOControl): id=%u", _channel));
      }

      // close device anyway
      if(OSAL_OK == OSAL_s32IOClose(_device))
      {
         _device = OSAL_ERROR;
      }
      else
      {
         ETG_TRACE_FATAL((" trace channel deregistration failed (OSAL_s32IOClose): id=%u", _channel));
      }
   }

#endif

   _created = false;

   _lock.unlock();
}

void TestCommandServer::registerClient(ITestCommandClient* client) const
{
   if(0 == client)
   {
      return;
   }

   _lock.lock();

   _clients.insert(client);

   _lock.unlock();
}

void TestCommandServer::deregisterClient(ITestCommandClient* client) const
{
   if(0 == client)
   {
      return;
   }

   _lock.lock();

   ::std::set< ITestCommandClient* >::iterator it = _clients.find(client);

   if(it != _clients.end())
   {
      _clients.erase(it);
   }

   _lock.unlock();
}

void TestCommandServer::traceCallback(const unsigned char* data)
{
   if(0 == data)
   {
      return;
   }

   _lock.lock();

   for(::std::set< ITestCommandClient* >::iterator it = _clients.begin(); it != _clients.end(); ++it)
   {
      (*it)->handleTestCommand(data);
   }

   _lock.unlock();
}

} //fw
