/**
 * @file FwStringUtils.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief String utilities.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of string utilities.
 */

#include "FwStringUtils.h"
#include "FwAssert.h"
#include <cstring>

namespace fw {

static const char bin2LowerAscii[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
static const char bin2UpperAscii[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };

void convertByte2HexString(char* hexString, const unsigned char binaryValue, const bool lowerCase /*= true*/)
{
   FW_IF_NULL_PTR_RETURN(hexString);

   if(true == lowerCase)
   {
      hexString[0] = bin2LowerAscii[(binaryValue >> 4) & 0x0F];
      hexString[1] = bin2LowerAscii[binaryValue & 0x0F];
   }
   else
   {
      hexString[0] = bin2UpperAscii[(binaryValue >> 4) & 0x0F];
      hexString[1] = bin2UpperAscii[binaryValue & 0x0F];
   }

   hexString[2] = '\0';
}

unsigned char convertHexString2Byte(const char* hexString)
{
   unsigned char binaryVal = 0;

   if((hexString) && (2 == strlen(hexString)))
   {
      unsigned char lowNibble = 0;
      unsigned char highNibble = 0;

      if(('0' <= hexString[0]) && (hexString[0] <= '9'))
      {
         // numeric character
         highNibble = (unsigned char)(((unsigned char)(hexString[0] - '0')) << 4);
      }
      else if(('a' <= hexString[0]) && (hexString[0] <= 'f'))
      {
         // alphabetic character
         highNibble = (unsigned char)(((unsigned char)((hexString[0] - 'a') + 0x0A)) << 4);
      }
      else if(('A' <= hexString[0]) && (hexString[0] <= 'F'))
      {
         // alphabetic character
         highNibble = (unsigned char)(((unsigned char)((hexString[0] - 'A') + 0x0A)) << 4);
      }
      else
      {
         // invalid character
         FW_NORMAL_ASSERT_ALWAYS();
      }

      if(('0' <= hexString[1]) && (hexString[1] <= '9'))
      {
         // numeric character
         lowNibble = (unsigned char)(((unsigned char)(hexString[1] - '0')));
      }
      else if(('a' <= hexString[1]) && (hexString[1] <= 'f'))
      {
         // alphabetic character
         lowNibble = (unsigned char)(((unsigned char)((hexString[1] - 'a') + 0x0A)));
      }
      else if(('A' <= hexString[1]) && (hexString[1] <= 'F'))
      {
         // alphabetic character
         lowNibble = (unsigned char)(((unsigned char)((hexString[1] - 'A') + 0x0A)));
      }
      else
      {
         // invalid character
         FW_NORMAL_ASSERT_ALWAYS();
      }

      binaryVal = highNibble | lowNibble;
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return binaryVal;
}

void convertBinary2HexString(::std::string& stringValue, const ::std::vector< unsigned char >& binaryValue, const bool lowerCase /*= true*/)
{
   char buffer[1 + (binaryValue.size() << 1)];
   memset(buffer, 0, sizeof(buffer));
   char hexString[3];
   for(size_t i = 0; i < binaryValue.size(); i++)
   {
      convertByte2HexString(hexString, binaryValue[i], lowerCase);
      buffer[2 * i    ] = hexString[0];
      buffer[2 * i + 1] = hexString[1];
   }
   stringValue = buffer;
}

bool convertHexString2Binary(::std::vector< unsigned char >& binaryValue, const ::std::string& stringValue)
{
   if(0 == (stringValue.size() % 2))
   {
      // valid input
      size_t stringCounter = 0;
      char hexString[3];
      hexString[2] = '\0';
      binaryValue.reserve((stringValue.size() >> 1));

      while((stringCounter + 1) < stringValue.size())
      {
         hexString[0] = stringValue[stringCounter];
         stringCounter++;
         hexString[1] = stringValue[stringCounter];
         stringCounter++;
         binaryValue.push_back(convertHexString2Byte(hexString));
      }

      return true;
   }
   else
   {
      // size of string is odd
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }
}

void convertString2LowerCase(::std::string& value)
{
   for(size_t i = 0; i < value.size(); i++)
   {
      if(('A' <= value[i]) && (value[i] <= 'Z'))
      {
         value[i] = (char)(value[i] + 32);
      }
   }
}

void convertString2UpperCase(::std::string& value)
{
   for(size_t i = 0; i < value.size(); i++)
   {
      if(('a' <= value[i]) && (value[i] <= 'z'))
      {
         value[i] = (char)(value[i] - 32);
      }
   }
}

bool makeValidUtf8String(::std::string& value, const char replacement /*= '?'*/)
{
   const size_t valueLength(value.size());

   if(0 == valueLength)
   {
      // empty string nothing to do
      //** ::std::cout << "makeValidUtf8String(): not replaced" << ::std::endl;
      return false;
   }

   // temporary string
   ::std::string tmp;
   tmp.reserve(valueLength);
   bool replaced(false);
   bool addReplaceChar(false);
   size_t currPos(0);
   size_t startPos(0); // start position of last valid string
   size_t endPos(0); // end position of last valid string
   bool validBytesAdded(false); // marker for available valid bytes
   bool startOfSequence(false); // marker for start of UTF-8 byte sequence
   unsigned char current;
   unsigned int nmbContinuationBytes(0);

   // check given string
   while(currPos < valueLength)
   {
      current = (unsigned char)value[currPos];

      // continuation byte
      if(0 < nmbContinuationBytes)
      {
         // encoding: 10xxxxxx
         if((current >> 6) == 0x02)
         {
            // continuation byte sequence OK

            // check if end of valid UTF-8 byte sequence reached
            if(1 == nmbContinuationBytes)
            {
               endPos = currPos;
               validBytesAdded = true;

               if(true == addReplaceChar)
               {
                  if('\0' != replacement)
                  {
                     tmp.push_back(replacement);
                  }

                  addReplaceChar = false;
               }

               startOfSequence = false;
            }

            ++currPos;

            // reduce number of continuation bytes
            --nmbContinuationBytes;

            // continue with next byte
            continue;
         }
         // other encoding => broken UTF-8 byte sequence
         else
         {
            //** ::std::cout << "makeValidUtf8String(): continuation byte sequence NOK: 0x" << ::std::hex << (unsigned int)current << ::std::dec << ::std::endl;

            // continuation byte sequence NOK
            replaced = true;

            // add pending replacement character
            if(true == addReplaceChar)
            {
               if('\0' != replacement)
               {
                  tmp.push_back(replacement);
               }
            }

            addReplaceChar = true;

            // copy valid bytes
            if(true == validBytesAdded)
            {
               if(endPos < startPos)
               {
                  FW_NORMAL_ASSERT_ALWAYS();
               }
               else
               {
                  tmp.append(value, startPos, 1 + (endPos - startPos));
               }
            }

            // do not increase current position, check current byte below

            // restart with current byte
            startPos = currPos;
            endPos = currPos;
            validBytesAdded = false;

            // reset number of continuation bytes
            nmbContinuationBytes = 0;
            startOfSequence = false;
         }
      }

      // single byte character
      // encoding: 0xxxxxxx
      if(current <= 0x7F)
      {
         // single byte OK
         endPos = currPos;
         validBytesAdded = true;

         if(true == addReplaceChar)
         {
            if('\0' != replacement)
            {
               tmp.push_back(replacement);
            }

            addReplaceChar = false;
         }

         ++currPos;

         // no continuation bytes
         nmbContinuationBytes = 0;
         startOfSequence = false;
      }
      // two byte character
      // encoding: 110xxxxx 10xxxxxx
      else if((current >> 5) == 0x06)
      {
         // leading byte OK
         ++currPos;

         // 1 continuation byte
         nmbContinuationBytes = 1;
         startOfSequence = true;
      }
      // three byte character
      // encoding: 1110xxxx 10xxxxxx 10xxxxxx
      else if((current >> 4) == 0x0E)
      {
         // leading byte OK
         ++currPos;

         // 2 continuation bytes
         nmbContinuationBytes = 2;
         startOfSequence = true;
      }
      // four byte character
      // encoding: 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
      else if((current >> 3) == 0x1E)
      {
         // leading byte OK
         ++currPos;

         // 3 continuation bytes
         nmbContinuationBytes = 3;
         startOfSequence = true;
      }
      // The maximum number of bytes per character is 4 according to RFC3629 which limited the character table to U+10FFFF
#if 0
      // five byte character
      // encoding: 111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
      else if((current >> 2) == 0x3E)
      {
         // leading byte OK
         ++currPos;

         // 4 continuation bytes
         nmbContinuationBytes = 4;
         startOfSequence = true;
      }
      // six byte character
      // encoding: 1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
      else if((current >> 1) == 0x7E)
      {
         // leading byte OK
         ++currPos;

         // 5 continuation bytes
         nmbContinuationBytes = 5;
         startOfSequence = true;
      }
#endif
      // invalid byte
      else
      {
         //** ::std::cout << "makeValidUtf8String(): UTF-8 byte sequence NOK: 0x" << ::std::hex << (unsigned int)current << ::std::dec << ::std::endl;

         // UTF-8 byte sequence NOK
         replaced = true;

         // add pending replacement character
         if(true == addReplaceChar)
         {
            if('\0' != replacement)
            {
               tmp.push_back(replacement);
            }
         }

         addReplaceChar = true;

         // copy valid bytes
         if(true == validBytesAdded)
         {
            if(endPos < startPos)
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
            else
            {
               tmp.append(value, startPos, 1 + (endPos - startPos));
            }
         }

         ++currPos;

         // restart with next byte
         startPos = currPos;
         endPos = currPos;
         validBytesAdded = false;

         // reset number of continuation bytes
         nmbContinuationBytes = 0;
         startOfSequence = false;
      }
   }

   //** ::std::cout << "makeValidUtf8String(): replaced=" << replaced << " addReplaceChar=" << addReplaceChar << " validBytesAdded=" << validBytesAdded << ::std::endl;

   // check for incomplete UTF-8 byte sequence
   if(true == startOfSequence)
   {
      replaced = true;

      // add pending replacement character
      if(true == addReplaceChar)
      {
         if('\0' != replacement)
         {
            tmp.push_back(replacement);
         }
      }

      addReplaceChar = true;
   }

   if(true == replaced)
   {
      // check for pending valid bytes to be added
      if(true == validBytesAdded)
      {
         if(endPos < startPos)
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }
         else
         {
            tmp.append(value, startPos, 1 + (endPos - startPos));
         }
      }

      // check for pending character replacement
      if(true == addReplaceChar)
      {
         if('\0' != replacement)
         {
            tmp.push_back(replacement);
         }
      }

      // swap
      value.swap(tmp);

      //** ::std::cout << "makeValidUtf8String(): replaced" << ::std::endl;
      return true;
   }
   else
   {
      //** ::std::cout << "makeValidUtf8String(): not replaced" << ::std::endl;
      return false;
   }
}

bool convertHexString2BcdCode(uint64_t& bcdCode, const ::std::string& stringValue, const size_t requiredLength)
{
   if(0 == requiredLength)
   {
      return false;
   }

   size_t requiredLengthInternal(requiredLength);
   const size_t nmbNibblePerBcdCode(sizeof(uint64_t) << 1);
   if(nmbNibblePerBcdCode < requiredLengthInternal)
   {
      // limit to maximum length
      requiredLengthInternal = nmbNibblePerBcdCode;
   }

   if(requiredLengthInternal > stringValue.size())
   {
      return false;
   }

   uint64_t returnCode(0);

   for(size_t i = 0; i < requiredLengthInternal; i++)
   {
      const char stringChar(stringValue[i]);
      uint64_t nibble;

      if(('0' <= stringChar) && (stringChar <= '9'))
      {
         // numeric character
         nibble = (stringChar - '0');
      }
      else
      {
         // invalid character
         return false;
      }

      returnCode <<= 4;
      returnCode |= nibble;
   }

   bcdCode = returnCode;
   return true;
}

} //fw
