/**
 * @file FwSemaphore.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Semaphore handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Semaphore handling declaration.
 */

#include "FwSemaphore.h"
#include "TraceDefinitions.h"
#include "FwAssert.h"
#include "FwTrace.h"

#include <cerrno>
#include <cstring>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/FwSemaphore.cpp.trc.h"
#endif
#endif

namespace fw {

Semaphore::Semaphore() :
_created(false)
{
}

Semaphore::~Semaphore()
{
   destroy();
}

void Semaphore::create(void)
{
   if(false == _created)
   {
      memset(&_sem, 0, sizeof(_sem));
      if(-1 == sem_init(&_sem, 0, 0))
      {
         // check result for debugging purpose

         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_init.html:
            The sem_init() function shall fail if:

            [EINVAL]
               The value argument exceeds {SEM_VALUE_MAX}.
               => value is 0 and cannot exceed SEM_VALUE_MAX
            [ENOSPC]
               A resource required to initialize the semaphore has been exhausted, or the limit on semaphores ( {SEM_NSEMS_MAX}) has been reached.
               => should never happen
            [EPERM]
               The process lacks the appropriate privileges to initialize the semaphore.
               => should never happen else there is a general problem
            [ENOSYS]
               pshared is nonzero, but the system does not support process-shared semaphores
               => pshared is 0
          *
          */

         int errNumber = errno;
         ETG_TRACE_ERR((" create(): sem_init failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      _created = true;
   }
}

void Semaphore::destroy(void)
{
   if(true == _created)
   {
      (void)sem_destroy(&_sem); // ignore return value

      _created = false;
   }
}

void Semaphore::post(void)
{
   if(true == _created)
   {
      if(-1 == sem_post(&_sem))
      {
         // check result for debugging purpose

         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_post.html:
            The sem_post() function may fail if:

            [EINVAL]
               The sem argument does not refer to a valid semaphore.
               => a valid semaphore is provided
            [EOVERFLOW]
               The maximum allowable value for a semaphore would be exceeded.
               => should never happen
          *
          */

         int errNumber = errno;
         ETG_TRACE_ERR((" post(): sem_post failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
}

void Semaphore::wait(void)
{
   if(true == _created)
   {
      for(;;)
      {
         if(-1 == sem_wait(&_sem))
         {
            /*
             * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_wait.html:
               The sem_trywait() and sem_wait() functions shall fail if:

               [EAGAIN]
                  The semaphore was already locked, so it cannot be immediately locked by the sem_trywait() operation ( sem_trywait() only).
                  => valid for sem_trywait() only, no need to handle

               The sem_trywait() and sem_wait() functions may fail if:

               [EDEADLK]
                  A deadlock condition was detected.
                  => handle as error
               [EINTR]
                  A signal interrupted this function.
                  => handle as normal behavior, do a while loop
               [EINVAL]
                  The sem argument does not refer to a valid semaphore.
                  => a valid semaphore is provided
               [ETIMEDOUT]
                  The call timed out before the semaphore could be locked.
                  => valid for sem_timedwait() only, no need to handle
             *
             */

            int errNumber = errno;

            if(EINTR != errNumber)
            {
               // error
               ETG_TRACE_ERR((" wait(): sem_wait failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            }
         }
         else
         {
            // success
            break;
         }
      }
   }
}

} //fw
