/**
 * @file FwBluetoothStringUtils.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief String utilities.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of Bluetooth specific string utilities.
 */

#include "FwBluetoothStringUtils.h"

namespace fw {

static const ::std::string bdAddressExampleWithoutColon = "aabbccddeeff";
static const ::std::string bdAddressExampleWithColon = "aa:bb:cc:dd:ee:ff";
static const unsigned short int defaultDeviceIdSpecificationId = 0x0103;
static const unsigned short int vendorIdSourceBluetoothSIG = 0x0001; // Bluetooth SIG assigned Device ID Vendor ID value from the Assigned Numbers document
static const unsigned short int vendorIdSourceUSBImplementersForum = 0x0002; // USB Implementer’s Forum assigned Vendor ID value
static const unsigned short int vendorIdSourceReserved = 0xFFFF; // Reserved e.g. used by initial configuration of RPI (ensure support for this)
static const unsigned short int appleVendorIdBluetoothSIG = 0x004C; // Apple, Inc.
static const unsigned short int appleVendorIdUSBImplementersForum = 0x05AC; // Apple, Inc.
static const ::std::string outgoingIAP2Uuid = "00000000decafadedecadeafdecacafe"; // iAP2 over Bluetooth UUID (outgoing): 0xfecacadeafdecadedefacade00000000
static const ::std::string incomingIAP2Uuid = "00000000decafadedecadeafdecacaff"; // iAP2 over Bluetooth UUID (incoming): 0xffcacadeafdecadedefacade00000000
static const ::std::string outgoingCPWUuid("2d8d2466e14d451c88bc7301abea291a"); // CarPlay wireless UUID (outgoing): 0x1a29eaab0173bc881c454de166248d2d
static const ::std::string incomingCPWUuid("d31fbf505d572797a24041cd484388ec"); // CarPlay wireless UUID (incoming): 0xec884348cd4140a29727575d50bf1fd3
static const ::std::string outgoingSPPUuid("0000110100001000800000805f9b34fb"); // general SPP UUID (outgoing)
static const ::std::string incomingSPPUuid("0000110100001000800000805f9b34fb"); // general SPP UUID (incoming, same as outgoing)
static const ::std::string incomingAAWUuid("4de17a0052cb11e6bdf40800200c9a66"); // AndroidAuto wireless UUID (incoming): 4de17a00-52cb-11e6-bdf4-0800200c9a66
static const ::std::string uuid128woHyphen = "0000000000001000800000805f9b34fb"; // UUID 128 bit, without hyphen
static const ::std::string uuid128wiHyphen = "00000000-0000-1000-8000-00805f9b34fb"; // UUID 128 bit, with hyphen

bool isValidBdAddress(const ::std::string& address, const bool withColon /*= false*/, const bool allowEmpty /*= false*/)
{
   // from interface point of view an empty BT address is also valid in some cases
   if((true == allowEmpty) && (0 == address.size()))
   {
      return true;
   }

   if(true == withColon)
   {
      if(bdAddressExampleWithColon.size() != address.size())
      {
         return false;
      }
   }
   else
   {
      if(bdAddressExampleWithoutColon.size() != address.size())
      {
         return false;
      }
   }

   for(size_t i = 0; i < address.size(); i++)
   {
      if((true == withColon) && (':' == bdAddressExampleWithColon[i]))
      {
         // colon expected
         if(':' == address[i])
         {
            // given address has colon
         }
         else
         {
            // given address has different character
            return false;
         }
      }
      else if(('0' <= address[i]) && (address[i] <= '9'))
      {
         // numeric character
      }
      else if(('a' <= address[i]) && (address[i] <= 'f'))
      {
         // alphabetic character
      }
      else if(('A' <= address[i]) && (address[i] <= 'F'))
      {
         // alphabetic character
      }
      else
      {
         // invalid character
         return false;
      }
   }

   return true;
}

bool isValid128BitUuid(const ::std::string& uuid, const bool withHyphen /*= false*/, const bool allowEmpty /*= false*/)
{
   // from interface point of view an empty UUID is also valid in some cases
   if((true == allowEmpty) && (0 == uuid.size()))
   {
      return true;
   }

   if(true == withHyphen)
   {
      if(uuid128wiHyphen.size() != uuid.size())
      {
         return false;
      }
   }
   else
   {
      if(uuid128woHyphen.size() != uuid.size())
      {
         return false;
      }
   }

   for(size_t i = 0; i < uuid.size(); i++)
   {
      if((true == withHyphen) && ('-' == uuid128wiHyphen[i]))
      {
         // hyphen expected
         if('-' == uuid[i])
         {
            // given UUID has hyphen
         }
         else
         {
            // given UUID has different character
            return false;
         }
      }
      else if(('0' <= uuid[i]) && (uuid[i] <= '9'))
      {
         // numeric character
      }
      else if(('a' <= uuid[i]) && (uuid[i] <= 'f'))
      {
         // alphabetic character
      }
      else if(('A' <= uuid[i]) && (uuid[i] <= 'F'))
      {
         // alphabetic character
      }
      else
      {
         // invalid character
         return false;
      }
   }

   return true;
}

bool isAppleDevice(const unsigned short int vendorId, const unsigned short int vendorIdSource)
{
   if(((vendorIdSource == vendorIdSourceBluetoothSIG) && (vendorId == appleVendorIdBluetoothSIG)) ||
      ((vendorIdSource == vendorIdSourceUSBImplementersForum) && (vendorId == appleVendorIdUSBImplementersForum)))
   {
      return true;
   }

   return false;
}

unsigned short int getDeviceIdSpecificationId(void)
{
   return defaultDeviceIdSpecificationId;
}

unsigned short int getVendorIdSourceBluetoothSIG(void)
{
   return vendorIdSourceBluetoothSIG;
}

unsigned short int getVendorIdSourceUSBImplementersForum(void)
{
   return vendorIdSourceUSBImplementersForum;
}

unsigned short int getVendorIdSourceReserved(void)
{
   return vendorIdSourceReserved;
}

unsigned short int getAppleVendorIdBluetoothSIG(void)
{
   return appleVendorIdBluetoothSIG;
}

unsigned short int getAppleVendorIdUSBImplementersForum(void)
{
   return appleVendorIdUSBImplementersForum;
}

const ::std::string& getOutgoingIAP2Uuid(void)
{
   return outgoingIAP2Uuid;
}

const ::std::string& getIncomingIAP2Uuid(void)
{
   return incomingIAP2Uuid;
}

const ::std::string& getOutgoingCPWUuid(void)
{
   return outgoingCPWUuid;
}

const ::std::string& getIncomingCPWUuid(void)
{
   return incomingCPWUuid;
}

const ::std::string& getOutgoingSPPUuid(void)
{
   return outgoingSPPUuid;
}

const ::std::string& getIncomingSPPUuid(void)
{
   return incomingSPPUuid;
}

const ::std::string& getIncomingAAWUuid(void)
{
   return incomingAAWUuid;
}

const ::std::string& getBaseUuidWithoutHyphen(void)
{
   return uuid128woHyphen;
}

const ::std::string& getBaseUuidWithHyphen(void)
{
   return uuid128wiHyphen;
}

} //fw
