/**
 * @file BoschRegistry.cpp
 *
 * @par SW-Component
 * Framework
 *
 * @brief Bosch registry handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Bosch registry handling.
 */

#include "BoschRegistry.h"
#include "TraceDefinitions.h"
#include "Utils.h"
#define ETRACE_S_IMPORT_INTERFACE_ETG
#include "etrace_fw.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONN_FRAMEWORK_GENERAL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BoschRegistry.cpp.trc.h"
#endif
#endif

namespace fw {

BoschRegistry::BoschRegistry()
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   _fd = OSAL_ERROR;

#else

   counter = 0;

#endif
}

BoschRegistry::~BoschRegistry()
{
}

void BoschRegistry::open(const ::std::string& registryPath)
{
   if(registryPath.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   if(OSAL_ERROR == _fd)
   {
      _fd = OSAL_IOOpen(registryPath.c_str(), OSAL_EN_READWRITE);
   }

   if(OSAL_ERROR == _fd)
   {
      ETG_TRACE_FATAL((" open registry failed (OSAL_IOOpen)"));
   }

#else

   counter++;

#endif
}

void BoschRegistry::close(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   if(OSAL_ERROR != _fd)
   {
      if(OSAL_OK == OSAL_s32IOClose(_fd))
      {
         _fd = OSAL_ERROR;
      }
      else
      {
         ETG_TRACE_FATAL((" close registry failed (OSAL_s32IOClose)"));
      }
   }

#else

   counter++;

#endif
}

void BoschRegistry::write(const ::std::string& key, const ::std::string& value) const
{
   if(key.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(value.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   if(OSAL_ERROR != _fd)
   {
      OSAL_trIOCtrlRegistry regItem;

      regItem.pcos8Name = (tPCS8)key.c_str();
      regItem.s32Type = OSAL_C_S32_VALUE_STRING;
      regItem.u32Size = (tU32)(value.length() + 1); // including NULL termination
      regItem.ps8Value = (tPU8)(const_cast<char*>(value.c_str()));

      if(OSAL_ERROR == OSAL_s32IOControl(_fd, OSAL_C_S32_IOCTRL_REGSETVALUE, (intptr_t)(&regItem)))
      {
         ETG_TRACE_FATAL((" write registry failed (OSAL_s32IOControl)"));
      }
   }

#endif
}

bool BoschRegistry::read(::std::string& value, const ::std::string& key) const
{
   if(key.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }

   bool result = false;

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   if(OSAL_ERROR != _fd)
   {
      OSAL_trIOCtrlRegistry regItem;
      char buffer[OSAL_C_U32_MAX_VALUENAME] = { 0 };

      regItem.pcos8Name = (tPCS8)key.c_str();
      regItem.s32Type = OSAL_C_S32_VALUE_STRING;
      regItem.u32Size = sizeof(buffer);
      regItem.ps8Value = (tPU8)(buffer);

      if(OSAL_ERROR == OSAL_s32IOControl(_fd, OSAL_C_S32_IOCTRL_REGGETVALUE, (intptr_t)(&regItem)))
      {
         ETG_TRACE_FATAL((" read registry failed (OSAL_s32IOControl)"));
      }
      else
      {
         value = buffer;
         result = true;
      }
   }

#else

   value = "dummy";
   result = true;

#endif

   return result;
}

} //fw
