/**
 * @defgroup ThreadFactory ThreadFactory
 * @author Matthias Thoemel
 *
 * Public interface for the Thread Factory
 * @{
 */
 
#ifndef _THREADFACTORY_H_
#define _THREADFACTORY_H_

#include <pthread.h>
#include <semaphore.h>

class TFThread 
{
public:
	virtual void Do(int param, void *userPtr) = 0;
};

class ThreadFactory
{
    
public:

    /**
     * Constructor
     * 
     */
    ThreadFactory()  ;

    /**
     * Constructor
     *
     * @param[in] number of threads needed during runtime
     * @param[in] prefix of thread names
     *
     */
    ThreadFactory(const int numberOfThreads, const char* threadPrefix);

    /**
     * Starts a user function in a thread from the thread factory's thread pool
     *
     * @param[in] thread pointer to the user class with a Do method in it
     * @param[in] param additional parameter to select between functions that should be run
     *                  as threads in the users class
     * @param[in] userContext additional pointer to user supplied data which will be routed to
     *                        the user function as first parameter
     * @return index of triggered thread in thread context
     */
    int Do(TFThread *thread, int param , void *userContext);

    /**
     * This function is used by LocalSPM to create the ThreadFactory.
     * Inform LocalSPM that Create is ready -> CreateDone(0)
     *
     * @return void
     */
    void Create(void);

    /**
     * Interface to create a thread.
     *
     * @param[in] threadIndex index in thread context used to cancel related thread
     *
     * @return < 0: error, = 0: OK
     */
    int CreateThread(const int threadIndex);

    /**
     * Interface to cancel a thread created by the threadfactory.
     *
     * @param[in] threadIndex index in thread context used to cancel related thread
     *
     * @return < 0: error, = 0: OK
     */
    int CancelThread(const int threadIndex);

    /**
     * Interface to kill a thread created by the threadfactory.
     *
     * @param[in] threadIndex index in thread context used to kill related thread
     *
     * @return < 0: error, = 0: OK
     */
    int KillThread(const int threadIndex);

    /**
     * Interface to kill all the threads created by the threadfactory.
     * Should be last point while shutting down the module and should be called from the main thread.
     *
     * @return < 0: error, = 0: OK
     */
    int KillAllThreads(void);

    /**
     * Interface to set the name of thie current thread. This method hould be called in the threads entry method
     *
     * @param[in] name thread name to set
     *
     * @return none
     */
    static void SetName(const char *name);

    int SendSignal(const int threadIndex, const int sig);
    typedef pthread_t tThreadID;
    static tThreadID GetThreadId();
    static int SetPriority(int prio);
    int GetThreadsRunning();
    int GetMaxThreads() {return mNoOfThreads;};

private:

	typedef struct {
		void *userContext;
		TFThread *threadFunction;
		int functionID;
		int count;
		int returnValue;
		pthread_mutex_t mtx;
		int stopThread;
		int busy;
		sem_t *freeWorkerThreads;
		pthread_t mThreadHandle;
		pthread_attr_t mThreadAttr;
		char *prefix;
	} tThreadContext;

	static void *startFunction(void *context);
    static void ResetName(tThreadContext *ctx);
    static bool JoinTimeoutCallBack(timer_t timerID , void* instance ,const void *userData);

	enum {mMaxNoOfThreads = 60, mDefaultNoOfThreads = 30};
	tThreadContext mThreadContext[mMaxNoOfThreads];
	pthread_mutex_t mLock;
	sem_t mFreeWorkerThreads;
	// weg pthread_mutex_t mOneWorkerFree;
	bool			mInitialized;
	int mNoOfThreads;
	enum {mSizeOfThreadPrefix = 5}; // without NULL termination
	char mThreadPrefix[mSizeOfThreadPrefix + 1];


};

#endif //_THREADFACTORY_H_
/** @} */
