/**
 * @file FwUtils.h
 *
 * @par SW-Component
 * Framework
 *
 * @brief Utilities.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Utilities.
 */

#ifndef _FW_UTILS_H_
#define _FW_UTILS_H_

#include <string>
#include <vector>
#include <unistd.h>

namespace fw {

/**
 * Check if the given process is running.
 *
 * @param[in] processName: name of the process
 *
 * @return = true: process is running,
 * @return = false: process is not running
 */
bool isProcessRunning(const ::std::string& processName);

/**
 * Get MAC address for given network interface.
 *
 * @param[out] macAddress: MAC address of the network interface
 * @param[out] macAddressString: MAC address of the network interface as string
 * @param[in] networkInterface: name of the network interface e.g. "wlan0"
 *
 * @return = true: network interface found,
 * @return = false: network interface not found
 */
bool getNetworkMacAddress(::std::vector<unsigned char>& macAddress, ::std::string& macAddressString, const ::std::string& networkInterface);

/**
 * Calculate absolute difference.
 *
 * @param[in] value1: value 1
 * @param[in] value2: value 2
 *
 * @return = absolute difference
 */
long int calcAbsoluteDifference(const long int value1, const long int value2);

/**
 * Get process id.
 *
 * @return = process id
 */
pid_t getProcessId(void);

/**
 * Get thread id.
 *
 * @return = thread id
 */
pid_t getThreadId(void);

/**
 * Get thread name.
 *
 * @param[in] buffer: buffer used to write thread name
 * @param[in] bufferSize: size of buffer
 *
 * @return = thread name
 */
const char* getThreadName(char* buffer, const size_t bufferSize);

/**
 * Get current time.
 *
 * @param[in] buffer: buffer used to write current time
 * @param[in] bufferSize: size of buffer
 *
 * @return = current time
 */
const char* getTime(char* buffer, const size_t bufferSize);

/**
 * Checks if a file exists.
 *
 * @param[in] fileName: path + name of a file
 *
 * @return = true: file exists,
 * @return = false: file does not exist or error occurred
 */
bool doesFileExist(const ::std::string& fileName);

/**
 * Get group id for given group name.
 *
 * @param[out] groupId: group id
 * @param[in] groupName: group name
 *
 * @return = true: group name found,
 * @return = false: group name not found
 */
bool getGroupId4GroupName(__gid_t& groupId, const ::std::string& groupName);

/**
 * Get user id for given user name.
 *
 * @param[out] userId: user id
 * @param[in] userName: user name
 *
 * @return = true: user name found,
 * @return = false: user name not found
 */
bool getUserId4UserName(__uid_t& userId, const ::std::string& userName);

} //fw

#endif //_FW_UTILS_H_
