/**
 * @file FwMemoryChecker.h
 *
 * @par SW-Component
 * Framework
 *
 * @brief Memory checker.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Memory checker.
 */

#ifndef _FW_MEMORY_CHECKER_H_
#define _FW_MEMORY_CHECKER_H_

#include "FwLock.h"
#include <map>
#include <stdint.h> // using <cstdint> compiler complains that std=c++0x or -std=gnu++0x compiler option must be enabled

// NOTE: use feature switch VARIANT_S_FTR_ENABLE_FW_MEMORY_CHECKER

#define FW_MEMORY_CHECK_NEW_OBJ(a) ::fw::MemoryChecker::getInstance().memoryCheckNewObj(a, __FILE__, __LINE__)
#define FW_MEMORY_CHECK_NEW_OBJ_SIZE(a, b) ::fw::MemoryChecker::getInstance().memoryCheckNewObj(a, b, __FILE__, __LINE__)
#define FW_MEMORY_CHECK_DEL_OBJ(a) ::fw::MemoryChecker::getInstance().memoryCheckDelObj(a, __FILE__, __LINE__)

namespace fw {

/**
 *
 */
class MemoryChecker
{
public:
   /**
    * Get instance.
    *
    * @return = instance
    */
   static MemoryChecker& getInstance(void);

   /**
    * Memory check for object created with new.
    *
    * @param[in] pointer: object
    * @param[in] fileName: file name
    * @param[in] lineNumber: line number
    */
   void memoryCheckNewObj(void* pointer, const char* fileName, const unsigned int lineNumber);

   /**
    * Memory check for object created with new.
    *
    * @param[in] pointer: object
    * @param[in] nmbBytes: allocated memory size
    * @param[in] fileName: file name
    * @param[in] lineNumber: line number
    */
   void memoryCheckNewObj(void* pointer, const size_t nmbBytes, const char* fileName, const unsigned int lineNumber);

   /**
    * Memory check for object to be deleted with delete.
    *
    * @param[in] pointer: object
    * @param[in] fileName: file name
    * @param[in] lineNumber: line number
    */
   void memoryCheckDelObj(void* pointer, const char* fileName, const unsigned int lineNumber);

private:
   /**
    * Default constructor.
    */
   MemoryChecker();

   /**
    * Destructor.
    */
   virtual ~MemoryChecker();

   const unsigned int _checkValue64; /**< check value for 64 */
   const unsigned int _checkValue128; /**< check value for 128 */
   const unsigned int _checkValue256; /**< check value for 256 */
   const unsigned int _checkValue512; /**< check value for 512 */
   unsigned int _counter; /**< counter */
   unsigned int _newCounter; /**< counter */
   unsigned int _delCounter; /**< counter */
   size_t _allocatedMemory; /**< allocated memory in bytes */
   LockForeverAndNonReentrant _lock; /**< lock */
   ::std::map< uintptr_t, unsigned int > _ptrList; /**< pointer list */
   ::std::map< uintptr_t, size_t > _memList; /**< memory list list */

   /**
    * Check pointer list.
    */
   void memoryCheckList(void);
};

} //fw

#endif //_FW_MEMORY_CHECKER_H_
