/**
 * @file FwFormattedDataPrint.h
 *
 * @par SW-Component
 * Framework
 *
 * @brief Formatted data print.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Formatted data print.
 */

#ifndef _FW_FORMATTED_DATA_PRINT_H_
#define _FW_FORMATTED_DATA_PRINT_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <stdint.h> // using <cstdint> compiler complains that std=c++0x or -std=gnu++0x compiler option must be enabled
#include <unistd.h>

namespace fw {

/**
 * Buffer size definition for output strings (including NULL termination).
 */
enum
{
   BUFFER_SIZE_32 = 32,
   BUFFER_SIZE_64 = 64,
   BUFFER_SIZE_128 =128
};

/**
 *
 */
class FormattedOutputU8
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    * @param[in] printAsHex: flag for printing in decimal or hex mode
    */
   FormattedOutputU8(const uint8_t data, const bool printAsHex = false);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputU8();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputU8();

private:
   // buffer size: decimal: 255 => 4; hex: 0xFF => 5
   char _buffer[BUFFER_SIZE_32]; /**< buffer with formatted output */
};

inline FormattedOutputU8::FormattedOutputU8() { _buffer[0] = '\0'; }
inline FormattedOutputU8::~FormattedOutputU8() { }

/**
 *
 */
class FormattedOutputU32
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    * @param[in] printAsHex: flag for printing in decimal or hex mode
    */
   FormattedOutputU32(const uint32_t data, const bool printAsHex = false);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputU32();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputU32();

private:
   // buffer size: decimal: 4294967296 => 11; hex: 0xFFFFFFFF => 11
   char _buffer[BUFFER_SIZE_32]; /**< buffer with formatted output */
};

inline FormattedOutputU32::FormattedOutputU32() { _buffer[0] = '\0'; }
inline FormattedOutputU32::~FormattedOutputU32() { }

/**
 *
 */
class FormattedOutputS64
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    * @param[in] printAsHex: flag for printing in decimal or hex mode
    */
   FormattedOutputS64(const int64_t data, const bool printAsHex = false);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputS64();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputS64();

private:
   // buffer size: decimal: -9223372036854775807 .. 9223372036854775807 => 21; hex (converted to unsigned): 0xFFFFFFFFFFFFFFFF => 19
   char _buffer[BUFFER_SIZE_32]; /**< buffer with formatted output */
};

inline FormattedOutputS64::FormattedOutputS64() { _buffer[0] = '\0'; }
inline FormattedOutputS64::~FormattedOutputS64() { }

/**
 *
 */
class FormattedOutputU64
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    * @param[in] printAsHex: flag for printing in decimal or hex mode
    */
   FormattedOutputU64(const uint64_t data, const bool printAsHex = false);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputU64();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputU64();

private:
   // buffer size: decimal: 18446744073709551615 => 21; hex: 0xFFFFFFFFFFFFFFFF => 19
   char _buffer[BUFFER_SIZE_32]; /**< buffer with formatted output */
};

inline FormattedOutputU64::FormattedOutputU64() { _buffer[0] = '\0'; }
inline FormattedOutputU64::~FormattedOutputU64() { }

/**
 *
 */
class FormattedOutputPtr
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    * @param[in] printAsHex: flag for printing in decimal or hex mode
    */
   FormattedOutputPtr(const uintptr_t data, const bool printAsHex = true);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputPtr();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputPtr();

private:
   // buffer size: decimal: 18446744073709551615 => 21; hex: 0xFFFFFFFFFFFFFFFF => 19 */
   char _buffer[BUFFER_SIZE_32]; /**< buffer with formatted output */
};

inline FormattedOutputPtr::FormattedOutputPtr() { _buffer[0] = '\0'; }
inline FormattedOutputPtr::~FormattedOutputPtr() { }

/**
 *
 */
class FormattedOutputPid
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    */
   FormattedOutputPid(const pid_t data);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputPid();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputPid();

private:
   // buffer size: decimal (example): 18446744073709551615 => 21; hex: 0xFFFFFFFFFFFFFFFF => 19 */
   char _buffer[BUFFER_SIZE_32]; /**< buffer with formatted output */
};

inline FormattedOutputPid::FormattedOutputPid() { _buffer[0] = '\0'; }
inline FormattedOutputPid::~FormattedOutputPid() { }

/**
 *
 */
class FormattedOutputFloat
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    * @param[in] printAsScientific: flag for printing in decimal floating point or scientific notation (mantissa/exponent) mode
    */
   FormattedOutputFloat(const float data, const bool printAsScientific = false);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputFloat();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputFloat();

private:
   // buffer size: 50 characters should be enough */
   char _buffer[BUFFER_SIZE_64]; /**< buffer with formatted output */
};

inline FormattedOutputFloat::FormattedOutputFloat() { _buffer[0] = '\0'; }
inline FormattedOutputFloat::~FormattedOutputFloat() { }

/**
 *
 */
class FormattedOutputDouble
{
public:
   /**
    * Constructor.
    *
    * @param[in] data: data
    * @param[in] printAsScientific: flag for printing in decimal floating point or scientific notation (mantissa/exponent) mode
    */
   FormattedOutputDouble(const double data, const bool printAsScientific = false);

   /**
    * Destructor.
    */
   inline virtual ~FormattedOutputDouble();

   /**
    * Get buffer with formatted output. Returned pointer is never NULL.
    *
    * @return = buffer with formatted output
    */
   inline const char* c_str(void) const { return _buffer; }

protected:
   /**
    * Default constructor. Do not use!!!
    */
   inline FormattedOutputDouble();

private:
   // buffer size: 60 characters should be enough */
   char _buffer[BUFFER_SIZE_64]; /**< buffer with formatted output */
};

inline FormattedOutputDouble::FormattedOutputDouble() { _buffer[0] = '\0'; }
inline FormattedOutputDouble::~FormattedOutputDouble() { }

} //fw

#endif //_FORMATTED_DATA_PRINT_H_
