/**
 * @file WblDbusWifiSetupProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Wbl DBUS proxy for WifiSetup.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Wbl DBUS proxy for WifiSetup.
 */

#include "WblDbusWifiSetupProxyIf.h"
#include "WblDbusParser.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

using namespace ::asf::core;
using namespace ::org::bosch::wbl::wifisetup;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblDbusWifiSetupProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace wbl {

WblDbusWifiSetupProxyIf::WblDbusWifiSetupProxyIf(ICcDbusIfControllerClient* client) :
DbusBaseProxyIf< IWblDbusWifiSetupCallbackIf, WifisetupProxy, IWblDbusWifiSetupTestProxyIf >(client)
{
}

WblDbusWifiSetupProxyIf::WblDbusWifiSetupProxyIf(ICcDbusIfControllerClient* client, IWblDbusWifiSetupTestProxyIf* testProxyIf) :
DbusBaseProxyIf< IWblDbusWifiSetupCallbackIf, WifisetupProxy, IWblDbusWifiSetupTestProxyIf >(client, testProxyIf)
{
}

WblDbusWifiSetupProxyIf::~WblDbusWifiSetupProxyIf()
{
}

void WblDbusWifiSetupProxyIf::onAvailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< WifisetupProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< WifisetupProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            (void)myProxy->sendModeRegister(*this);
            // not needed --- (void)myProxy->sendInterfaceRegister(*this);
            // not needed --- (void)myProxy->sendMacAddressRegister(*this);
            (void)myProxy->sendPowerStateRegister(*this);
            // not needed --- (void)myProxy->sendPowerFailureReasonRegister(*this);
            (void)myProxy->sendFrequencyRegister(*this);
            // not needed --- (void)myProxy->sendCurrentOperatingChannelRegister(*this);
            (void)myProxy->sendAPConfigRegister(*this);
            // not needed --- (void)myProxy->sendRestrictionsRegister(*this);
            (void)myProxy->sendSupportedFrequenciesRegister(*this);
#ifdef VARIANT_S_FTR_ENABLE_CONNECTION_STATUS
            (void)myProxy->sendConnectionStatusRegister(*this);
#endif

            internalOnAvailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onUnavailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< WifisetupProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< WifisetupProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            myProxy->sendDeregisterAll();

            internalOnUnavailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::handleSetCallback(IWblDbusWifiSetupCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   storeCallback(callbackIf, enableProxy, callbackId); // independent of path, name, type => update status to all
}

void WblDbusWifiSetupProxyIf::handleCreateProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalCreateProxy(callbackId, objPath, busName, busType);
}

void WblDbusWifiSetupProxyIf::handleDestroyProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalDestroyProxy(callbackId, objPath, busName, busType);
}

void WblDbusWifiSetupProxyIf::handleDestroyAllProxies(void)
{
   internalDestroyAllProxies();
}

void WblDbusWifiSetupProxyIf::handleDestroyAllRuntimeProxies(void)
{
   internalDestroyAllRuntimeProxies();
}

IDestroyAllProxies* WblDbusWifiSetupProxyIf::getDestroyer(void)
{
   return this;
}

void WblDbusWifiSetupProxyIf::setCallbackIf(IWblDbusWifiSetupCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processSetCallback(this, callbackIf, enableProxy, callbackId, objPath, busName, busType);
}

void WblDbusWifiSetupProxyIf::createProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processCreateProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

void WblDbusWifiSetupProxyIf::destroyProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processDestroyProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

act_t WblDbusWifiSetupProxyIf::sendDeAuthenticateStationRequest(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& macAddress, const ::std::string& deAuthenticationMode)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendDeAuthenticateStationRequest(*this, objPath, macAddress, deAuthenticationMode);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

         // NOTE: NOT IMPLEMENTED
         // NEEDED: add macAddress and deAuthenticationMode to args
         FW_NORMAL_ASSERT_ALWAYS();

         retAct = proxy->sendDeAuthenticateStationRequest(*this, args);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendManageBandwidthRequest(const unsigned int callbackId, const ::std::string& objPath, bool allocate, const ::std::string& address, const unsigned int bandwidth)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendManageBandwidthRequest(*this, objPath, allocate, address, bandwidth);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

         // NOTE: NOT IMPLEMENTED
         // NEEDED: add address and bandwidth to args
         FW_NORMAL_ASSERT_ALWAYS();

         retAct = proxy->sendManageBandwidthRequest(*this, allocate, args);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendSetRestrictionsRequest(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& technology, const ::std::string& restriction)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendSetRestrictionsRequest(*this, objPath, technology, restriction);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

         // NOTE: NOT IMPLEMENTED
         // NEEDED: add technology to args
         FW_NORMAL_ASSERT_ALWAYS();

         retAct = proxy->sendSetRestrictionsRequest(*this, args, restriction);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendModeGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendModeGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendModeGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendInterfaceGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendInterfaceGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendInterfaceGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendMacAddressGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendMacAddressGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendMacAddressGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendPowerStateGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendPowerStateGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendPowerStateGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendPowerFailureReasonGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendPowerFailureReasonGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendPowerFailureReasonGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendFrequencyGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendFrequencyGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendFrequencyGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendCurrentOperatingChannelGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendCurrentOperatingChannelGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendCurrentOperatingChannelGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendAPConfigGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendAPConfigGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendAPConfigGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendRestrictionsGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendRestrictionsGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendRestrictionsGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendSupportedFrequenciesGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendSupportedFrequenciesGet(*this, objPath);
      }
   }
   else
   {
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendSupportedFrequenciesGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusWifiSetupProxyIf::sendConnectionStatusGet(const unsigned int callbackId, const ::std::string& objPath)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath)) // independent of name, type
      {
         retAct = _testProxyIf->sendConnectionStatusGet(*this, objPath);
      }
   }
   else
   {
#ifdef VARIANT_S_FTR_ENABLE_CONNECTION_STATUS
      ::boost::shared_ptr< WifisetupProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath)) // independent of name, type
      {
         retAct = proxy->sendConnectionStatusGet(*this);
      }
#endif
   }

   addAct(callbackId, retAct);

   return retAct;
}

void WblDbusWifiSetupProxyIf::onDeAuthenticateStationError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< DeAuthenticateStationError >& error)
{
   onDeAuthenticateStationErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onDeAuthenticateStationResponse(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< DeAuthenticateStationResponse >& response)
{
   onDeAuthenticateStationResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onManageBandwidthError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< ManageBandwidthError >& error)
{
   onManageBandwidthErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onManageBandwidthResponse(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< ManageBandwidthResponse >& response)
{
   onManageBandwidthResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onSetRestrictionsError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< SetRestrictionsError >& error)
{
   onSetRestrictionsErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onSetRestrictionsResponse(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< SetRestrictionsResponse >& response)
{
   onSetRestrictionsResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onModeError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< ModeError >& error)
{
   onModeErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onModeUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< ModeUpdate >& update)
{
   onModeUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onInterfaceError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< InterfaceError >& error)
{
   onInterfaceErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onInterfaceUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< InterfaceUpdate >& update)
{
   onInterfaceUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onMacAddressError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< MacAddressError >& error)
{
   onMacAddressErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onMacAddressUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< MacAddressUpdate >& update)
{
   onMacAddressUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onPowerStateError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< PowerStateError >& error)
{
   onPowerStateErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onPowerStateUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< PowerStateUpdate >& update)
{
   onPowerStateUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onPowerFailureReasonError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< PowerFailureReasonError >& error)
{
   onPowerFailureReasonErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onPowerFailureReasonUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< PowerFailureReasonUpdate >& update)
{
   onPowerFailureReasonUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onFrequencyError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< FrequencyError >& error)
{
   onFrequencyErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onFrequencyUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< FrequencyUpdate >& update)
{
   onFrequencyUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onCurrentOperatingChannelError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< CurrentOperatingChannelError >& error)
{
   onCurrentOperatingChannelErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onCurrentOperatingChannelUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< CurrentOperatingChannelUpdate >& update)
{
   onCurrentOperatingChannelUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onAPConfigError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< APConfigError >& error)
{
   onAPConfigErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onAPConfigUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< APConfigUpdate >& update)
{
   onAPConfigUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onRestrictionsError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< RestrictionsError >& error)
{
   onRestrictionsErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onRestrictionsUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< RestrictionsUpdate >& update)
{
   onRestrictionsUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onSupportedFrequenciesError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< SupportedFrequenciesError >& error)
{
   onSupportedFrequenciesErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onSupportedFrequenciesUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< SupportedFrequenciesUpdate >& update)
{
   onSupportedFrequenciesUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

#ifdef VARIANT_S_FTR_ENABLE_CONNECTION_STATUS
void WblDbusWifiSetupProxyIf::onConnectionStatusError(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< ConnectionStatusError >& error)
{
   onConnectionStatusErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusWifiSetupProxyIf::onConnectionStatusUpdate(const ::boost::shared_ptr< WifisetupProxy >& proxy, const ::boost::shared_ptr< ConnectionStatusUpdate >& update)
{
   onConnectionStatusUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}
#endif

void WblDbusWifiSetupProxyIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnAvailable(available, objPath, busName, busType);
}

void WblDbusWifiSetupProxyIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnUnavailable(available, objPath, busName, busType);
}

void WblDbusWifiSetupProxyIf::onDeAuthenticateStationErrorCb(const ::boost::shared_ptr< DeAuthenticateStationError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onDeAuthenticateStationErrorCb(error, objPath, busName, busType, interfaceName);
}

void WblDbusWifiSetupProxyIf::onDeAuthenticateStationResponseCb(const ::boost::shared_ptr< DeAuthenticateStationResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onDeAuthenticateStationResponseCb(response, objPath, busName, busType, interfaceName);
}

void WblDbusWifiSetupProxyIf::onManageBandwidthErrorCb(const ::boost::shared_ptr< ManageBandwidthError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onManageBandwidthErrorCb(error, objPath, busName, busType, interfaceName);
}

void WblDbusWifiSetupProxyIf::onManageBandwidthResponseCb(const ::boost::shared_ptr< ManageBandwidthResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onManageBandwidthResponseCb(response, objPath, busName, busType, interfaceName);
}

void WblDbusWifiSetupProxyIf::onSetRestrictionsErrorCb(const ::boost::shared_ptr< SetRestrictionsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onSetRestrictionsErrorCb(error, objPath, busName, busType, interfaceName);
}

void WblDbusWifiSetupProxyIf::onSetRestrictionsResponseCb(const ::boost::shared_ptr< SetRestrictionsResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onSetRestrictionsResponseCb(response, objPath, busName, busType, interfaceName);
}

void WblDbusWifiSetupProxyIf::onModeErrorCb(const ::boost::shared_ptr< ModeError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onModeErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onModeErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onModeUpdateCb(const ::boost::shared_ptr< ModeUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onModeUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onModeUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onInterfaceErrorCb(const ::boost::shared_ptr< InterfaceError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onInterfaceErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onInterfaceErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onInterfaceUpdateCb(const ::boost::shared_ptr< InterfaceUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onInterfaceUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onInterfaceUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onMacAddressErrorCb(const ::boost::shared_ptr< MacAddressError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onMacAddressErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onMacAddressErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onMacAddressUpdateCb(const ::boost::shared_ptr< MacAddressUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onMacAddressUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onMacAddressUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onPowerStateErrorCb(const ::boost::shared_ptr< PowerStateError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onPowerStateErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPowerStateErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onPowerStateUpdateCb(const ::boost::shared_ptr< PowerStateUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onPowerStateUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPowerStateUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onPowerFailureReasonErrorCb(const ::boost::shared_ptr< PowerFailureReasonError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onPowerFailureReasonErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPowerFailureReasonErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onPowerFailureReasonUpdateCb(const ::boost::shared_ptr< PowerFailureReasonUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onPowerFailureReasonUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPowerFailureReasonUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onFrequencyErrorCb(const ::boost::shared_ptr< FrequencyError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onFrequencyErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onFrequencyErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onFrequencyUpdateCb(const ::boost::shared_ptr< FrequencyUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onFrequencyUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onFrequencyUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onCurrentOperatingChannelErrorCb(const ::boost::shared_ptr< CurrentOperatingChannelError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onCurrentOperatingChannelErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onCurrentOperatingChannelErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onCurrentOperatingChannelUpdateCb(const ::boost::shared_ptr< CurrentOperatingChannelUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onCurrentOperatingChannelUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onCurrentOperatingChannelUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onAPConfigErrorCb(const ::boost::shared_ptr< APConfigError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onAPConfigErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAPConfigErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onAPConfigUpdateCb(const ::boost::shared_ptr< APConfigUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onAPConfigUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAPConfigUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onRestrictionsErrorCb(const ::boost::shared_ptr< RestrictionsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onRestrictionsErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onRestrictionsErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onRestrictionsUpdateCb(const ::boost::shared_ptr< RestrictionsUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onRestrictionsUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onRestrictionsUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onSupportedFrequenciesErrorCb(const ::boost::shared_ptr< SupportedFrequenciesError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onSupportedFrequenciesErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onSupportedFrequenciesErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onSupportedFrequenciesUpdateCb(const ::boost::shared_ptr< SupportedFrequenciesUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onSupportedFrequenciesUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onSupportedFrequenciesUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

#ifdef VARIANT_S_FTR_ENABLE_CONNECTION_STATUS
void WblDbusWifiSetupProxyIf::onConnectionStatusErrorCb(const ::boost::shared_ptr< ConnectionStatusError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onConnectionStatusErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConnectionStatusErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusWifiSetupProxyIf::onConnectionStatusUpdateCb(const ::boost::shared_ptr< ConnectionStatusUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusWifiSetupCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onConnectionStatusUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConnectionStatusUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}
#endif

void WblDbusWifiSetupProxyIf::internalCreateProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   wbl::WblDbusParser parser;
   const ::std::string& port(parser.getPort(wbl::IF_WIFI_SETUP));
   const ::std::string& path(objPath);
   const ::std::string& name(parser.getBusName(wbl::IF_WIFI_SETUP));
   const DbusBusType type(parser.getBusType(wbl::IF_WIFI_SETUP));

   // check if proxy is available
   if(false == isProxyAvailable(objPath)) // independent of name, type
   {
      ETG_TRACE_USR3((" internalCreateProxy(): WifiSetup: callbackId=%u busType=%d busName=%20s objPath=%s (create)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // create proxy
      if(0 != _testProxyIf)
      {
         createTestProxy(port, path, name, type, *this);
         addTestProxy(objPath); // independent of name, type
      }
      else
      {
         ::boost::shared_ptr< WifisetupProxy > proxy = createProxy(port, path, name, type, *this);
         addProxy(proxy, objPath); // independent of name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalCreateProxy(): WifiSetup: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // proxy creation was already started; either service is available or creation is ongoing
      updateCurrentAvailableStatus(callbackId, getCreationState(objPath) /* independent of name, type */, path, name, type);
   }
}

void WblDbusWifiSetupProxyIf::internalDestroyProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy
   // HINT: sendDeregisterAll() is not provided by all proxies

   // check if proxy is available
   if(true == isProxyAvailable(objPath)) // independent of name, type
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): WifiSetup: callbackId=%u busType=%d busName=%20s objPath=%s (destroy)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // destroy proxy
      if(0 != _testProxyIf)
      {
         wbl::WblDbusParser parser;
         const ::std::string& path(objPath);
         const ::std::string& name(parser.getBusName(wbl::IF_WIFI_SETUP));
         const DbusBusType type(parser.getBusType(wbl::IF_WIFI_SETUP));

         destroyTestProxy(path, name, type);
         removeTestProxy(objPath); // independent of name, type
      }
      else
      {
         ::boost::shared_ptr< WifisetupProxy >* proxyPtr(getProxy(objPath)); // independent of name, type
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< WifisetupProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
         removeProxy(objPath); // independent of name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): WifiSetup: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));
   }

   // NOTE: check following: second callback is registered, anything to do?
}

void WblDbusWifiSetupProxyIf::internalDestroyAllProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllProxies(): WifiSetup"));

   // destroy all proxies
   if(0 != _testProxyIf)
   {
      destroyAllTestProxies();
      removeAllTestProxies();
   }
   else
   {
      for(size_t i = 0; i < getProxyListSize(); i++)
      {
         ::boost::shared_ptr< WifisetupProxy >* proxyPtr(getProxy(i));
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< WifisetupProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
      }
      removeAllProxies();
   }

   // remove callback information
   removeCallbacks();
}

void WblDbusWifiSetupProxyIf::internalDestroyAllRuntimeProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllRuntimeProxies(): WifiSetup"));

   // destroy all runtime proxies
   if(0 != _testProxyIf)
   {
      destroyAllTestProxies();
      removeAllTestProxies();
   }
   else
   {
      for(size_t i = 0; i < getProxyListSize(); i++)
      {
         ::boost::shared_ptr< WifisetupProxy >* proxyPtr(getProxy(i));
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< WifisetupProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
      }
      removeAllProxies();
   }

   // do not remove callback information
}

void WblDbusWifiSetupProxyIf::internalOnAvailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnAvailable(): WifiSetup"));

   // inform all matching callback handler
   setCreationState(available, objPath); // independent of name, type
   updateAvailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

void WblDbusWifiSetupProxyIf::internalOnUnavailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnUnavailable(): WifiSetup"));

   // inform all matching callback handler
   setCreationState(available, objPath); // independent of name, type
   updateUnavailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

} //wbl
} //ccdbusif
