/**
 * @file WblDbusConflictManagementProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Wbl DBUS proxy for ConflictManagement.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Wbl DBUS proxy for ConflictManagement.
 */

#include "WblDbusConflictManagementProxyIf.h"
#include "WblDbusParser.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

using namespace ::asf::core;
using namespace ::org::bosch::wbl::conflictmanagement;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblDbusConflictManagementProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace wbl {

WblDbusConflictManagementProxyIf::WblDbusConflictManagementProxyIf(ICcDbusIfControllerClient* client) :
DbusBaseProxyIf< IWblDbusConflictManagementCallbackIf, ConflictmanagementProxy, IWblDbusConflictManagementTestProxyIf >(client)
{
}

WblDbusConflictManagementProxyIf::WblDbusConflictManagementProxyIf(ICcDbusIfControllerClient* client, IWblDbusConflictManagementTestProxyIf* testProxyIf) :
DbusBaseProxyIf< IWblDbusConflictManagementCallbackIf, ConflictmanagementProxy, IWblDbusConflictManagementTestProxyIf >(client, testProxyIf)
{
}

WblDbusConflictManagementProxyIf::~WblDbusConflictManagementProxyIf()
{
}

void WblDbusConflictManagementProxyIf::onAvailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< ConflictmanagementProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< ConflictmanagementProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            (void)myProxy->sendConflictDetectedRegister(*this);
            (void)myProxy->sendConflictResolvedRegister(*this);
            (void)myProxy->sendActiveWiFiSetupsRegister(*this);
            (void)myProxy->sendWiFiPriorityListRegister(*this);
            (void)myProxy->sendConfigurationTimerRegister(*this);

            internalOnAvailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onUnavailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< ConflictmanagementProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< ConflictmanagementProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            myProxy->sendDeregisterAll();

            internalOnUnavailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::handleSetCallback(IWblDbusConflictManagementCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   storeCallback(callbackIf, enableProxy, callbackId); // independent of path, name, type => update status to all
}

void WblDbusConflictManagementProxyIf::handleCreateProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalCreateProxy(callbackId, objPath, busName, busType);
}

void WblDbusConflictManagementProxyIf::handleDestroyProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalDestroyProxy(callbackId, objPath, busName, busType);
}

void WblDbusConflictManagementProxyIf::handleDestroyAllProxies(void)
{
   internalDestroyAllProxies();
}

void WblDbusConflictManagementProxyIf::handleDestroyAllRuntimeProxies(void)
{
   internalDestroyAllRuntimeProxies();
}

IDestroyAllProxies* WblDbusConflictManagementProxyIf::getDestroyer(void)
{
   return this;
}

void WblDbusConflictManagementProxyIf::setCallbackIf(IWblDbusConflictManagementCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processSetCallback(this, callbackIf, enableProxy, callbackId, objPath, busName, busType);
}

void WblDbusConflictManagementProxyIf::createProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processCreateProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

void WblDbusConflictManagementProxyIf::destroyProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processDestroyProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

act_t WblDbusConflictManagementProxyIf::sendPrepareSetupRequest(const unsigned int callbackId, const ::std::string& wiFiMode)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendPrepareSetupRequest(*this, wiFiMode);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

         // NOTE: NOT IMPLEMENTED
         // NEEDED: add given parameter to args
         FW_NORMAL_ASSERT_ALWAYS();

         retAct = proxy->sendPrepareSetupRequest(*this, args);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusConflictManagementProxyIf::sendDeActivateSetupRequest(const unsigned int callbackId, const ::std::string& wiFiMode)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendDeActivateSetupRequest(*this, wiFiMode);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendDeActivateSetupRequest(*this, wiFiMode);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusConflictManagementProxyIf::sendWiFiStationModeUsageRequest(const unsigned int callbackId, const ::std::string& application, const ::std::string& usage, bool holding)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendWiFiStationModeUsageRequest(*this, application, usage, holding);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendWiFiStationModeUsageRequest(*this, application, usage, holding);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusConflictManagementProxyIf::sendResolveConflictRequest(const unsigned int callbackId, int32 conflictid, int32 action)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendResolveConflictRequest(*this, conflictid, action);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendResolveConflictRequest(*this, conflictid, action);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusConflictManagementProxyIf::sendActiveWiFiSetupsGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendActiveWiFiSetupsGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendActiveWiFiSetupsGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t WblDbusConflictManagementProxyIf::sendWiFiPriorityListGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendWiFiPriorityListGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendWiFiPriorityListGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void WblDbusConflictManagementProxyIf::sendWiFiPriorityListSet(const unsigned int callbackId, const ::std::vector< ::std::string >& wiFiPriorityList)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendWiFiPriorityListSet(*this, wiFiPriorityList);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendWiFiPriorityListSet(wiFiPriorityList);
      }
   }
}

act_t WblDbusConflictManagementProxyIf::sendConfigurationTimerGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendConfigurationTimerGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendConfigurationTimerGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void WblDbusConflictManagementProxyIf::sendConfigurationTimerSet(const unsigned int callbackId, int32 configurationTimer)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendConfigurationTimerSet(*this, configurationTimer);
      }
   }
   else
   {
      ::boost::shared_ptr< ConflictmanagementProxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendConfigurationTimerSet(configurationTimer);
      }
   }
}

void WblDbusConflictManagementProxyIf::onPrepareSetupError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< PrepareSetupError >& error)
{
   onPrepareSetupErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onPrepareSetupResponse(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< PrepareSetupResponse >& response)
{
   onPrepareSetupResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onDeActivateSetupError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< DeActivateSetupError >& error)
{
   onDeActivateSetupErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onDeActivateSetupResponse(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< DeActivateSetupResponse >& response)
{
   onDeActivateSetupResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onWiFiStationModeUsageError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< WiFiStationModeUsageError >& error)
{
   onWiFiStationModeUsageErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onWiFiStationModeUsageResponse(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< WiFiStationModeUsageResponse >& response)
{
   onWiFiStationModeUsageResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onResolveConflictError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ResolveConflictError >& error)
{
   onResolveConflictErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onResolveConflictResponse(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ResolveConflictResponse >& response)
{
   onResolveConflictResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onConflictDetectedError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ConflictDetectedError >& error)
{
   onConflictDetectedErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onConflictDetectedSignal(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ConflictDetectedSignal >& signal)
{
   onConflictDetectedSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onConflictResolvedError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ConflictResolvedError >& error)
{
   onConflictResolvedErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onConflictResolvedSignal(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ConflictResolvedSignal >& signal)
{
   onConflictResolvedSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onActiveWiFiSetupsError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ActiveWiFiSetupsError >& error)
{
   onActiveWiFiSetupsErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onActiveWiFiSetupsUpdate(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ActiveWiFiSetupsUpdate >& update)
{
   onActiveWiFiSetupsUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onWiFiPriorityListError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< WiFiPriorityListError >& error)
{
   onWiFiPriorityListErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onWiFiPriorityListUpdate(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< WiFiPriorityListUpdate >& update)
{
   onWiFiPriorityListUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onConfigurationTimerError(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ConfigurationTimerError >& error)
{
   onConfigurationTimerErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onConfigurationTimerUpdate(const ::boost::shared_ptr< ConflictmanagementProxy >& proxy, const ::boost::shared_ptr< ConfigurationTimerUpdate >& update)
{
   onConfigurationTimerUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void WblDbusConflictManagementProxyIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnAvailable(available, objPath, busName, busType);
}

void WblDbusConflictManagementProxyIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnUnavailable(available, objPath, busName, busType);
}

void WblDbusConflictManagementProxyIf::onPrepareSetupErrorCb(const ::boost::shared_ptr< PrepareSetupError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onPrepareSetupErrorCb(error, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onPrepareSetupResponseCb(const ::boost::shared_ptr< PrepareSetupResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onPrepareSetupResponseCb(response, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onDeActivateSetupErrorCb(const ::boost::shared_ptr< DeActivateSetupError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onDeActivateSetupErrorCb(error, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onDeActivateSetupResponseCb(const ::boost::shared_ptr< DeActivateSetupResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onDeActivateSetupResponseCb(response, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onWiFiStationModeUsageErrorCb(const ::boost::shared_ptr< WiFiStationModeUsageError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onWiFiStationModeUsageErrorCb(error, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onWiFiStationModeUsageResponseCb(const ::boost::shared_ptr< WiFiStationModeUsageResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onWiFiStationModeUsageResponseCb(response, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onResolveConflictErrorCb(const ::boost::shared_ptr< ResolveConflictError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onResolveConflictErrorCb(error, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onResolveConflictResponseCb(const ::boost::shared_ptr< ResolveConflictResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onResolveConflictResponseCb(response, objPath, busName, busType, interfaceName);
}

void WblDbusConflictManagementProxyIf::onConflictDetectedErrorCb(const ::boost::shared_ptr< ConflictDetectedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onConflictDetectedErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConflictDetectedErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onConflictDetectedSignalCb(const ::boost::shared_ptr< ConflictDetectedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onConflictDetectedSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConflictDetectedSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onConflictResolvedErrorCb(const ::boost::shared_ptr< ConflictResolvedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onConflictResolvedErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConflictResolvedErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onConflictResolvedSignalCb(const ::boost::shared_ptr< ConflictResolvedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onConflictResolvedSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConflictResolvedSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onActiveWiFiSetupsErrorCb(const ::boost::shared_ptr< ActiveWiFiSetupsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onActiveWiFiSetupsErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onActiveWiFiSetupsErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onActiveWiFiSetupsUpdateCb(const ::boost::shared_ptr< ActiveWiFiSetupsUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onActiveWiFiSetupsUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onActiveWiFiSetupsUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onWiFiPriorityListErrorCb(const ::boost::shared_ptr< WiFiPriorityListError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onWiFiPriorityListErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onWiFiPriorityListErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onWiFiPriorityListUpdateCb(const ::boost::shared_ptr< WiFiPriorityListUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onWiFiPriorityListUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onWiFiPriorityListUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onConfigurationTimerErrorCb(const ::boost::shared_ptr< ConfigurationTimerError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onConfigurationTimerErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConfigurationTimerErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::onConfigurationTimerUpdateCb(const ::boost::shared_ptr< ConfigurationTimerUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IWblDbusConflictManagementCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onConfigurationTimerUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConfigurationTimerUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void WblDbusConflictManagementProxyIf::internalCreateProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   wbl::WblDbusParser parser;
   const ::std::string& port(parser.getPort(wbl::IF_CONFLICT_MANAGEMENT));
   const ::std::string& path(parser.getObjectPath(wbl::IF_CONFLICT_MANAGEMENT));
   const ::std::string& name(parser.getBusName(wbl::IF_CONFLICT_MANAGEMENT));
   const DbusBusType type(parser.getBusType(wbl::IF_CONFLICT_MANAGEMENT));

   // check if proxy is available
   if(false == isProxyAvailable()) // independent of path, name, type
   {
      ETG_TRACE_USR3((" internalCreateProxy(): ConflictManagement: callbackId=%u busType=%d busName=%20s objPath=%s (create)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // create proxy
      if(0 != _testProxyIf)
      {
         createTestProxy(port, path, name, type, *this);
         addTestProxy(); // independent of path, name, type
      }
      else
      {
         ::boost::shared_ptr< ConflictmanagementProxy > proxy = createProxy(port, path, name, type, *this);
         addProxy(proxy); // independent of path, name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalCreateProxy(): ConflictManagement: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // proxy creation was already started; either service is available or creation is ongoing
      updateCurrentAvailableStatus(callbackId, getCreationState() /* independent of path, name, type */, path, name, type);
   }
}

void WblDbusConflictManagementProxyIf::internalDestroyProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy
   // HINT: sendDeregisterAll() is not provided by all proxies

   // check if proxy is available
   if(true == isProxyAvailable()) // independent of path, name, type
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): ConflictManagement: callbackId=%u busType=%d busName=%20s objPath=%s (destroy)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // destroy proxy
      if(0 != _testProxyIf)
      {
         wbl::WblDbusParser parser;
         const ::std::string& path(parser.getObjectPath(wbl::IF_CONFLICT_MANAGEMENT));
         const ::std::string& name(parser.getBusName(wbl::IF_CONFLICT_MANAGEMENT));
         const DbusBusType type(parser.getBusType(wbl::IF_CONFLICT_MANAGEMENT));

         destroyTestProxy(path, name, type);
         removeTestProxy(); // independent of path, name, type
      }
      else
      {
         ::boost::shared_ptr< ConflictmanagementProxy >* proxyPtr(getProxy()); // independent of path, name, type
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< ConflictmanagementProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
         removeProxy(); // independent of path, name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): ConflictManagement: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));
   }

   // NOTE: check following: second callback is registered, anything to do?
}

void WblDbusConflictManagementProxyIf::internalDestroyAllProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllProxies(): ConflictManagement"));

   // destroy all proxies
   if(0 != _testProxyIf)
   {
      destroyAllTestProxies();
      removeAllTestProxies();
   }
   else
   {
      for(size_t i = 0; i < getProxyListSize(); i++)
      {
         ::boost::shared_ptr< ConflictmanagementProxy >* proxyPtr(getProxy(i));
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< ConflictmanagementProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
      }
      removeAllProxies();
   }

   // remove callback information
   removeCallbacks();
}

void WblDbusConflictManagementProxyIf::internalDestroyAllRuntimeProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllRuntimeProxies(): ConflictManagement"));

   // destroy all runtime proxies
   // nothing to do because this is a static proxy

   // do not remove callback information
}

void WblDbusConflictManagementProxyIf::internalOnAvailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnAvailable(): ConflictManagement"));

   // inform all matching callback handler
   setCreationState(available); // independent of path, name, type
   updateAvailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

void WblDbusConflictManagementProxyIf::internalOnUnavailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnUnavailable(): ConflictManagement"));

   // inform all matching callback handler
   setCreationState(available); // independent of path, name, type
   updateUnavailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

} //wbl
} //ccdbusif
