/**
 * @file WblDbusParser.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief WBL DBUS Property Parser.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details WBL DBUS Property Parser.
 */

#ifndef _WBL_DBUS_PARSER_H_
#define _WBL_DBUS_PARSER_H_

#include "CcDbusIfTypes.h"
#include "WblDbusTypes.h"
#include "asf/dbus/DBusVariant.h"

namespace ccdbusif {
namespace wbl {

/**
 * DBUS property parser class for WBL.
 */
class WblDbusParser
{
public:
   WblDbusParser();
   virtual ~WblDbusParser();

   /**
    * Parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] matchingInterfaces: interfaces matching the given properties
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseProperties(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] data: introspection data
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseIntrospection(const ::std::string& data,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] addedObjectPath: added object path
    * @param[in] inInterfaces: map with interfaces (mutable)
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseInterfacesAdded(const ::std::string& addedObjectPath, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] removedObjectPath: removed object path
    * @param[in] inInterfaces: vector with interfaces
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseInterfacesRemoved(const ::std::string& removedObjectPath, const ::std::vector< ::std::string >& inInterfaces,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] propertyName: property to be found/parsed
    * @param[in] interface: related interface
    *
    * @return = true: property found,
    * @return = false: property not found
    */
   bool findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const ::std::string& propertyName, const int interface) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] propertyValue: received property value (mutable)
    * @param[in] propertyName: received property name
    * @param[in] interface: related interface
    */
   void parseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName, const int interface) const;

   /**
    * Convert function.
    *
    * @return = root object path
    */
   const ::std::string& getRootObjectPath(void) const;

   /**
    * Convert function.
    *
    * @param[in] interface: interface
    *
    * @return = interface name
    */
   const ::std::string& getInterfaceName(const Interface interface) const;

   /**
    * Convert function.
    *
    * @param[in] interface: interface
    *
    * @return = object path
    */
   const ::std::string& getObjectPath(const Interface interface) const;

   /**
    * Convert function.
    *
    * @param[in] interface: interface
    *
    * @return = bus name
    */
   const ::std::string& getBusName(const Interface interface) const;

   /**
    * Convert function.
    *
    * @param[in] interface: interface
    *
    * @return = bus type
    */
   DbusBusType getBusType(const Interface interface) const;

   /**
    * Convert function.
    *
    * @param[in] interface: interface
    *
    * @return = port
    */
   const ::std::string& getPort(const Interface interface) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   LastIntendedModeProperty getLastIntendedModeProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getLastIntendedModeProperty2String(const LastIntendedModeProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = enum value
    */
   PowerState getPowerState2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: enum value
    *
    * @return = equivalent string name
    */
   const ::std::string& getPowerState2String(const PowerState value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = enum value
    */
   WiFiMode getWiFiMode2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: enum value
    *
    * @return = equivalent string name
    */
   const ::std::string& getWiFiMode2String(const WiFiMode value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = enum value
    */
   WiFiFrequency getWiFiFrequency2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: enum value
    *
    * @return = equivalent string name
    */
   const ::std::string& getWiFiFrequency2String(const WiFiFrequency value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = enum value
    */
   WiFiType getWiFiType2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: enum value
    *
    * @return = equivalent string name
    */
   const ::std::string& getWiFiType2String(const WiFiType value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = enum value
    */
   SupportedSetting getSupportedSetting2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: enum value
    *
    * @return = equivalent string name
    */
   const ::std::string& getSupportedSetting2String(const SupportedSetting value) const;

   /**
    * Set traces enabled flag.
    *
    * @param[in] enabled: traces enabled flag
    */
   void setTraces(const bool enabled);

   /**
    * Get traces enabled flag.
    *
    * @return = traces enabled flag
    */
   inline bool getTraces(void) const { return _tracesEnabled; }

private:
   bool _tracesEnabled; /**< traces enabled flag */
};

} //wbl
} //ccdbusif

#endif //_WBL_DBUS_PARSER_H_
